from collections import Counter
import threading

from mcpp.parse import Sitter, get_identifiers
from mcpp.queries import Q_ARGLIST, Q_IDENTIFIER, Q_FUNCTION_PARAMETER, \
    Q_POINTER_EXPR, Q_ASSIGNMENT_EXPR, Q_BINARY_EXPR, Q_CALL_NAME, \
    Q_IF_STMT, Q_SWITCH_STMT, Q_DO_STMT, Q_WHILE_STMT, Q_FOR_STMT, Q_CONDITION, \
    Q_IF_WITHOUT_ELSE


def v1(root, sitter, lang, calls=None):
    """
    V1: number of variables as parameters for callee functions
    """
    sitter.add_queries({
        "Q_ARGLIST": Q_ARGLIST
    })

    vars_in_calls = []
    for arg_list in sitter.captures("Q_ARGLIST", root, lang).get("args", []):
        variables = get_identifiers(sitter, arg_list, lang, filter=calls)
        vars_in_calls.extend(variables)

    return {
        "V1": len(vars_in_calls)
    }


def v2(root, sitter, lang, calls=None):
    """
    V2: number of variables as parameters for callee functions
    """
    sitter.add_queries({
        "Q_FUNCTION_PARAMETER": Q_FUNCTION_PARAMETER
    })
    params = sitter.captures("Q_FUNCTION_PARAMETER", root, lang).get("param", [])
    return {
        "V2": len(params)
    }


def v3_v4(root, sitter, lang, calls=None):
    """
    V3: number of pointer arithmetic operations
    V4: number of variables involved in pointer arithmetics
    """
    sitter.add_queries({
        "Q_POINTER_EXPR": Q_POINTER_EXPR
    })
    arith_ops = [
        "+", "++", "+=",
        "-", "--", "-=",
        "*=",  # * excluded (same as pointer reference)
        "/", "/=",
        "^", "^=",
        "&=",  # & excluded (same as pointer dereference)
        "|", "|="
    ]

    pointer_arith = []
    pointer_arith_vars = []
    for pointer in sitter.captures("Q_POINTER_EXPR", root, lang).get("pointer", []):
        if any(arith in pointer.parent.text.decode() for arith in arith_ops):
            pointer_arith.append(pointer)
            variables = get_identifiers(sitter, pointer.parent, lang, filter=calls)
            pointer_arith_vars.extend(variables)

    return {
        "V3": len(pointer_arith),
        "V4": len(pointer_arith_vars)
    }


def v5(root, sitter, lang, calls=None):
    """
    V5: maximum number of pointer arithmetic operations a variable is involved in
    """
    sitter.add_queries({
        "Q_BINARY_EXPR": Q_BINARY_EXPR,
        "Q_ASSIGNMENT_EXPR": Q_ASSIGNMENT_EXPR,
        "Q_CALL_NAME": Q_CALL_NAME
    })
    arith_ops = [
        "+", "++", "+=",
        "-", "--", "-=",
        "*", "*=",
        "/", "/="
    ]

    var_count = Counter()
    candidates = sitter.captures("Q_BINARY_EXPR", root, lang).get("expr", []) + sitter.captures("Q_ASSIGNMENT_EXPR", root, lang).get("expr", [])
    for node in candidates:
        if len(node.children) != 3:
            continue
        op_text = node.children[1].text.decode()
        if any(arith in op_text for arith in arith_ops):
            variables = get_identifiers(sitter, node, lang, filter=calls)
            var_count.update(variables)
    if len(var_count) > 0:
        max_count = var_count.most_common(1)[0][1]
    else:
        max_count = 0
    return {
        "V5": max_count
    }


def v6_v7(root, sitter, lang, calls=None):
    """
    V6: number of nested control structures
    V7: maximum level of control nesting
    """
    queries = {
        "Q_IF_STMT": Q_IF_STMT,
        "Q_SWITCH_STMT": Q_SWITCH_STMT,
        "Q_DO_STMT": Q_DO_STMT,
        "Q_WHILE_STMT": Q_WHILE_STMT,
        "Q_FOR_STMT": Q_FOR_STMT
    }
    sitter.add_queries(queries)

    nested_controls = []
    max_nesting_level = 0
    for q in queries.keys():
        for node in sitter.captures(q, root, lang).get("stmt", []):
            nesting_level = _control_nesting_level(node)
            if nesting_level > 0:
                nested_controls.append(node)
            max_nesting_level = max(max_nesting_level, nesting_level)

    return {
        "V6": len(nested_controls),
        "V7": max_nesting_level
    }


def _control_nesting_level(node):
    control_types = [
        "if_statement",
        "switch_statement",
        "do_statement",
        "while_statement",
        "for_statement"
    ]
    parent = node.parent
    num_control_ancestors = 0
    while parent is not None:
        if parent.type in control_types:
            num_control_ancestors += 1
        parent = parent.parent
    return num_control_ancestors


def v8(root, sitter, lang, calls=None):
    """
    V8: maximum number of control-dependent control structures
    """
    queries = {
        "Q_IF_STMT": Q_IF_STMT,
        "Q_SWITCH_STMT": Q_SWITCH_STMT,
        "Q_DO_STMT": Q_DO_STMT,
        "Q_WHILE_STMT": Q_WHILE_STMT,
        "Q_FOR_STMT": Q_FOR_STMT,
        "Q_CONDITION": Q_CONDITION
    }
    sitter.add_queries(queries)

    # count dependent controls under another control: key = start_byte of parent in function
    control_dependent_controls = Counter()
    threads = []
    thread_lock = threading.Lock()
    for q in queries.keys():
        t = threading.Thread(target=_v8_single_query,
                             args=(root, sitter, lang, calls, q,
                                   control_dependent_controls, thread_lock))
        t.start()
        threads.append(t)
    for t in threads:
        t.join()

    return {
        "V8": max([0] + list(control_dependent_controls.values()))
    }


def _v8_single_query(root, sitter, lang, calls, query, control_dependent_controls, thread_lock):
    tag = "condition" if "Q_CONDITION" in query else "stmt"
    for node in sitter.captures(query, root, lang).get(tag, []):
        parents = _traverse_parent_controls(node)
        if len(parents) > 0:
            with thread_lock:
                control_dependent_controls[parents[-1].start_byte] += 1


def _traverse_parent_controls(node):
    """ Climb up the AST and emit all control nodes. """
    control_types = [
        "if_statement",
        "switch_statement",
        "do_statement",
        "while_statement",
        "for_statement"
    ]
    parent_controls = []
    parent = node.parent
    while parent is not None:
        if parent.type in control_types:
            parent_controls.append(parent)
        parent = parent.parent
    return parent_controls


def v9(root, sitter, lang, calls=None):
    """
    V9: maximum number of data-dependent control structures
    """
    sitter.add_queries({
        "Q_CONDITION": Q_CONDITION,
        "Q_BINARY_EXPR": Q_BINARY_EXPR
    })
    logical_ops = [
        "&", "&&",
        "|", "||"
    ]

    conditions = sitter.captures("Q_CONDITION", root, lang).get("condition", [])
    var_count = Counter()
    for condition in conditions:
        bin_expr = sitter.captures("Q_BINARY_EXPR", condition, lang).get("expr", [])
        for expr in bin_expr:
            if len(expr.children) != 3:
                continue
            left, op, right = expr.children
            if op.text.decode() in logical_ops:
                var_count.update(get_identifiers(sitter, expr, lang, filter=calls))

    return {
        "V9": max([0] + list(var_count.values()))
    }


def v10(root, sitter, lang, calls=None):
    """
    V10: number of if statements without else
    """
    sitter.add_queries({
        "Q_IF_WITHOUT_ELSE": Q_IF_WITHOUT_ELSE
    })

    if_without_else = sitter.captures("Q_IF_WITHOUT_ELSE", root, lang).get("stmt", [])
    return {
        "V10": len(if_without_else)
    }


def v11(root, sitter, lang, calls=None):
    """
    V11: number of variables in control structures (in each predicate)
    """
    sitter.add_queries({
        "Q_CONDITION": Q_CONDITION
    })

    num_controlled_vars = 0
    conditions = sitter.captures("Q_CONDITION", root, lang).get("condition", [])
    for condition in conditions:
        num_controlled_vars += len(get_identifiers(sitter, condition, lang, filter=calls))
    return {
        "V11": num_controlled_vars
    }
