#ifndef _REP_PROCESS_H_
#define _REP_PROCESS_H_

#include "InfraMed/InfraMed/InfraMed.h"
#include "InfraMed/InfraMed/MedPidRepository.h"
#include "MedProcessTools/MedProcessTools/MedSamples.h"
#include "MedProcessTools/MedProcessTools/MedProcessUtils.h"
#include <SerializableObject/SerializableObject/SerializableObject.h>
#include "MedProcessTools/MedProcessTools/MedValueCleaner.h"
#include <MedMat/MedMat/MedMat.h>
#include <omp.h>
#include <cmath>

#define DEFAULT_REP_CLNR_NTHREADS 8

//.......................................................................................
/** Define types of repository processors
*/
//.......................................................................................
/// Define types of repository processors
typedef enum {
	REP_PROCESS_MULTI, ///<"multi_processor" or "multi" to activate RepMultiProcessor
	REP_PROCESS_BASIC_OUTLIER_CLEANER,///<"basic_outlier_cleaner" or "basic_cln" to activate RepBasicOutlierCleaner
	REP_PROCESS_NBRS_OUTLIER_CLEANER,///<"nbrs_outlier_cleaner" or "nbrs_cln" to activate RepNbrsOutlierCleaner
	REP_PROCESS_CONFIGURED_OUTLIER_CLEANER,///<"configured_outlier_cleaner" or "conf_cln" to activate RepConfiguredOutlierCleaner
	REP_PROCESS_RULEBASED_OUTLIER_CLEANER,///<"rulebased_outlier_cleaner" or "rule_cln" to activate RepRuleBasedOutlierCleaner
	REP_PROCESS_CALC_SIGNALS,///<"calc_signals" or "calculator" to activate RepCalcSimpleSignals
	REP_PROCESS_COMPLETE, ///<"complete" to activate RepPanelCompleter
	REP_PROCESS_CHECK_REQ, ///<"req" or "requirements" check compliance with minimal requirement to activate RepCheckReq
	REP_PROCESS_SIM_VAL, ///<"sim_val" or "sim_val_handler" handle multiple simultanous values to activate RepSimValHandler
	REP_PROCESS_SIGNAL_RATE, ///<"signal_rate" combine complition for Drug rate based on Drug amount to activate RepSignalRate
	REP_PROCESS_COMBINE, ///<"combine" flatten signals to 1 signal by dates. if conflict chooses based on order given. to activate RepCombineSignals
	REP_PROCESS_SPLIT, ///<"split" split signal to two signals based on set of values - usefull for example to give diffrent rule\factor to diffrent drug units.  to actiate RepSplitSignal
	REP_PROCESS_AGGREGATION_PERIOD, ///<"aggregation_period" - creates RepAggregationPeriod
	REP_PROCESS_BASIC_RANGE_CLEANER,///<"basic_range_cleaner" or "range_cln" to activate RepBasicRangeCleaner
	REP_PROCESS_AGGREGATE, ///<"aggregate" - aggregate signal in sliding time window to calc some aggregation function. to activate RepAggregateSignal
	REP_PROCESS_HISTORY_LIMIT, ///<"history_limit" chomps the history for a signal to be at a certain given time window relative to the prediction point. creates RepHistoryLimit
	REP_PROCESS_CREATE_REGISTRY, ///<"create_registry" creates a registry signal (TimeRange to values). creates RepCreateRegistry
	REP_PROCESS_CREATE_BIT_SIGNAL, ///<"bit_signal" creates a state of categories (typically drugs) encoded in bits. creates RepCreateBitSignal
	REP_PROCESS_CATEGORY_DESCENDERS, ///< "category_descenders" creates all descenders values for each category value. Creates RepCategoryDescenders
	REP_PROCESS_REODER_CHANNELS, ///< "reorder_channels" reorder signal channels. Creates RepReoderChannels
	REP_PROCESS_FILTER_BY_CHANNELS, ///< "filter_channels" reorder signal channels. Creates RepFilterByChannel
	REP_PROCESS_NUMERIC_NOISER, ///<"numeric_noiser" adds gaussian noise to value and uniform noise to time for numeric signal. Creates RepNumericNoiser
	REP_PROCESS_FILTER_BY_DIAG, ///< "filter_by_diag". Creates RepClearSignalByDiag
	REP_PROCESS_LAST
} RepProcessorTypes;

/** @file
* RepProcessor is the parent class for processing a MedRepository or PidDynamicRec\n
* Basic functionalities:\n
*		learn : learn the processoring parameters from a given list of ids and a rpository \n
*		apply : process a dynamic PidDynamicRec\n
*/
class RepProcessor : public SerializableObject {
public:

	RepProcessorTypes processor_type = REP_PROCESS_LAST; ///< type of repository processor

	unordered_set<string> req_signals; ///< names of signals required for processsing
	unordered_set<int> req_signal_ids; ///< ids of signals required for processing

	unordered_set<string> aff_signals; ///< names of signals affected by processing
	unordered_set<int> aff_signal_ids; ///< ids of signals affected by processing
	bool unconditional = false; ///< indicated that processor should ALWAYS be applied

	vector<string> attributes; ///< attributes generated by the processor (optional)

	virtual ~RepProcessor() { clear(); }
	virtual void clear() {  };

	/// <summary> 
	/// virtual signals are created only in rep processors but can be used by any rep processor that comes after
	/// or any feture generator as a regular signal.
	/// a virtual signal can be defined as one only once - in the rep processor that actually creates it.
	/// The other rep processors and feature generators using it are simply blind to the fact that it is virtual 
	/// and use it by name as a regular signal.
	/// </summary>
	vector<pair<string, int>> virtual_signals; ///< list of all virtual signals CREATED by this rep processor
	vector<pair<string, string>> virtual_signals_generic;

	// create a new rep_processor
	/// <summary> create a new repository processor from name </summary>
	static RepProcessor *make_processor(string name);
	/// <summary> create a new repository processor from name and a parameters string</summary>
	static RepProcessor *make_processor(string type, string params);
	/// <summary> create a new repository processor from type </summary>
	static RepProcessor *make_processor(RepProcessorTypes type);
	/// <summary> create a new repository processor from type and a parameters string</summary>
	static RepProcessor *make_processor(RepProcessorTypes type, string params);

	/// <summary> create a new repository processor from parameters string which contains rp_type </summary>
	static RepProcessor *create_processor(string &params);

	/// <summary> initialize from a params object :  Should be implemented for inheriting classes that have parameters </summary>
	virtual int init(void *params) { return 0; };
	/// <summary> initialize from a map :  Should be implemented for inheriting classes that have parameters </summary>
	virtual int init(map<string, string>& mapper) { return 0; };
	/// <summary> initialize to default values :  Should be implemented for inheriting classes that have parameters </summary>
	virtual void init_defaults() {};

	/// <summary> set signal-name :  Should be implemented for inheriting classes that have signalName </summary>
	virtual void set_signal(const string& _signalName) { return; };

	/// <summary> set signal-ids :  Should be implemented for inheriting classes that have signalId  </summary>
	virtual void set_signal_ids(MedSignals& sigs) { return; }

	// Required Signals functions : get all signals that are required by the processor
	/// <summary> Append required signal names to set : parent function just uses req_signals  </summary>
	virtual void get_required_signal_names(unordered_set<string>& signalNames);
	// Required Signals functions : get all signals that are required by the processor
	/// <summary> Append required signal names to set only if processor is actually required to produce any of preReqSignals : parent function just uses req_signals  </summary>
	virtual void get_required_signal_names(unordered_set<string>& signalNames, unordered_set<string> preReqSignals);

	/// <summary> Fill req_signal_ids : parent function just fills from req_signals </summary>
	virtual void set_required_signal_ids(MedDictionarySections& dict);

	/// <summary> rep processors CREATING virtual signals need to implement this: adding their signals to the pile </summary>
	virtual void add_virtual_signals(map<string, int> &_virtual_signals, map<string, string> &_virtual_signals_generic) const {
		for (auto &v : virtual_signals) _virtual_signals[v.first] = v.second;
		for (auto &v : virtual_signals_generic) _virtual_signals_generic[v.first] = v.second;
	};

	// Required Signals functions : get all signals that are required by the processor
	/// <summary> Append required signal names to set : parent function just uses req_signals  </summary>
	virtual void get_required_signal_ids(unordered_set<int>& signalIds);
	// Required Signals functions : get all signals that are required by the processor
	/// <summary> Append required signal names to set only if processor is actually required to produce any of preReqSignals : parent function just uses req_signals  </summary>
	virtual void get_required_signal_ids(unordered_set<int>& signalIds, unordered_set<int> preReqSignals);

	// Affected Signals functions;
	/// <summary> Fill aff_signal_ids : parent function just fills from aff_signals </summary>
	virtual void set_affected_signal_ids(MedDictionarySections& dict);
	/// <summary>  Check if a signal is affected by processor </summray>
	/// <returns> true if affected, false if not </returns>
	inline bool is_signal_affected(int signalId) { return (aff_signal_ids.find(signalId) != aff_signal_ids.end()); }
	inline bool is_signal_affected(string& signalName) { return (aff_signals.find(signalName) != aff_signals.end()); }

	/// <summary> make changes to RepProcessor according to available signals in Repository </summary>
	virtual void fit_for_repository(MedPidRepository& rep) {};

	///Register section id to section name of new virtual signals
	virtual void register_virtual_section_name_id(MedDictionarySections& dict) { };

	// check filtering
	/// <summary> Check if processor (and 'sub'-processors within) should be applied according to set of required signals  </summray>
	/// <returns> true if processor is not required and can be filtered, false otherwise </returns>
	virtual bool filter(unordered_set<string>& reqSignals);

	/// <summary> Init required tables : Should be implemented for inheriting classes that have such tables </summary>
	virtual void init_tables(MedDictionarySections& dict, MedSignals& sigs) { return; }

	/// <summary> Init attributes information : Should be implemented for inheriting classes that have attributes </summary>
	virtual void init_attributes() { return; }

	// Learning
	/// <summary> learn processing model on a subset of samples. Apply set of preceeding processors on DynamicPidRec before learning : 
	// Should be implemented for inheriting classes that require learning </summary>
	virtual int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processors) { return 0; };
	/// <summary> learn processing model on a subset of samples only if required. Apply set of preceeding processors on DynamicPidRec before learning : 
	// May be implemented for inheriting classes that require learning </summary>
	virtual int _conditional_learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processors, unordered_set<int>& neededSignalIds);

	// Learning envelopes - Here because of issues with overloading and inheritance
	/// <summary> learn processing model on a subset of ids. Apply set of preceeding processors on DynamicPidRec before learning </summary>
	int learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processors) { return _learn(rep, samples, prev_processors); };
	/// <summary> learn on all pids in repository, using fake samples - works only for repProcessors that ignore sample dates</summary>
	int learn(MedPidRepository& rep);
	/// <summary> learn on subset of samples without preceesing processors  </summary>
	int learn(MedPidRepository& rep, MedSamples& samples) { vector<RepProcessor *> temp;  return _learn(rep, samples, temp); }
	/// <summary> learn processing model on a subset of samples only if required. Apply set of preceeding processors on DynamicPidRec before learning : 
	virtual int conditional_learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processors, unordered_set<int>& neededSignalIds) {
		return _conditional_learn(rep, samples, prev_processors, neededSignalIds);
	}
	/// <summary> learn processing model on a subset of ids only if required without preceesing processors </summary>
	int conditional_learn(MedPidRepository& rep, MedSamples& samples, unordered_set<int>& neededSignalIds) { vector<RepProcessor *> temp;  return _conditional_learn(rep, samples, temp, neededSignalIds); }

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be implemented for all inheriting classes.
	/// <summary> if time_points is empty, processinng is done for each version for all times </summary>
	virtual int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_vals) { return -1; };
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points only if required : May be implemented for inheriting classes </summary>
	virtual int _conditional_apply(PidDynamicRec& rec, vector<int>& time_points, unordered_set<int>& neededSignalIds, vector<vector<float>>& attributes_vals);

	// next is needed for efficient applying of a single records
	virtual int _apply_simple(PidDynamicRec& rec, vector<int>& time_points) {
		vector<vector<float>> attributes_vals;
		return _apply(rec, time_points, attributes_vals);
	}

	// Applying envelopes - Here because of issues with overloading and inheritance
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points</summary>
	int apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_vals) { return _apply(rec, time_points, attributes_vals); }
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points only if required : if any of the signals in neededSignalIds is actually affected by processor </summary>
	int conditional_apply(PidDynamicRec& rec, vector<int>& time_points, unordered_set<int>& neededSignalIds, vector<vector<float>>& attributes_vals) {
		return _conditional_apply(rec, time_points, neededSignalIds, attributes_vals);
	}
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points given by samples </summary>
	int apply(PidDynamicRec& rec, MedIdSamples& samples);
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points given by samples only if required </summary>
	int conditional_apply(PidDynamicRec& rec, MedIdSamples& samples, unordered_set<int>& neededSignalIds);
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points given by samples only if required, not affecting attributes</summary>
	int conditional_apply_without_attributes(PidDynamicRec& rec, const MedIdSamples& samples, unordered_set<int>& neededSignalIds);


	// debug prints
	/// used for debug prints, each inheriting class can overload this one to get a more precise debug print. rp_flag can be used to transfer verbosity levels.
	/// the default print just prints the basic type, etc.
	virtual void dprint(const string &pref, int rp_flag);

	///<summary>
	/// prints summary of rep_processor job. optional, called after apply.
	/// for example - prints how many values were cleaned
	///</summary>
	virtual void make_summary() {};

	/// returns for each used signal it's used categories
	virtual void get_required_signal_categories(unordered_map<string, vector<string>> &signal_categories_in_use) const {};

	// Serialization (including type)
	ADD_CLASS_NAME(RepProcessor)
		ADD_SERIALIZATION_FUNCS(processor_type, req_signals, aff_signals, unconditional)
		void *new_polymorphic(string derived_class_name);

	/// <summary> get size of processor + processor_type </summary>
	size_t get_processor_size();
	/// <summary> seialize processor + processor_type </summary>
	size_t processor_serialize(unsigned char *blob);

	/// <summary> optional printing of processor </summary>
	virtual void print() { fprintf(stderr, "No implementation for print()\n"); }
};

// Utilities
/// <summary> get RepProcessorTypes from name </summary>
RepProcessorTypes rep_processor_name_to_type(const string& procesor_name);

//.......................................................................................
/** RepMultiProcessor is a repository processor which contains a vector of simpler processors that can be
* learned/applied  in parallel. Useful for applying same cleaners on a set of signals, for example
*/
//.......................................................................................
class RepMultiProcessor : public RepProcessor {
public:
	vector<RepProcessor *> processors; ///< Set of processors

	vector<vector<int>> attributes_map; ///< A map from the index of an attribute in the list of attributes of each processor to the index in the list of attributes of the multi-processor

	/// <summary> Constructor </summary>
	RepMultiProcessor() { processor_type = REP_PROCESS_MULTI; };
	~RepMultiProcessor() { clear(); };

	void clear();


	/// <summary> Add processors to set  </summary>
	void add_processors_set(RepProcessorTypes type, vector<string>& signals);
	/// <summary> Add processors to set with initialization string  </summary>
	void add_processors_set(RepProcessorTypes type, vector<string>& signals, string init_string);
	/// <summary> Required Signals ids : Fill the member vector - req_signal_ids </summary>
	void set_required_signal_ids(MedDictionarySections& dict);
	/// <summary> Reporting back virtual signals if there are any </summary>
	void add_virtual_signals(map<string, int> &_virtual_signals, map<string, string> &_virtual_signals_generic) const;

	/// <summary> Required Signals names : Fill the unordered set signalNames </summary>
	void get_required_signal_names(unordered_set<string>& signalNames);
	/// <summary> Append required signal names to set only if processor is actually required to produce any of preReqSignals </summary>
	virtual void get_required_signal_names(unordered_set<string>& signalNames, unordered_set<string> preReqSignals);

	/// <summary> Required Signals ids : Fill the unordered set signalNames </summary>
	void get_required_signal_ids(unordered_set<int>& signalIds);
	/// <summary> Append required signal names to set only if processor is actually required to produce any of preReqSignals </summary>
	virtual void get_required_signal_ids(unordered_set<int>& signalIds, unordered_set<int> preReqSignals);

	/// <summary> Affected Signals : Fill the member set aff_signal_ids </summary>
	void set_affected_signal_ids(MedDictionarySections& dict);


	void register_virtual_section_name_id(MedDictionarySections& dict);
	// check filtering
	/// <summary> Check if processor (and 'sub'-processors within) should be applied according to set of required signals  </summray>
	/// <returns> true if processor is not required and can be filtered, false otherwise </returns>
	bool filter(unordered_set<string>& reqSignals);

	/// <summary> Set signal-ids for all linked signals </summary>
	void set_signal_ids(MedSignals& sigs);

	/// <summary> Init required tables : Should be implemented for inheriting classes that have such tables </summary>
	void init_tables(MedDictionarySections& dict, MedSignals& sigs) { for (RepProcessor * proc : processors) { proc->init_tables(dict, sigs); } }

	/// <summary> Init attributes information : Should be implemented for inheriting classes that have attributes </summary>
	void init_attributes();

	/// <summary> learn processors </summary>
	int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processors);
	int _conditional_learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processors, unordered_set<int>& neededSignalIds);

	/// <summary> Apply processors </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_vals);
	int _apply_simple(PidDynamicRec& rec, vector<int>& time_points);
	/// <summary> Apply processors that affect any of the needed signals </summary>
	int _conditional_apply(PidDynamicRec& rec, vector<int>& time_points, unordered_set<int>& neededSignals, vector<vector<float>>& attributes_mat);

	/// <summary> make changes to RepProcessor according to available signals in Repository </summary>
	void fit_for_repository(MedPidRepository& rep);

	/// debug prints
	void dprint(const string &pref, int rp_flag);

	void make_summary();

	void get_required_signal_categories(unordered_map<string, vector<string>> &signal_categories_in_use) const;

	/// serialization
	ADD_CLASS_NAME(RepMultiProcessor)
		ADD_SERIALIZATION_FUNCS(processor_type, processors)

		/// <summary> Print processors information </summary>
		void print() { for (auto& processor : processors) processor->print(); }
};

#define DEF_REP_TRIMMING_SD_NUM 7
#define DEF_REP_REMOVING_SD_NUM 14

class remove_stats {
public:
	int total_removed = 0, total_pids_touched = 0;
	int total_records = 0, total_pids = 0;

	/// restarts stats for new apply
	void restart();

	/// prints stats to screen for cleaner
	void print_summary(const string &cleaner_info, const string &signal_name,
		int minimal_pid_cnt, float print_summary_critical_cleaned, bool prnt_flg) const;
};

//.......................................................................................
/**
* A simple cleaner considering each value of a certain signal separatley
*/
//.......................................................................................
class RepBasicOutlierCleaner : public RepProcessor, public MedValueCleaner {
private:
	ofstream log_file;
	remove_stats _stats;
	bool is_categ = false;
public:

	string signalName; 	///< name of signal to clean
	int signalId;	///< id of signal to clean
	int time_channel = 0; ///< time channel to consider in cleaning
	int val_channel = 0; ///< value cahnnel to consider in cleaning

	string nRem_attr = ""; ///< Attribute name (in sample) for number of removed. not recorded if empty
	string nTrim_attr = ""; ///< Attribute name (in sample) for number of trimmed. not recorded if empty
	string nRem_attr_suffix = ""; ///< Attribute suffix (name is sample is signalName_suffix) for number of removed. not recorded if empty
	string nTrim_attr_suffix = ""; ///< Attribute suffix (name is sample is signalName_suffix) for number of trimmed. not recorded if empty
	string verbose_file; ///< cleaning output_file for debuging

	bool print_summary = false; ///< If true will always print clean summary
	float print_summary_critical_cleaned = (float)0.05; ///< beyond this value will print summary

	/// <summary> default constructor </summary>
	RepBasicOutlierCleaner() { init_defaults(); }
	/// <summary> default constructor + setting signal name </summary>
	RepBasicOutlierCleaner(const string& _signalName) { init_defaults(); signalId = -1; signalName = _signalName; init_lists(); }
	/// <summary> default constructor + setting signal name + initialize from string </summary>
	RepBasicOutlierCleaner(const string& _signalName, string init_string) { init_defaults(); signalId = -1; signalName = _signalName; init_from_string(init_string); }
	/// <summary> default constructor + setting signal name + initialize from parameters </summary>
	RepBasicOutlierCleaner(const string& _signalName, ValueCleanerParams *_params) { init_defaults(); signalId = -1; signalName = _signalName; init_lists(); MedValueCleaner::init(_params); }

	/// <summary> Initialize to default values </summary>
	void init_defaults() {
		processor_type = REP_PROCESS_BASIC_OUTLIER_CLEANER;
		params.trimming_sd_num = DEF_REP_TRIMMING_SD_NUM; params.removing_sd_num = DEF_REP_REMOVING_SD_NUM; params.nbrs_sd_num = 0;
		params.take_log = 0;
		params.doTrim = params.doRemove = true;
		signalId = -1;
		params.type = VAL_CLNR_ITERATIVE;
		params.missing_value = MED_MAT_MISSING_VALUE;
		verbose_file = "";
	};

	/// <summary> Set signal name and fill affected and required signals sets </summary> 
	void set_signal(const string& _signalName) { signalId = -1; signalName = _signalName; init_lists(); }

	/// <summary> Set signal id </summary>
	virtual void set_signal_ids(MedSignals& sigs);

	/// <summary> Fill required- and affected-signals sets </summary>
	int init(void *processor_params) { return MedValueCleaner::init(processor_params); };
	/// The parsed fields from init command.
	/// @snippet RepProcess.cpp RepBasicOutlierCleaner::init
	virtual int init(map<string, string>& mapper);
	/// Fill req- and aff-signals vectors
	void init_lists();

	/// <summary> Init attributes information : Should be implemented for inheriting classes that have attributes </summary>
	void init_attributes();

	/// <summary> learn cleaning boundaries </summary>
	int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processor);
	/// <summary> Learning : learn cleaning boundaries using MedValueCleaner's iterative approximation of moments </summary>
	int iterativeLearn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processor);
	/// <summary> Learning : learn cleaning boundaries using MedValueCleaner's quantile approximation of moments </summary>
	int quantileLearn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processor);

	/// <summary> Apply cleaning model </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	virtual ~RepBasicOutlierCleaner() { if (!verbose_file.empty() && log_file.is_open()) log_file.close(); };

	/// <summary> Debug printing </summary>
	void print() { dprint("", 1); }
	void dprint(const string &pref, int rp_flag);

	void make_summary();

	/// Serialization
	ADD_CLASS_NAME(RepBasicOutlierCleaner)
		ADD_SERIALIZATION_FUNCS(processor_type, signalName, time_channel, val_channel, req_signals, aff_signals, params.take_log, params.missing_value, params.doTrim, params.doRemove,
			trimMax, trimMin, removeMax, removeMin, nRem_attr, nTrim_attr, nRem_attr_suffix, nTrim_attr_suffix, verbose_file,
			print_summary, print_summary_critical_cleaned)
};

/** Parameters for configured outliers cleaner
*/
//.......................................................................................
class confRecord : public SerializableObject {
public:
	double logicalLow, logicalHigh, confirmedLow, confirmedHigh;
	double trimLow, trimHigh; //<trimming if not above remove borders
	string distLow, distHigh; //"none" "norm" or "log" 
	int val_channel = 0;
	ADD_CLASS_NAME(confRecord)
		ADD_SERIALIZATION_FUNCS(logicalLow, logicalHigh, confirmedLow, confirmedHigh,
			distLow, distHigh, trimLow, trimHigh, val_channel)
};
MEDSERIALIZE_SUPPORT(confRecord)

//.......................................................................................
/** RepConfiguredOutlierCleaner is a simple cleaner considering each value of a certain signal separatley,
* but this time use configuration file that holds for each signal the logical values, statistically confirmed
* values  and distribution for relearning statistical values
*/
//.......................................................................................
class RepConfiguredOutlierCleaner : public  RepBasicOutlierCleaner {
private:
	string confFileName; ///< configuration file and mapping
	confRecord outlierParam; ///< a map from signal name to outliers parameters
public:
	string cleanMethod; ///< cleaning method :  "logical" "confirmed" or "learned"

	RepConfiguredOutlierCleaner() { init_defaults(); }

	/// <summary> Initialize to default values </summary>
	void init_defaults() {
		processor_type = REP_PROCESS_CONFIGURED_OUTLIER_CLEANER;
		params.trimming_sd_num = DEF_REP_TRIMMING_SD_NUM; params.removing_sd_num = DEF_REP_REMOVING_SD_NUM; params.nbrs_sd_num = 0;
		params.take_log = 0;
		params.doTrim = params.doRemove = true;
		signalId = -1;
		params.type = VAL_CLNR_ITERATIVE;
		params.missing_value = MED_MAT_MISSING_VALUE;
		verbose_file = "";
	};

	/// <summary> learn cleaning boundaries </summary>
	int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processor);

	/// The parsed fields from init command.
	/// @snippet RepProcess.cpp RepConfiguredOutlierCleaner::init
	int init(map<string, string>& mapper);

	// Apply cleaning model -inheritted
	void set_signal_ids(MedSignals& sigs);

	/// Serialization
	ADD_CLASS_NAME(RepConfiguredOutlierCleaner)
		ADD_SERIALIZATION_FUNCS(processor_type, signalName, time_channel, val_channel, req_signals, aff_signals, params.take_log, params.missing_value, params.doTrim, params.doRemove,
			trimMax, trimMin, removeMax, removeMin, confFileName, cleanMethod, outlierParam, nRem_attr, nTrim_attr, nRem_attr_suffix, nTrim_attr_suffix)

		void print();
};

void learnDistributionBorders(float& borderHi, float& borderLo, vector<float> filteredValues);
// a function that takes sorted vector of filtered values and estimates the +- 7 sd borders based on the center of distribution
// predefined calibration constants are used for estimation of the borders. 


/**
* A cleaner that is based on rules that describe relations of signal values to each other.\n
* This is a static cleaner ( no learning involved).\n
\n
Rules:\n
Rule1: BMI = Weight / Height ^ 2 * 1e4\n
Rule2:MCH = (Hemoglobin / RBC) * 10(units) \n
Rule3:MCV = (Hematocrit / RBC) * 10(units)\n
Rule4:MCHC - M = (MCH / MCV) * 100 (units)\n
Rule5:Eosinophils# + Monocytes# + Basophils# + Lymphocytes# + Neutrophils# <= WBC\n
Rule6:MPV = Platelets_Hematocrit / Platelets\n
Rule7:UrineAlbumin <= UrineTotalProtein\n
Rule8:UrineAlbumin_over_Creatinine = UrineAlbumin / UrineCreatinine\n
Rule9:LDL + HDL <= Cholesterol\n
Rule10:NonHDLCholesterol + HDL = Cholesterol\n
Rule11:HDL_over_nonHDL = HDL / NonHDLCholesterol\n
Rule12:HDL_over_Cholesterol = HDL / Cholesterol\n
Rule13:HDL_over_LDL = HDL / LDL\n
Rule14:HDL_over_LDL = 1 / LDL_over_HDL\n
Rule15:Cholesterol_over_HDL = Cholesterol / HDL\n
Rule16:---------------------\n
Rule17:Cholesterol_over_HDL = 1 / HDL_over_Cholestrol\n
Rule18:LDL_over_HDL = LDL / HDL\n
Rule19:Albumin <= Protein_Total\n
Rule20:FreeT4 <= T4\n
Rule21:NRBC <= RBC\n
Rule22:CHADS2_VASC >= CHADS2\n
Rule23: BP.Systolic(channel 0) >= BP.Diastoilic(channel 1)
*/
class RepRuleBasedOutlierCleaner : public RepProcessor {
	// get multiple signals and clean them according to consistency  with other signals from same date
public:

	/// Signals to clean
	vector <int> signalIds;
	vector<int> consideredRules;///< only rules in this list will be considered in this cleaner (read list from jason)
								/// rule number 0 means apply all rules. Empty vector: do nothing in this cleaner.

	string nRem_attr = ""; ///< Attribute name (in sample) for number of removed. not recorded if empty
	string nRem_attr_suffix = ""; ///< Attribute suffix (name is sample is signalName_suffix) for number of removed. not recorded if empty

	float tolerance = 0.1F;
	int time_window = 0; ///< the size of time window to search for signals
	string verbose_file; ///< cleaning output_file for debuging
	float calc_res = 0; ///< signal resolution calc, 0 no resolution

	bool print_summary = false; ///< If true will always print clean summary
	float print_summary_critical_cleaned = (float)0.05; ///< beyond this value will print summary

	/// static map from rule to participating signals
	map <int, vector<string>>rules2Signals = {
	{1,{"BMI","Weight","Height"}},
	{2,{"MCH", "Hemoglobin","RBC"}},
	{3,{"MCV","Hematocrit","RBC"} },
	{4,{"MCHC-M","MCH","MCV"}},
	{5,{"Eosinophils#","Monocytes#","Basophils#","Lymphocytes#","Neutrophils#","WBC" }},
	{6,{ "MPV","Platelets_Hematocrit","Platelets" }},
	{7,{"UrineAlbumin","UrineTotalProtein" }},
	{8,{"UrineAlbumin_over_Creatinine","UrineAlbumin","UrineCreatinine" }},
	{9,{"LDL","HDL","Cholesterol"}},
	{10,{"NonHDLCholesterol","HDL","Cholesterol"}},
	{11,{"HDL_over_nonHDL","HDL","NonHDLCholesterol"}},
	{12,{"HDL_over_Cholesterol","HDL","Cholesterol"}},
	{13,{"HDL_over_LDL","HDL","LDL"}},
	{14,{"HDL_over_LDL","LDL_over_HDL"}},
	{15,{"Cholesterol_over_HDL","Cholesterol","HDL"}},
	{17,{"Cholesterol_over_HDL","HDL_over_Cholesterol"}}, //rule 16 canceled
	{18,{"LDL_over_HDL","LDL","HDL"}},
	{19,{"Albumin","Protein_Total"}},
	{20,{"FreeT4","T4"}},
	{21,{"NRBC","RBC"}},
	{22,{"CHADS2","CHADS2_VASC"}},
	{23, {"BP"}}
	};

	map<int, string> rules2RemoveSignal; ///< which signal to remove if contradiction found. If not exists default to remove all

	vector <int> rulesToApply;
	unordered_map<string, pair<int, int>> signal_channels; ///< signal channels (if exists). first is time, second is for val
	unordered_map<int, pair<int, int>> signal_id_channels; ///< signal channels (if exists). first is time, second is for val

	/// Helpers
	set <int> reqSignalIds, affSignalIds;
	unordered_map<int, vector<int>> rules_sids;
	unordered_map<int, vector<bool>> affected_by_rules;

	// Constructors 
	RepRuleBasedOutlierCleaner() : RepProcessor() { init_defaults(); }

	void init_defaults();

	void parse_rules_signals(const string &path);
	void parse_sig_channels(const string &path);


	// Init
	int init(void *processor_params) { return 0; };

	/// The parsed fields from init command.
	/// @snippet RepProcess.cpp RepRuleBasedOutlierCleaner::init
	int init(map<string, string>& mapper);

	/// <summary> Init attributes information : Should be implemented for inheriting classes that have attributes </summary>
	void init_attributes();

	///set signals
	void set_signal_ids(MedSignals& sigs);

	/// <summary> Init required tables : Should be implemented for inheriting classes that have such tables </summary>
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);

	// Learning
	/// <summary> In this class there's never learning - we init tables and return 0 immediately </summary>
	int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processors) { init_tables(rep.dict, rep.sigs); return 0; };

	/// Apply cleaning model 
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	~RepRuleBasedOutlierCleaner() { if (!verbose_file.empty() && log_file.is_open()) log_file.close(); };

	// Check if some required signals are missing and remove corresponding rules
	void fit_for_repository(MedPidRepository& rep);

	// set affected and required signals lists according to rules
	void init_lists();

	void make_summary();

	void dprint(const string &pref, int rp_flag);

	/// Serialization
	ADD_CLASS_NAME(RepRuleBasedOutlierCleaner)
		ADD_SERIALIZATION_FUNCS(processor_type, time_window, calc_res, rules2Signals, rulesToApply, rules2RemoveSignal, signal_channels, consideredRules, tolerance, req_signals, aff_signals, nRem_attr,
			nRem_attr_suffix, verbose_file, print_summary, print_summary_critical_cleaned)

private:
	///ruleUsvs hold the signals in the order they appear in the rule in the rules2Signals above
	/// apply the rule and return true if data is consistent with the rule
	bool  applyRule(int rule, const  vector<UniversalSigVec> &ruleUsvs,
		const vector<int> &val_channels, const vector<int> &sPointer);
	unordered_map<int, string> affected_ids_to_name;
	ofstream log_file;
	unordered_map<string, remove_stats> _rmv_stats;

	/// select which rules to apply according to consideredRules
	void select_rules_to_apply();

};

#define DEF_REP_NBRS_NBRS_SD_NUM 5
#define DEF_REP_NBRS_TRIM_SD_NUM 7
#define DEF_REP_NBRS_REMOVING_SD_NUM 14
//.......................................................................................
/** RepNbrsOutlierCleaner is cleaner that looks at the neighbourhood of a certain signal value
*/
//.......................................................................................
class RepNbrsOutlierCleaner : public RepProcessor, public MedValueCleaner {
public:

	string signalName; ///< name of signal to clean
	int signalId; ///< id of signal to clean
	int time_channel = 0; ///< time channel to consider in cleaning
	int val_channel = 0; ///< value cahnnel to consider in cleaning

	int nbr_time_width = 7; ///< size of neighborhood for defining neighboring values
	int nbr_time_unit = MedTime::Days; ///< time unit for defining neighboring values

	string nRem_attr = ""; ///< Attribute name (in sample) for number of removed. not recorded if empty
	string nTrim_attr = ""; ///< Attribute name (in sample) for number of trimmed. not recorded if empty
	string nRem_attr_suffix = ""; ///< Attribute suffix (name is sample is signalName_suffix) for number of removed. not recorded if empty
	string nTrim_attr_suffix = ""; ///< Attribute suffix (name is sample is signalName_suffix) for number of trimmed. not recorded if empty

	/// <summary> default constructor </summary>
	RepNbrsOutlierCleaner() { init_defaults(); }
	/// <summary> default constructor + setting signal name </summary>
	RepNbrsOutlierCleaner(const string& _signalName) { init_defaults(); signalId = -1; signalName = _signalName; init_lists(); }
	/// <summary> default constructor + setting signal name + initialize from string </summary>
	RepNbrsOutlierCleaner(const string& _signalName, string init_string) { init_defaults(); signalId = -1; signalName = _signalName; init_from_string(init_string); }
	/// <summary> default constructor + setting signal name + initialize from parameters </summary>
	RepNbrsOutlierCleaner(const string& _signalName, ValueCleanerParams *_params) { init_defaults(); signalId = -1; signalName = _signalName; init_lists(); MedValueCleaner::init(_params); }

	/// <summary> Initialize to default values </summary>
	void init_defaults() {
		processor_type = REP_PROCESS_NBRS_OUTLIER_CLEANER;
		params.trimming_sd_num = DEF_REP_NBRS_TRIM_SD_NUM; params.removing_sd_num = DEF_REP_NBRS_REMOVING_SD_NUM; params.nbrs_sd_num = DEF_REP_NBRS_NBRS_SD_NUM;
		params.take_log = 0;
		params.doTrim = params.doRemove = true;
		signalId = -1;
		params.type = VAL_CLNR_ITERATIVE;
		params.missing_value = MED_MAT_MISSING_VALUE;
	};

	/// <summary> Set signal name and fill affected and required signals sets </summary> 
	void set_signal(const string& _signalName) { signalId = -1; signalName = _signalName; init_lists(); }

	/// <summary> Set signal id </summary>
	void set_signal_ids(MedSignals& sigs) { signalId = sigs.sid(signalName); }

	/// <summary> Fill required- and affected-signals sets </summary>
	int init(void *processor_params) { return MedValueCleaner::init(processor_params); };
	/// The parsed fields from init command.
	/// @snippet RepProcess.cpp RepNbrsOutlierCleaner::init
	int init(map<string, string>& mapper);
	void init_lists();
	/// <summary> Init attributes information : Should be implemented for inheriting classes that have attributes </summary>
	void init_attributes();

	/// <summary> learn cleaning boundaries </summary>
	int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processor);
	/// <summary> Learning : learn cleaning boundaries using MedValueCleaner's iterative approximation of moments </summary>
	int iterativeLearn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processor);
	/// <summary> Learning : learn cleaning boundaries using MedValueCleaner's quantile approximation of moments </summary>
	int quantileLearn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processor);

	/// <summary> Apply cleaning model </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	// Serialization
	ADD_CLASS_NAME(RepNbrsOutlierCleaner)
		ADD_SERIALIZATION_FUNCS(processor_type, signalName, time_channel, val_channel, req_signals, aff_signals, params.take_log, params.missing_value, params.doTrim, params.doRemove,
			trimMax, trimMin, removeMax, removeMin, nbr_time_unit, nbr_time_width, nbrsMax, nbrsMin, nRem_attr, nTrim_attr, nRem_attr_suffix, nTrim_attr_suffix)

		/// <summary> Print processors information </summary>
		void print();
};

//.......................................................................................
/** SimValHandler handles multiple values at the same time **/
//.......................................................................................

// Modes of handling multiple values at the same time
typedef enum {
	SIM_VAL_FIRST_VAL,
	SIM_VAL_LAST_VAL,
	SIM_VAL_MEAN,
	SIM_VAL_REM,
	SIM_VAL_REM_DIFF,
	SIM_VAL_MIN,
	SIM_VAL_MAX,
	MULT_VAL_LAST
} SimValHandleTypes;

class RepSimValHandler : public RepProcessor {
public:
	string signalName; 	///< name of signal to handle
	int signalId;	///< id of signal to handle
	vector<int> time_channels; ///< time channels to consider. All if empty
	SimValHandleTypes handler_type; ///< type of handling multiple-values

	int nValChannels; ///< number of value-channels, important for rem-diff and mean modes

	string nHandle_attr = ""; ///< Attribute name (in sample) for number of multiple-values handled. not recorded if empty
	string nHandle_attr_suffix = ""; ///< Attribute suffix (name is sample is signalName_suffix) for number of multiple-values handled. not recorded if empty
	bool debug = false; ///<If True will print out till 3 examples for samples have been changed

	/// <summary> default constructor </summary>
	RepSimValHandler() { init_defaults(); processor_type = REP_PROCESS_SIM_VAL; }
	/// <summary> default constructor + setting signal name </summary>
	RepSimValHandler(const string& _signalName) { init_defaults(); processor_type = REP_PROCESS_SIM_VAL;; signalId = -1; signalName = _signalName; init_lists(); }
	/// <summary> default constructor + setting signal name + initialize from string </summary>
	RepSimValHandler(const string& _signalName, string init_string) { init_defaults(); processor_type = REP_PROCESS_SIM_VAL; signalId = -1; signalName = _signalName; init_from_string(init_string); }

	/// <summary> Set signal name and fill affected and required signals sets </summary> 
	void set_signal(const string& _signalName) { signalId = -1; signalName = _signalName; init_lists(); }

	/// <summary> Set signal id </summary>
	void set_signal_ids(MedSignals& sigs) { signalId = sigs.sid(signalName); }

	/// The parsed fields from init command.
	/// @snippet RepProcess.cpp RepSimValHandler::init
	virtual int init(map<string, string>& mapper);

	/// Fill req- and aff-signals vectors
	void init_lists();

	/// init tables - get time-channels
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);

	/// <summary> Init attributes information : Should be implemented for inheriting classes that have attributes </summary>
	void init_attributes();

	/// <summary> Apply multiple-value handling </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	/// <summary> get SimHandleType from name </summary>
	/// @snippet RepProcess.cpp RepSimValHandler::get_sim_val_handle_type
	static SimValHandleTypes get_sim_val_handle_type(string& name);

	/// Serialization
	ADD_CLASS_NAME(RepSimValHandler)
		ADD_SERIALIZATION_FUNCS(processor_type, signalName, time_channels, req_signals, aff_signals, nHandle_attr, nHandle_attr_suffix, handler_type)
private:
	void handle_block(int start, int end, UniversalSigVec& usv, vector<int>& remove, int& nRemove, vector<pair<int, vector<float>>>& change, int& nChange, int& nTimes);
	int verbose_cnt = 0;
};

//.......................................................................................
/** RepPanelCompleter fills-in calculatable signal values. Enriching existing signals

	The available completions are currently -
		1. RedLineCompleter : MCV,HCT,RBC,MCH,MCHC,HGB
		2. WhiteLineCompleter : WBC,Eosonophils#,Eosonophils%,Neutrophils#,Neutrophils%,Lymphocytes#,Lymphocytes%,Monocytes#,Monocytes%,Basophils#,Basophils%
		3. PlateletsCompleter : Platelets, Platelets_Hematocrit and MPV
		4. LipidsCompleter : Cholesterol,LDL,HDL,HDL_over_Cholesterol,Cholesterol_over_HDL,HDL_over_LDL,LDL_over_HDL,NonHDLCholesterol,HDL_over_nonHDL,Tryglicerids
		5. eGFRCompleter : Creatinine, eGFR_CKD_EPI,eGFR_MDRD
		6. BMICompleter : BMI,Weight,Height

	Signals above are the default values. They can be changed, keeping the order, or set to NULL, in which case the completion is not-performed

*/
//.......................................................................................

typedef enum {
	REP_CMPLT_RED_LINE_PANEL, ///< complete values of the red blood line
	REP_CMPLT_WHITE_LINE_PANEL, ///< complete values of the white blood line
	REP_CMPLT_PLATELETS_PANEL, ///< complete values of platelets measurements
	REP_CMPLT_LIPIDS_PANEL, ///< complete lipd values
	REP_CMPLT_EGFR_PANEL, ///< complete eGFR values
	REP_CMPLT_BMI_PANEL, ///< complete BMI/HIGHT/WEIGHT values
	REP_CMPLT_GCS, ///< complete Glasgow Coma Score
	REP_CMPLT_LAST ///< Last not in use
} PanelCompleterTypes;

typedef enum {
	RED_PNL_MCV,
	RED_PNL_HCT,
	RED_PNL_RBC,
	RED_PNL_MCH,
	RED_PNL_MCHC,
	RED_PNL_HGB,
	RED_PNL_LAST
} RedPanelSignals;

typedef enum {
	WHITE_PNL_WBC,
	WHITE_PNL_EOS_N, WHITE_PNL_EOS_P,
	WHITE_PNL_NEU_N, WHITE_PNL_NEU_P,
	WHITE_PNL_LYM_N, WHITE_PNL_LYM_P,
	WHITE_PNL_MON_N, WHITE_PNL_MON_P,
	WHITE_PNL_BAS_N, WHITE_PNL_BAS_P,
	WHITE_PNL_LAST
} WhitePanelSignals;

typedef enum {
	PLT_PNL_PLTS,
	PLT_PNL_PLT_HCT,
	PLT_PNL_MPV,
	PLT_PNL_LAST
} PltsPanelSignals;

typedef enum {
	LIPIDS_PNL_CHOL,
	LIPIDS_PNL_LDL,
	LIPIDS_PNL_HDL,
	LIPIDS_PNL_HDL_OVER_CHOL,
	LIPIDS_PNL_CHOL_OVER_HDL,
	LIPIDS_PNL_HDL_OVER_LDL,
	LIPIDS_PNL_LDL_OVER_HDL,
	LIPIDS_PNL_NON_HDL_CHOL,
	LIPIDS_PNL_HDL_OVER_NON_HDL,
	LIPIDS_PNL_TRGS,
	LIPIDS_PNL_VLDL,
	LIPIDS_PNL_LAST
} LipidsPanelSignals;

typedef enum {
	EGFR_PNL_CRT,
	EGFR_PNL_CKD_EPI,
	EGFR_PNL_MDRD,
	EGFR_PNL_LAST
} eGFRPanelSignals;

typedef enum {
	BMI_PNL_BMI,
	BMI_PNL_WGT,
	BMI_PNL_HGT,
	BMI_PNL_HGT_SQR,
	BMI_PNL_LAST
} BMIPanelSignals;

typedef enum {
	GCS_PNL,
	GCS_PNL_EYE,
	GCS_PNL_MOTOR,
	GCS_PNL_VERBAL,
	GCS_PNL_LAST
} GCSPanelSignals;

/**
* A Repository panel completer for a complete panel of signals that are dependent and given on same time.
* TODO: add support for look in the past threshold
*/
class RepPanelCompleter : public RepProcessor {
public:
	// Signals for completions
	vector<vector<string> > panel_signal_names;
	vector<vector<int> > panel_signal_ids;

	// Extra signal ids
	int bdateId, genderId;
	string genderSignalName;

	// Missing value indication
	float missing_val = MED_MAT_MISSING_VALUE;

	// Handling multiple values
	SimValHandleTypes sim_val_handler = SIM_VAL_LAST_VAL;

	// Signals meta-data : original and final resolution and factors
	string metadata_file;
	vector<vector<float> > original_sig_res, final_sig_res, sig_conversion_factors;

	RepPanelCompleter() { processor_type = REP_PROCESS_COMPLETE; init_defaults(); }

	/// @snippet PanelCompleter.cpp RepPanelCompleter::init
	int init(map<string, string>& mapper);

	/// <summary> initialize to default values :  Should be implemented for inheriting classes that have parameters </summary>
	void init_defaults();

	// Change signal names from defualt
	int update_signal_names(string panel, string& names);

	// Change panels to handle
	int update_panels(string& panels);

	// initialize signal ids
	void set_signal_ids(MedSignals& sigs);

	// Check if some required signals are missing and make them virtual or remove relevant panel completer
	void fit_for_repository(MedPidRepository& rep);

	// dictionary based initializations
	void init_tables(MedDictionarySections &dict, MedSignals& sigs);

	/// Fill req- and aff-signals vectors
	void init_lists();

	// Read conversion and resolution info
	void read_metadata();

	// Learning
	/// <summary> In this class there's never learning - we init tables and return 0 immediately </summary>
	int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processors) { init_tables(rep.dict, rep.sigs); read_metadata(); return 0; };

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be implemented for all inheriting classes </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	// calculators implementations
	int apply_red_line_completer(PidDynamicRec& rec, vector<int>& time_points);
	int apply_white_line_completer(PidDynamicRec& rec, vector<int>& time_points);
	int apply_platelets_completer(PidDynamicRec& rec, vector<int>& time_points);
	int apply_lipids_completer(PidDynamicRec& rec, vector<int>& time_points);
	int apply_eGFR_completer(PidDynamicRec& rec, vector<int>& time_points);
	int apply_BMI_completer(PidDynamicRec& rec, vector<int>& time_points);
	int apply_GCS_completer(PidDynamicRec& rec, vector<int>& time_points);

	// Utilities
	// Age/Gender
	int perpare_for_age_and_gender(PidDynamicRec& rec, int& age, int& bYear, int& gender);

	// Rounding - take care of resolution both in original and final units 
	inline float completer_round(float value, float orig_res, float final_res, float factor) { return  set_resolution(set_resolution(value / factor, orig_res) * factor, final_res); }
	inline float set_resolution(float value, float res) { return res * std::round(value / res); }

	// Generating panels from usvs
	void get_panels(vector<UniversalSigVec>& usvs, vector<int>& panel_times, vector<vector<float>>& panels, int time_limit, int panel_size);

	// Applying formulas
	int triplet_complete(vector<float>& panel, float factor, int x_idx, int y_idx, int z_idx, vector<float>& orig_res, vector<float>& final_res, vector<float>& conv, vector<int>& changed);
	int sum_complete(vector<float>& panel, int sum, vector<int>& summands, vector<float>& orig_res, vector<float>& final_res, vector<float>& conv, vector<int>& changed);
	int reciprocal_complete(vector<float>& panel, float factor, int x_idx, int y_idx, vector<float>& orig_res, vector<float>& final_res, vector<float>& conv, vector<int>& changed);
	int egfr_complete(vector<float>& panel, float age, int gender, vector<float>& orig_res, vector<float>& final_res, vector<float>& conv, vector<int>& changed);

	// Updating signals in dynamic-rec
	int update_signals(PidDynamicRec& rec, int iver, vector<vector<float>>& panels, vector<int>& panel_times, vector<int>& sigs_ids, vector<int>& changed);

	// serialization. meta-data file is kept for information but not used in apply
	void print();
	ADD_CLASS_NAME(RepPanelCompleter)
		ADD_SERIALIZATION_FUNCS(processor_type, panel_signal_names, missing_val, sim_val_handler, original_sig_res, final_sig_res, sig_conversion_factors, metadata_file, req_signals, aff_signals, virtual_signals)

private:

	map<string, PanelCompleterTypes> panel2type = {
		{ "red_line",REP_CMPLT_RED_LINE_PANEL },
		{ "white_line",REP_CMPLT_WHITE_LINE_PANEL },
		{ "platelets",REP_CMPLT_PLATELETS_PANEL },
		{ "lipids",REP_CMPLT_LIPIDS_PANEL },
		{ "egfr",REP_CMPLT_EGFR_PANEL },
		{ "bmi",REP_CMPLT_BMI_PANEL },
		{ "gcs", REP_CMPLT_GCS }
	};

	// definitions and defaults for each panel-completer
	map<string, vector<string> > panel2signals = {
		{ "red_line", {"MCV","Hematocrit","RBC","MCH","MCHC-M","Hemoglobin"}},
		{ "white_line", {"WBC", "Eosinophils#", "Eosinophils%", "Neutrophils#", "Neutrophils%", "Lymphocytes#", "Lymphocytes%", "Monocytes#", "Monocytes%", "Basophils#", "Basophils%"}},
		{ "platelets", {"Platelets", "Platelets_Hematocrit","MPV"}},
		{ "lipids",{"Cholesterol", "LDL", "HDL", "HDL_over_Cholesterol", "Cholesterol_over_HDL", "HDL_over_LDL", "LDL_over_HDL", "NonHDLCholesterol", "HDL_over_nonHDL", "Triglycerides", "VLDL"}},
		{ "egfr", {"Creatinine","eGFR_CKD_EPI","eGFR_MDRD"}},
		{ "bmi", {"BMI","Weight","Height"}},
		{"gcs", {"GCS", "GCS_Eye", "GCS_Motor", "GCS_Verbal"}}
	};

	vector<int> white_panel_nums = { WHITE_PNL_EOS_N,WHITE_PNL_NEU_N,WHITE_PNL_LYM_N,WHITE_PNL_MON_N,WHITE_PNL_BAS_N };
	vector<int> white_panel_precs = { WHITE_PNL_EOS_P,WHITE_PNL_NEU_P,WHITE_PNL_LYM_P,WHITE_PNL_MON_P,WHITE_PNL_BAS_P };
	vector<int> chol_types1 = { LIPIDS_PNL_NON_HDL_CHOL,LIPIDS_PNL_HDL };
	vector<int> chol_types2 = { LIPIDS_PNL_LDL, LIPIDS_PNL_HDL, LIPIDS_PNL_VLDL };
	vector<int> gcs_panel_parts = { GCS_PNL_EYE,GCS_PNL_MOTOR,GCS_PNL_VERBAL };

	//Convertion map for GCS:
	unordered_map<int, int> eye_vals, verbal_vals, motor_vals;
	void convert_gcs_signals(vector<float> &panel);
};

//.......................................................................................


//.......................................................................................
/** RepCalcSimpleSignals is a rep processor containing several calculators to calculate new
	signals. It supports an internal list of calculators, and the user can select one of them.

	Each calculator can create one or more new virtual signals. Each of these has a specific type.
	The user can configure the virtual signal names, and can also pass a list of float parameters to
	the calculator, making it parametric.

	When adding a new calculator make sure to :
	(1) fill in the calc2defs map (as explained below), to define the calculator name, type, req signals,
		default virual names and their types, and default parameters.
	(2) add the new calculator type to the enum
	(3) write the matching _apply function for the specific calculator and make sure the general _apply calls it .

	NOTE: in RepCalcSimpleSignals there is NO learning. Calculated signals that have a learning stage should
		  be implmented in a separate class that will also have space to keep the learning process results.

	supported signals calculated (this will be also the virtual signal name):

	calc_eGFR : calculating EGFR (CKD_EPI formula) at each point in which Creatinine is available

*/
//.......................................................................................
/**
* Abstract class for calculator - has do_calc which recieves vector for values to calculate
*/
class SimpleCalculator : public SerializableObject {
public:
	vector<string> output_signal_names;
	float missing_value = (float)MED_MAT_MISSING_VALUE; ///< missing value 
	string calculator_name = ""; ///< just for debuging
	int work_channel = 0; ///< the working channel
	bool need_time = false; ///< if needed time
	bool keep_only_in_range = false; ///< keeps only in range values

	///init function of calculator
	virtual int init(map<string, string>& mapper) { return 0; };
	///validates correctness of inputs
	virtual void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const {};
	/// the calc option
	virtual bool do_calc(const vector<float> &vals, float &res) const { HMTHROW_AND_ERR("Error %s::do_calc not implemented\n", my_class_name().c_str()); };
	/// list output signals with default naming
	virtual void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type) { HMTHROW_AND_ERR("Error %s::do_calc not implemented\n", my_class_name().c_str()); };
	/// init operator based on repo if needed
	virtual void init_tables(MedDictionarySections& dict, MedSignals& sigs, const vector<string> &input_signals) {};

	virtual void get_required_signal_categories(unordered_map<string, vector<string>> &signal_categories_in_use) const {};

	virtual void fit_for_repository(MedPidRepository &rep, vector<pair<string, string>> &_virtual_signals) {};

	/// @snippet RepProcess.cpp SimpleCalculator::make_calculator
	static SimpleCalculator *make_calculator(const string &calc_type);

	virtual ~SimpleCalculator() {};

	void *new_polymorphic(string derived_class_name);

	ADD_CLASS_NAME(SimpleCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range)


};

/**
* Dummy top create empty virtual signal
*/
class EmptyCalculator : public SimpleCalculator {
public:
	EmptyCalculator() { calculator_name = "empty"; keep_only_in_range = false; };

	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);
	bool do_calc(const vector<float> &vals, float &res) const;

	ADD_CLASS_NAME(EmptyCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range)
};

/**
* Calcs ratio between 2 inputs signals. has ability to power each signal before dividing and multiply by factor
* Calc is when we have V = { V1, V2 } as inputs. res := factor * V1^power_mone / V2^power_base
*/
class RatioCalculator : public SimpleCalculator {
public:
	float factor = 1; ///< factor multiply
	float power_base = 1; ///< power base input
	float power_mone = 1; ///< power mone input

	RatioCalculator() { calculator_name = "ratio"; keep_only_in_range = false; };
	/// @snippet RepCalculators.cpp RatioCalculator::init
	int init(map<string, string>& mapper);

	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);
	bool do_calc(const vector<float> &vals, float &res) const;

	ADD_CLASS_NAME(RatioCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range, factor, power_base, power_mone)
};

/**
* KFRE calculator based on Age, Gender and eGFR (in that order)
*/
class KfreCalculator : public SimpleCalculator {
public:

	// Valid values are 2,3 or 6
	//int model_id = 2;
	int n_variables = 4;

	int kfre_version = 2;
	int prediction_years = 5;
	std::string region = "original";		// original,north_american,non_noth_american,pooled
	int region_id;
	bool discard_range_check = false;		// when true, allow computation for input parameters out of valid range

	std::map<std::string, int> region2id = {
		{ "original",			0 },
		{ "north_american",		1 },
		{ "non_north_american",	2 },
		{ "pooled",				3 },
	};

	KfreCalculator() { calculator_name = "KFRE"; need_time = true; keep_only_in_range = true; };
	int init(map<string, string>& mapper);
	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);
	bool do_calc(const vector<float> &vals, float &res) const;

	ADD_CLASS_NAME(KfreCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, n_variables, n_variables, prediction_years, kfre_version, region, discard_range_check)
};

/**
* eGFR calculator based on Creatinine, Gender and Age.
* the signals input order is: Creatinine, Gender, Age
*/
class eGFRCalculator : public SimpleCalculator {
public:
	float ethnicity = 0; ///< ethnicity, for now only support 0
	bool mdrd = false; ///< If true will use MDRD calculation

	eGFRCalculator() { calculator_name = "eGFR_CKD_EPI"; need_time = true; keep_only_in_range = true; };
	/// @snippet RepCalculators.cpp eGFRCalculator::init
	int init(map<string, string>& mapper);
	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);
	bool do_calc(const vector<float> &vals, float &res) const;

	ADD_CLASS_NAME(eGFRCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range, ethnicity, mdrd)
};

/**
* does log operation
*/
class logCalculator : public SimpleCalculator {
public:
	logCalculator() { calculator_name = "log"; keep_only_in_range = false; };

	/// @snippet RepCalculators.cpp logCalculator::init
	int init(map<string, string>& mapper);
	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);

	bool do_calc(const vector<float> &vals, float &res) const;

	ADD_CLASS_NAME(logCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range)
};

/**
* does linear sum with factors on vector of input signals, has b0 as constant
* res := b0 + sum_sigma(i=1..N){ factor[i] * input[i]}
*/
class SumCalculator : public SimpleCalculator {
public:
	vector<float> factors; ///< the factors given by ","
	float b0 = 0; ///< a constant

	SumCalculator() { calculator_name = "sum"; };
	/// @snippet RepCalculators.cpp SumCalculator::init
	int init(map<string, string>& mapper);
	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);
	bool do_calc(const vector<float> &vals, float &res) const;

	ADD_CLASS_NAME(SumCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range, factors, b0)
};

/**
* A simple Range check that return in_range_val if within range and returns out_range_val
* if outside range.
*/
class RangeCalculator : public SimpleCalculator {
public:
	float min_range; ///< min range check
	float max_range; ///< max range check
	float in_range_val = 1; ///< return value when within range
	float out_range_val = 0; ///< return value when not within range

	RangeCalculator() { calculator_name = "range"; min_range = MED_MAT_MISSING_VALUE; max_range = MED_MAT_MISSING_VALUE; };
	/// @snippet RepCalculators.cpp RangeCalculator::init
	int init(map<string, string>& mapper);
	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);
	bool do_calc(const vector<float> &vals, float &res) const;

	ADD_CLASS_NAME(RangeCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range, min_range, max_range, in_range_val, out_range_val)
};

/**
* A multiply operation with power on both arguments. V = { V1, V2 }
* res := b0 * pie_multiply(i=1..N) {input[i]^powers[i]}
*/
class MultiplyCalculator : public SimpleCalculator {
public:
	vector<float> powers; ///< power for args
	float b0 = 1; ///< init value

	MultiplyCalculator() { calculator_name = "multiply"; };
	/// @snippet RepCalculators.cpp MultiplyCalculator::init
	int init(map<string, string>& mapper);

	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);

	bool do_calc(const vector<float> &vals, float &res) const;

	ADD_CLASS_NAME(MultiplyCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range, powers, b0)
};

/**
* A is in set operation which return binary output
* res := in_range_val if is in set otherwise out_range_val
*/
class SetCalculator : public SimpleCalculator {
public:
	vector<string> sets;
	float in_range_val = 1; ///< return value when within range
	float out_range_val = 0; ///< return value when not within range
	bool regex_on_sets = false;
	SetCalculator() { calculator_name = "set"; };
	/// @snippet RepCalculators.cpp SetCalculator::init
	int init(map<string, string>& mapper);

	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);
	void init_tables(MedDictionarySections& dict, MedSignals& sigs, const vector<string> &input_signals);

	bool do_calc(const vector<float> &vals, float &res) const;

	void get_required_signal_categories(unordered_map<string, vector<string>> &signal_categories_in_use) const;

	ADD_CLASS_NAME(SetCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range, input_signal, sets, in_range_val, out_range_val)
private:
	vector<char> Flags;
	string input_signal = "";
};

/**
* A is in as exists operation which return binary output
* res := in_range_val if signal exists otherwise out_range_val
*/
class ExistsCalculator : public SimpleCalculator {
public:
	float in_range_val = 1; ///< return value when within range
	float out_range_val = 0; ///< return value when not within range

	ExistsCalculator() { calculator_name = "exists"; keep_only_in_range = true; need_time = true; };
	/// @snippet RepCalculators.cpp ExistsCalculator::init
	int init(map<string, string>& mapper);

	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);

	bool do_calc(const vector<float> &vals, float &res) const;

	ADD_CLASS_NAME(ExistsCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range, in_range_val, out_range_val)
};

class ConstantValueCalculator : public SimpleCalculator {
private:
	float numeric_val;
public:
	string value;
	bool is_numeric;
	vector<string> additional_dict_vals;
	ConstantValueCalculator() {
		calculator_name = "constant_value"; is_numeric = false; value = ""; numeric_val = 1;
	};

	/// @snippet RepCalculators.cpp ConstantValueCalculator::init
	int init(map<string, string>& mapper);

	void validate_arguments(const vector<string> &input_signals, const vector<string> &output_signals) const;
	void list_output_signals(const vector<string> &input_signals, vector<pair<string, string>> &_virtual_signals, const string &output_type);

	bool do_calc(const vector<float> &vals, float &res) const;

	void fit_for_repository(MedPidRepository &rep, vector<pair<string, string>> &_virtual_signals);

	ADD_CLASS_NAME(ConstantValueCalculator)
		ADD_SERIALIZATION_FUNCS(calculator_name, missing_value, work_channel, need_time, keep_only_in_range, value, is_numeric, numeric_val, output_signal_names, additional_dict_vals)
};

/**
* A simple class for calculating virtual signals. please reffer to SimpleCalculator to see which
* calculation are supported right now
*/
class RepCalcSimpleSignals : public RepProcessor {

public:
	vector<string> V_names; ///< names of signals created by the calculator (a calculator can create more than a single signal at a time)

	string calculator; ///< calculator asked for by user
	int work_channel = 0; ///< the channel to work on all singals - and save results to
	int time_channel = 0; ///<the time channel. todo in the future - accept array with same order and size of signals
	string output_signal_type; ///< a parameter to be pass to list signal to afect output type

	float missing_value = (float)MED_MAT_MISSING_VALUE;

	//vector<float> coeff; ///< it is possible to transfer a vector of params to the calculator, to enable parametric calculators.

	vector<string> signals; ///< it is possible to transfer a vector of required signals, to override default ones.
	int signals_time_unit = MedTime::Undefined; ///< Time unit of timer and all signals 

	int max_time_search_range = 0; ///< how much time we are allowed to look backward to calculate. to look forward we need to fix the function
	string calculator_init_params = ""; ///< string init params for calculator

	RepCalcSimpleSignals() {
		processor_type = REP_PROCESS_CALC_SIGNALS;
		output_signal_type = "T(i),V(f)";
	}
	~RepCalcSimpleSignals();

	/// @snippet RepProcess.cpp RepCalcSimpleSignals::init
	int init(map<string, string>& mapper);

	// making sure V_ids and sigs_ids are initialized
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);

	void fit_for_repository(MedPidRepository &rep);

	// Learning
	/// <summary> In this class there's never learning - we return 0 immediately </summary>
	int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processors) { init_tables(rep.dict, rep.sigs); return 0; };

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be implemented for all inheriting classes </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	void print();

	void get_required_signal_categories(unordered_map<string, vector<string>> &signal_categories_in_use) const;

	void register_virtual_section_name_id(MedDictionarySections& dict);

	// serialization
	ADD_CLASS_NAME(RepCalcSimpleSignals)
		ADD_SERIALIZATION_FUNCS(processor_type, calculator, calculator_init_params, max_time_search_range, signals_time_unit,
			signals, V_names, req_signals, aff_signals, virtual_signals, virtual_signals_generic, work_channel, time_channel, calculator_logic, output_signal_type)
		void post_deserialization() {
		SimpleCalculator *p = SimpleCalculator::make_calculator(calculator);
		pass_time_last = p->need_time;
		delete p;
		//if (calculator_logic == NULL) {
		//	printf("write calculator_logic again for %s\n%s\n",
		//		calculator.c_str(), calculator_init_params.c_str());
		//	calculator_logic = SimpleCalculator::make_calculator(calculator);
		//	if (!calculator_init_params.empty())
		//		calculator_logic->init_from_string(calculator_init_params);
		//	calculator_logic->missing_value = missing_value;
		//	vector<pair<string, string>> default_virtual_signals;
		//	calculator_logic->list_output_signals(signals, default_virtual_signals); //init calculator
		//}

	}

private:
	// definitions and defaults for each calculator - all must be filled in for a new calculator
	bool pass_time_last = false; ///< pass last signal as time
	/// from a calculator name to the list of required signals
	const map<string, vector<string>> calc2req_sigs = {
		//--------- level 1 - calculated from raw signals (level0)
		//the general hospital processor's signals must be overridden from outside
		{ "calc_eGFR", {"Creatinine", "GENDER", "BDATE"}}
	};

	vector<int> V_ids; ///< ids of signals created by the calculator (for faster usage at run time: save name conversions)
	vector<int> sigs_ids; /// <ids of signals used as input by the calculator (for faster usage at run time: save name conversions)
	vector<bool> static_input_signals;

	int apply_calc_in_time(PidDynamicRec& rec, vector<int>& time_points);

	SimpleCalculator *calculator_logic = NULL;
	int out_n_val_ch, out_n_time_ch;
};

/**
* RepCreateReigsty creates registries (e.g. Diabetes, HyperTension, CKD, etc) as virtual signals
*/



/**
* Combines multiple signals to one signal. has ability to factorize each source signal
*/
class RepCombineSignals : public RepProcessor {
public:
	string output_name; ///< names of signal created by the processor
	vector<string> signals; ///< names of input signals used by the processor
	vector<float> factors; ///< factor for each signal
	int factor_channel; ///< the factor_channel_number
	string signal_type; ///< the signal type definition to create

	RepCombineSignals() {
		processor_type = REP_PROCESS_COMBINE;
		output_name = "";
		factor_channel = 1;
		signal_type = "T(i,i),V(f,f)";
	}

	void register_virtual_section_name_id(MedDictionarySections& dict);

	/// @snippet RepProcess.cpp RepCombineSignals::init
	int init(map<string, string>& mapper);

	void init_tables(MedDictionarySections& dict, MedSignals& sigs);
	void set_required_signal_ids(MedDictionarySections& dict) {};
	void set_affected_signal_ids(MedDictionarySections& dict) {};

	void fit_for_repository(MedPidRepository& rep);
	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be implemented for all inheriting classes </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	void print();
	ADD_CLASS_NAME(RepCombineSignals)
		ADD_SERIALIZATION_FUNCS(processor_type, output_name, signals, factors,
			unconditional, req_signals, aff_signals, virtual_signals,
			virtual_signals_generic, signal_type, factor_channel)
private:
	int v_out_sid = -1;
	vector<int> sigs_ids;
	int v_out_n_time_ch, v_out_n_val_ch;
};

/**
* split signal based on set of values. supports only 2 channel for value, first is used
* for search in sets (and keeps the same value) 2 is being factorized
* output signal is haveing 1 time channel and 2 value channels
*/
class RepSplitSignal : public RepProcessor {
public:
	string input_name; ///< names of input signal used by the processor
	vector<string> names; ///< names of signal created by the processor
	vector<float> factors; ///< factor for each output signal
	vector<string> sets; ///< the sets to check if signal value is in set
	int val_channel; ///< the val channel to look for the sets
	string output_signal_type; ///< same as input signal - will remove later after change init process of RepProcessor to fetch input signal type

	RepSplitSignal() { processor_type = REP_PROCESS_SPLIT; input_name = ""; val_channel = 0; output_signal_type = "T(i),T(i),V(f),V(f)"; }

	void register_virtual_section_name_id(MedDictionarySections& dict);

	/// initialize signal ids
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);
	void set_required_signal_ids(MedDictionarySections& dict) {};
	void set_affected_signal_ids(MedDictionarySections& dict) {};

	/// @snippet RepProcess.cpp RepSplitSignal::init
	int init(map<string, string>& mapper);

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be implemented for all inheriting classes </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	void get_required_signal_categories(unordered_map<string, vector<string>> &signal_categories_in_use) const;

	void print();
	ADD_CLASS_NAME(RepSplitSignal)
		ADD_SERIALIZATION_FUNCS(processor_type, input_name, names, factors, sets, unconditional,
			req_signals, aff_signals, virtual_signals, virtual_signals_generic, val_channel, output_signal_type)
private:
	int in_sid = -1;
	vector<int> V_ids;
	vector<char> Flags;
	int v_out_n_time_ch, v_out_n_val_ch;
};


/**
* creates a signal of time ranges in which the input signal (usually medication) was active
* ranges are combined based on the period parameter
* the signal value is not transfered to the output signal
*/
class RepAggregationPeriod : public RepProcessor {
public:
	string input_name; ///< name of input signal used by the processor
	string output_name; ///< name of signal created by the processor
	vector<string> sets; ///< the sets to check if signal value is in set
	int period; ///< period to consider active after signal in win time units

	int time_unit_win = MedTime::Undefined;		///< the time unit in which the period is given. Default: Undefined
	int time_unit_sig = MedTime::Undefined;		///< the time init in which the signal is given. Default: Undefined

private:
	int in_sid;
	vector<int> V_ids;
	vector<char> lut;

public:
	RepAggregationPeriod() :
		input_name(""), output_name(""), period(0), time_unit_sig(global_default_windows_time_unit), time_unit_win(global_default_windows_time_unit), in_sid(-1) {
		processor_type = REP_PROCESS_AGGREGATION_PERIOD;
	}

	/// initialize signal ids
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);

	/// @snippet RepProcess.cpp RepAggregationPeriod::init
	int init(map<string, string>& mapper);

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be implemented for all inheriting classes </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	void print();

	void get_required_signal_categories(unordered_map<string, vector<string>> &signal_categories_in_use) const;

	ADD_CLASS_NAME(RepAggregationPeriod)
		ADD_SERIALIZATION_FUNCS(processor_type, input_name, output_name, sets, period, req_signals, aff_signals, virtual_signals, virtual_signals_generic, time_unit_win, time_unit_sig, in_sid, V_ids, lut)

};

/**
* A filter on signal events using other range signal with time periods
*/
typedef enum {
	all = 0,
	first = 1,
	last = 2
} range_op_type;
class RepBasicRangeCleaner : public RepProcessor {
private:
	vector<char> lut;
public:

	string signal_name; 	///< name of signal to clean
	string ranges_name; ///< name of signal that defines ranges
	string output_name; ///< name of output virtual signal - defaults to signal_name + "_" + ranges_name
	int signal_id;	///< id of signal to clean
	int ranges_id; ///< id of signal the defines ranges
	int output_id; ///< id of output signal
	int time_channel; ///< time channel to consider in cleaning in the original signal
	int range_time_channel; ///< range time channel to consider in cleaning. this time channel will be used to check we're not using the future in the range signal
	int output_type; ///< output signal type - should be identical to input signal type default to range + val type. Or string for generic type
	int get_values_in_range = 1; ///< if 1 (default) : keep only records time in range, if 0 : stay with the values out of range
	range_op_type range_operator = range_op_type::all; ///< options are all(default), first, last - which range to use in the reference signal
	int range_val_channel = -1; ///< the val channel in range signal to filter range signal. If < 0 or empty sets will not filter
	vector<string> sets; ///< sets use to filter ranges_name signal on range_val_channel. If empty or range_val_channel, no filters
	bool regex_on_sets = 0;  ///< Whether to use aggregation of .*sets[i].* regex to create new sets vector (override original sets)
	int last_n = 0;  ///< instead of looking on sets, take last value from range. 0 means current state, 1 one before, etc.j
	bool do_on_last_n = false; ///< if true, will use only last nth (using time channel 0) range signal backward from sample time and use this value to filter range. uses last_n value

	/// <summary> default constructor </summary>
	RepBasicRangeCleaner() :
		signal_name(""), ranges_name(""), output_name(""), signal_id(-1), ranges_id(-1),
		output_id(-1), time_channel(0), output_type(3) {
		processor_type = REP_PROCESS_BASIC_RANGE_CLEANER;
		range_time_channel = 1;
	}

	/// initialize signal ids
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);

	void register_virtual_section_name_id(MedDictionarySections& dict);

	bool get_last_n_value(int time, const UniversalSigVec& range_sig, float& last_value);

	/// <summary> Fill required- and affected-signals sets </summary>
	/// The parsed fields from init command.
	/// @snippet RepProcess.cpp RepBasicRangeCleaner::init
	virtual int init(map<string, string>& mapper);

	/// <summary> Apply cleaning model </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	void fit_for_repository(MedPidRepository& rep);

	/// Serialization
	ADD_CLASS_NAME(RepBasicRangeCleaner)
		ADD_SERIALIZATION_FUNCS(processor_type, signal_name, ranges_name, output_name, time_channel,
			req_signals, aff_signals, signal_id, ranges_id, output_id, virtual_signals, virtual_signals_generic,
			output_type, get_values_in_range, range_operator, range_val_channel, sets, range_time_channel, last_n, do_on_last_n, regex_on_sets)

		/// <summary> Print processors information </summary>
		void print();
};

/**
* Normalize Signal Values by time - divide by time to calculate rate
*/
class RepSignalRate : public RepProcessor {
public:
	string output_name; ///< names of signals created by the completer
	string input_name; ///< names of input signals used by the completer
	int work_channel; ///< which channel to change and divide by time
	float factor; ///< additional constant factor 
	string output_signal_type; ///< the output signal type

	RepSignalRate() {
		processor_type = REP_PROCESS_SIGNAL_RATE; output_name = { "calc_drug_rate" };
		work_channel = 0;  factor = 1;
		output_signal_type = "T(i,i),V(f)";
	}

	/// @snippet RepProcess.cpp RepSignalRate::init
	int init(map<string, string>& mapper);
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);
	void register_virtual_section_name_id(MedDictionarySections& dict);
	void set_required_signal_ids(MedDictionarySections& dict) {};
	void set_affected_signal_ids(MedDictionarySections& dict) {};

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be implemented for all inheriting classes </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	void print();
	ADD_CLASS_NAME(RepSignalRate)
		ADD_SERIALIZATION_FUNCS(processor_type, input_name, output_name,
			work_channel, factor, unconditional, req_signals, aff_signals,
			virtual_signals, virtual_signals_generic, output_signal_type)
private:
	int v_out_sid = -1;
	int in_sid = -1;
	int v_out_n_time_ch, v_out_n_val_ch;
};

/**
* An Aggregator for signal in sliding time window
*/
class RepAggregateSignal : public RepProcessor {
public:
	string signalName; ///< the name of inout signal
	string output_name; ///< the name of virtual signal
	int work_channel; ///< channel for value
	int start_time_channel;; ///< time channel for start time
	int end_time_channel; ///< time channel for end time
	float factor; ///< final factor to multiply
	int time_window; ///< back time window to search for
	int time_unit; ///< time unit
	float drop_missing_rate; ///< If missing time points beyond this threshold will drop measurement
	bool buffer_first; ///< If true will wait for first buffer to complete
	string output_signal_type; ///< same as input signal - will remove later after change init process of RepProcessor to fetch input signal type


	RepAggregateSignal() {
		processor_type = REP_PROCESS_AGGREGATE;
		work_channel = 0;
		start_time_channel = 0;
		end_time_channel = 0;
		factor = 1;
		time_unit = global_default_windows_time_unit;
		drop_missing_rate = 1;

		output_name = "calc_aggregate";
		time_window = 0;
		buffer_first = true;
		output_signal_type = "T(i),V(f)";
	}

	/// @snippet RepProcess.cpp RepAggregateSignal::init
	int init(map<string, string>& mapper);
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);
	void register_virtual_section_name_id(MedDictionarySections& dict);
	void set_required_signal_ids(MedDictionarySections& dict) {};
	void set_affected_signal_ids(MedDictionarySections& dict) {};

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be 
	/// implemented for all inheriting classes 
	/// </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	void print();
	ADD_CLASS_NAME(RepAggregateSignal)
		ADD_SERIALIZATION_FUNCS(processor_type, signalName, output_name, work_channel, factor, time_window, time_unit,
			start_time_channel, end_time_channel, drop_missing_rate, buffer_first, unconditional, req_signals, aff_signals, virtual_signals, virtual_signals_generic, output_signal_type)
private:
	int v_out_sid = -1;
	int in_sid = -1;
	int v_out_n_time_ch, v_out_n_val_ch;
};

//---------------------------------------------------------------------------------------------------------------
// helper structures for RepCreateBitSignal:
//---------------------------------------------------------------------------------------------------------------
struct category_time_interval {
	int first_appearance = 0;
	int last_appearance = 0;
	int n_appearances = 0;
	int last_time = 0; // last_appearance + duration

	category_time_interval() {};
	category_time_interval(int _first, int _last, int _n, int _last_time) { first_appearance = _first; last_appearance = _last; n_appearances = _n; last_time = _last_time; }

	void set(int _first, int _last, int _n, int _last_time) { first_appearance = _first; last_appearance = _last; n_appearances = _n; last_time = _last_time; }
};

struct category_event_state {
	int time = 0;
	int appear_time = 0;
	int categ = 0;
	int type = 0; // 0 : stop, 1: start, 2: last appearance

	category_event_state() {};
	category_event_state(int _time, int _appear_time, int _categ, int _type) { time = _time; appear_time = _appear_time; categ = _categ; type = _type; }
	void set(int _time, int _appear_time, int _categ, int _type) { time = _time; appear_time = _appear_time; categ = _categ; type = _type; }


	bool operator <(const category_event_state &c) const {
		if (time < c.time) return true;
		if (time > c.time) return false;
		if (type < c.type) return true;
		//if (type > c.type) return false;
		//if (end_time < c.end_time) return true;
		return false;
	}
};

struct combination_state {
	int start;
	int state;
	vector<int> last;

	combination_state() {};
	combination_state(int time, int _state, int N) { start = time; state = _state; last.assign(N, -1); }
};


//---------------------------------------------------------------------------------------------------------------
// RepCreateBitSignal
// Given N<32 categories defined as groups of sets on a categorial signal, creates a united signal
// holding in bit j the occurance of each category. Each occurance of a category at time t implies 
// a time span of [t, t+max(min_duration, duration in signal (if available))] in which we define it as 
// occuring.
// The resulted signal includes each time point in which any of the categories changed, and the new "state" 
// at each such time point. "0" state starts at the first sig time point (unless it starts with the category).
// This is useful for example when working on drugs for a specific conditions (such as diabetes drugs)
// It then creates all states of treatment for the patient in a single easy to use signal.
// A virtual dictionary will be added as well.
// 
//---------------------------------------------------------------------------------------------------------------
class RepCreateBitSignal : public RepProcessor {
public:
	string in_sig = ""; ///< the name of input categorial signal
	string out_virtual = ""; ///< the name of the output virtual signal
	int t_chan = 0; ///< time channel for categorial data
	int c_chan = 0; ///< channel for categorial data
	int duration_chan = 1; ///< if >=0 the channel of duration for the categorial signal
	int min_duration = 60; ///< minimal duration of a category
	vector<int> min_durations = {}; ///<option to state the min_duration for each category specifically
	int max_duration = 180; ///< maximal duration of a category
	float duration_add = 0;
	float duration_mult = 0.1f; ///< will add to each duration (duration_add + duration_mult*duration)
	int dont_look_back = 7; ///< how many days back to not look at in in_sig. This is usefull in many cases
	int min_clip_time = 7; ///< minimal number of days to consider clipping after due to change to other drugs.
	int last_clip_period = 30; ///< after joining periods, we clip the last one to be exactly last_clip_period after the last appearance of the drug
	vector<int> min_jitters = { 0,0,0 };	///< minimal length for allowed jitters : A-AB-B,ABC-AB-A,AB-A-AC
	int max_min_jitters;
	vector<string> categories_names; ///< the names of the categories to create, categories_names[j] will sit at bit j (1 << j)
	vector<vector<string>> categories_sets; ///< the sets defining each category.
	int time_unit_sig = MedTime::Date; ///< time unit of the time channel in in_sig
	int time_unit_duration = MedTime::Days; ///< time unit of the duration (both in signal and in min_duration)
	bool change_at_prescription_mode = false; ///<  change_at_prescription_mode = true - change signal at every prescription (only the relevant drugs). When false change only when prescription changes
	MedDictionarySections *_dict; // for debug
	// next will NOT be serialized, and is here for debug reasons
	string print_dict = "";
	int time_channels = 1;
	vector<char> all_cat_lut;
	vector<vector<char>> categories_luts;

	RepCreateBitSignal() { processor_type = REP_PROCESS_CREATE_BIT_SIGNAL; };

	/// @snippet RepProcess.cpp RepAggregateSignal::init
	int init(map<string, string>& mapper);
	void get_min_jitters(string& jitters_s);
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);
	void register_virtual_section_name_id(MedDictionarySections& dict);
	void set_required_signal_ids(MedDictionarySections& dict) {};
	void set_affected_signal_ids(MedDictionarySections& dict) {};

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be 
	/// implemented for all inheriting classes 
	/// </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	void get_required_signal_categories(unordered_map<string, vector<string>> &signal_categories_in_use) const;

	//void print();
	ADD_CLASS_NAME(RepCreateBitSignal)
		ADD_SERIALIZATION_FUNCS(processor_type, in_sig, out_virtual, aff_signals, req_signals,
			t_chan, c_chan, duration_chan, min_duration, min_durations, max_duration, duration_add, duration_mult, dont_look_back, min_clip_time, last_clip_period, categories_names, categories_sets, time_unit_sig, time_unit_duration, change_at_prescription_mode,
			virtual_signals_generic, time_channels, min_jitters)

private:
	int v_out_sid = -1;
	int in_sid = -1;
	vector<string> registry_values;
};

//---------------------------------------------------------------------------------------------------------------


//.......................................................................................
/** RepCheckReq does not actually process the repository but rather check each
	sample for compliance with some requirment, and adjust the sample's corresponding
	attribute.
	Currently, possible minimal requirements are -
	1.	Given a set of signals and a time window, return how many signals
		do not appear within the window prior to the sample time
*/
//.......................................................................................

class RepCheckReq : public RepProcessor {

public:
	vector<string> signalNames; ///< Required signals
	vector<int> signalIds; ///< Required signals ids

	vector<int> time_channels; ///< signals' time channels
	int win_from = 0, win_to = 0; ///< window to consider ;
	int window_time_unit = MedTime::Days; ///< window time-unit

	string attrName = "MissingReq"; ///< attribute name (set to 1 if compliant)

	// Helper
	vector<int> sig_time_units;

	/// <summary> constructor </summary>
	RepCheckReq() { processor_type = REP_PROCESS_CHECK_REQ; unconditional = true; }

	/// @snippet RepProcess.cpp RepCheckReq::init
	int init(map<string, string>& mapper);

	/// <summary> Set signal id </summary>
	void set_signal_ids(MedSignals& sigs);

	/// <summary> Fill req- and aff-signals vectors </summary>
	void init_lists();

	/// <summary> initialize time-nits </summary>
	void init_tables(MedDictionarySections& dict, MedSignals& sigs);

	/// <summary> Init attributes information : Should be implemented for inheriting classes that have attributes </summary>
	void init_attributes() { attributes = { attrName }; }

	// Learning - none done

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be implemented for all inheriting classes </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	// serialization.
	ADD_CLASS_NAME(RepCheckReq)
		ADD_SERIALIZATION_FUNCS(processor_type, signalNames, time_channels, win_from, win_to, window_time_unit, attrName, req_signals)
};

//----------------------------------------------------------------------------------------
// RepHistoryLimit : given a signal : chomps history to be at a given window relative
//                   to prediction points
//----------------------------------------------------------------------------------------
class RepHistoryLimit : public RepProcessor {
public:
	string signalName; 	///< name of signal to clean
	int signalId;	///< id of signal to clean
	int time_channel = 0; ///< time channel to consider in cleaning
	int truncate_time_channel = -1; ///< If given this time channel value will be truncated to "pred_time" - win_to
	int win_time_unit = global_default_windows_time_unit;
	int rep_time_unit = global_default_time_unit; // we assume this is also the samples time unit
	int win_from = 0;
	int win_to = 0;
	int delete_sig = 0; /// simply delete the signal (set it's len to 0).
	int take_last_events = -1; /// It given >0 and delete_sig==0 and win_from==0,win_to==0  will use only last events number in the signal to keep

	RepHistoryLimit() { init_defaults(); }

	// default init
	void init_defaults() { processor_type = REP_PROCESS_HISTORY_LIMIT; }

	// preparations
	void set_signal_ids(MedSignals& sigs) { signalId = sigs.sid(signalName); }

	// learn - nothing to do
	int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processor) { return 0; }

	// apply
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	/// @snippet RepProcess.cpp RepHistoryLimit::init
	int init(map<string, string>& mapper);

	// helper
	int get_sub_usv_data(UniversalSigVec &usv, int from_time, int to_time, vector<char> &data, int &len);

	/// Fill req- and aff-signals vectors
	void init_lists();

	ADD_CLASS_NAME(RepHistoryLimit)
		ADD_SERIALIZATION_FUNCS(processor_type, signalName, time_channel, win_time_unit, rep_time_unit,
			win_from, win_to, delete_sig, take_last_events, req_signals, aff_signals, truncate_time_channel)

};

//----------------------------------------------------------------------------------------
// RepNumericNoiser : given a numeric signal : adds gaussian noise to each value, with std as user-determined
// fraction of signal std, uniform distribution to time, and probability of dropping value
//----------------------------------------------------------------------------------------
class RepNumericNoiser : public RepProcessor {
private:
	int signalId =-1;	///< id of signal to clean
	double stdev;
	bool on_learning = false;
	bool apply_in_test;

	random_device rd;
	//mt19937 gens;
	vector<mt19937> gens;

	//mt19937 generator;

public:
	string signalName; 	///< name of signal to clean
	
	int val_channel = 0; ///< value channel to consider in cleaning
	int time_channel = 0; ///< time channel to consider in cleaning
	int time_noise = 0; ///< uniform dist of x days will be added to time-signal
	float value_noise = 0; ///< x times signal std will be added to each value
	int truncation = 2; 
	float drop_probability = (float)0.0; ///<probability of dropping lab


	RepNumericNoiser() { init_defaults(); }


	// default init
	void init_defaults() { 
		processor_type = REP_PROCESS_NUMERIC_NOISER; 
		int N_TH = omp_get_max_threads();
		gens.resize(3*N_TH);
		for (size_t i = 0; i < 3*N_TH; ++i)
			gens[i] = mt19937(rd());
	}

	// preparations
	void set_signal_ids(MedSignals& sigs) { signalId = sigs.sid(signalName); }

	// learn - nothing to do
	int _learn(MedPidRepository& rep, MedSamples& samples, vector<RepProcessor *>& prev_processor);

	// apply
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	/// @snippet RepProcess.cpp RepHistoryLimit::init
	int init(map<string, string>& mapper);

	/// Fill req- and aff-signals vectors
	void init_lists();

	ADD_CLASS_NAME(RepNumericNoiser)
		ADD_SERIALIZATION_FUNCS(processor_type, signalName, val_channel, time_channel,
			time_noise, value_noise, drop_probability, req_signals, aff_signals, stdev, truncation, apply_in_test)

};

class RepReoderChannels : public RepProcessor {
private:
	int sid = -1;
public:
	vector<int> new_order;
	string signal_name = "";

	RepReoderChannels() {
		processor_type = REP_PROCESS_REODER_CHANNELS;
	}

	void set_signal_ids(MedSignals& sigs);
	void set_signal(const string& _signalName) { sid = -1; signal_name = _signalName; }

	/// @snippet RepReoderChannels.cpp RepReoderChannels::init
	int init(map<string, string>& mapper);

	// Applying
	/// <summary> apply processing on a single PidDynamicRec at a set of time-points : Should be implemented for all inheriting classes </summary>
	int _apply(PidDynamicRec& rec, vector<int>& time_points, vector<vector<float>>& attributes_mat);

	void print();

	ADD_CLASS_NAME(RepReoderChannels)
		ADD_SERIALIZATION_FUNCS(processor_type, req_signals, aff_signals, virtual_signals, virtual_signals_generic, signal_name, new_order)
};

//.......................................................................................
//.......................................................................................
// Utility Functions
//.......................................................................................
//.......................................................................................

/// <summary> Get values of a signal from a set of samples applying a set of preceeding cleaners </summary>
int get_values(MedRepository& rep, MedSamples& samples, int signalId, int time_channel, int val_channel, float range_min, float range_max, vector<float>& values,
	vector<RepProcessor *>& prev_cleaners);
/// <summary> Get values of a signal from a set of samples </summary>
int get_values(MedRepository& rep, MedSamples& samples, int signalId, int time_channel, int val_channel, float range_min, float range_max, vector<float>& values);

//=======================================
// Joining the MedSerialze wagon
//=======================================
MEDSERIALIZE_SUPPORT(RepProcessor)
MEDSERIALIZE_SUPPORT(RepMultiProcessor)
MEDSERIALIZE_SUPPORT(RepBasicOutlierCleaner)
MEDSERIALIZE_SUPPORT(RepRuleBasedOutlierCleaner)
MEDSERIALIZE_SUPPORT(RepConfiguredOutlierCleaner)
MEDSERIALIZE_SUPPORT(RepNbrsOutlierCleaner)
MEDSERIALIZE_SUPPORT(RepCalcSimpleSignals)
MEDSERIALIZE_SUPPORT(RepSimValHandler)
MEDSERIALIZE_SUPPORT(RepPanelCompleter)
MEDSERIALIZE_SUPPORT(RepCombineSignals)
MEDSERIALIZE_SUPPORT(RepSplitSignal)
MEDSERIALIZE_SUPPORT(RepAggregationPeriod)
MEDSERIALIZE_SUPPORT(RepBasicRangeCleaner)
MEDSERIALIZE_SUPPORT(RepSignalRate)
MEDSERIALIZE_SUPPORT(RepAggregateSignal)
MEDSERIALIZE_SUPPORT(RepCheckReq)
MEDSERIALIZE_SUPPORT(RepHistoryLimit)
MEDSERIALIZE_SUPPORT(RepCreateBitSignal)
MEDSERIALIZE_SUPPORT(RepReoderChannels)
MEDSERIALIZE_SUPPORT(RepNumericNoiser)
MEDSERIALIZE_SUPPORT(SimpleCalculator)
MEDSERIALIZE_SUPPORT(SetCalculator)
MEDSERIALIZE_SUPPORT(MultiplyCalculator)
MEDSERIALIZE_SUPPORT(logCalculator)
MEDSERIALIZE_SUPPORT(RatioCalculator)
MEDSERIALIZE_SUPPORT(eGFRCalculator)
MEDSERIALIZE_SUPPORT(SumCalculator)
MEDSERIALIZE_SUPPORT(RangeCalculator)
MEDSERIALIZE_SUPPORT(ExistsCalculator)
MEDSERIALIZE_SUPPORT(EmptyCalculator)
MEDSERIALIZE_SUPPORT(ConstantValueCalculator)

#endif
