# MemoryStack - Python SDK

Official Python SDK for MemoryStack - A semantic memory layer for AI applications.

## Installation

```bash
pip install memorystack
```

## Quick Start






pypi-AgEIcHlwaS5vcmcCJDZkZmUxMGFmLTEyYmMtNDAyMi1iNGNiLTU1MDRkY2RhYzc1ZAACKlszLCIwNDc0NjQyOS02MjQ0LTQxNDItODYyNy0yN2U3Zjg0Y2M1MzciXQAABiAZwEkHDjeCykKD9gEK1l91J9OOKHBv82rgfMOoPkZpjA



Set your password to the token value, including the pypi- prefix
For example, if you are using Twine to upload your projects to PyPI, set up your $HOME/.pypirc file like this:

[pypi]
  username = __token__
  password = pypi-AgEIcHlwaS5vcmcCJDZkZmUxMGFmLTEyYmMtNDAyMi1iNGNiLTU1MDRkY2RhYzc1ZAACKlszLCIwNDc0NjQyOS02MjQ0LTQxNDItODYyNy0yN2U3Zjg0Y2M1MzciXQAABiAZwEkHDjeCykKD9gEK1l91J9OOKHBv82rgfMOoPkZpjA
```python
from memorystack import MemoryStackClient

# Initialize the client
client = MemoryStackClient(
    api_key="your-api-key",
    base_url="https://memorystack.app/api/v1"
)

# Add a conversation to memory
result = client.add_conversation(
    "I love Python programming",
    "That's great! Python is excellent for AI and data science.",
    user_id="user_123"  # optional: for B2B use cases
)

print(f"Created {result.memories_created} memories")
```

## Features

- ✅ **Simple API**: Easy-to-use methods for memory management
- ✅ **Multi-Agent Support**: Full agent lifecycle, teams, and handoffs
- ✅ **Type Hints**: Full type annotations for better IDE support
- ✅ **Multimodal**: Support for text, images, documents, and audio
- ✅ **Agent Patterns**: Sequential workflows, routing, autonomous agents
- ✅ **B2B & B2C**: Works for both personal and multi-tenant applications
- ✅ **Pagination**: Built-in cursor-based pagination
- ✅ **Error Handling**: Comprehensive exception handling

## Usage

### Initialize Client

```python
from memorystack import MemoryStackClient
import os

client = MemoryStackClient(
    api_key=os.environ["MEMORYSTACK_API_KEY"],
    base_url="https://memorystack.app/api/v1"
)
```

### Create Memories

#### Simple Conversation

```python
# Add a simple conversation
result = client.add_conversation(
    "What is the capital of France?",
    "The capital of France is Paris."
)
```

#### With User ID (B2B)

```python
# For B2B applications with end users
result = client.add_conversation(
    "I prefer dark mode",
    "I'll remember that preference.",
    user_id="alice_123"  # end user ID
)
```

#### Single Message

```python
# Add a single message
client.add_message(
    "My favorite color is blue",
    user_id="user_456"
)
```

### List Memories

#### Personal Memories

```python
# Get your personal memories
memories = client.get_personal_memories(limit=20)

print(f"Found {memories.count} memories")
for memory in memories.results:
    print(f"- {memory.content} ({memory.memory_type})")
```

#### User Memories (B2B)

```python
# Get memories for a specific end user
user_memories = client.get_user_memories("alice_123", limit=50)
```

#### With Pagination

```python
cursor = None
all_results = []

while True:
    response = client.list_memories(
        user_id="alice_123",
        limit=50,
        cursor=cursor
    )
    
    all_results.extend(response.results)
    cursor = response.next_cursor
    
    if not cursor:
        break

print(f"Total memories: {len(all_results)}")
```

### Search Memories

```python
# Semantic search
results = client.search_memories(
    query="user preferences",
    limit=10,
    mode="hybrid"  # hybrid, vector, or text
)

for memory in results["results"]:
    print(f"- {memory['content']} (score: {memory.get('similarity', 'N/A')})")
```

### Get Usage Statistics

```python
stats = client.get_stats()

print(f"Plan: {stats.plan_tier}")
print(f"API Calls: {stats.usage['current_month_api_calls']} / {stats.usage['monthly_api_limit']}")
print(f"Total Memories: {stats.totals['total_memories']}")
```

## Error Handling

```python
from memorystack import (
    MemoryStackError,
    AuthenticationError,
    RateLimitError,
    ValidationError
)

try:
    client.add_message("Hello world")
except AuthenticationError as e:
    print(f"Auth failed: {e.message}")
except RateLimitError as e:
    print(f"Rate limit: {e.message}")
except ValidationError as e:
    print(f"Invalid request: {e.message}")
except MemoryStackError as e:
    print(f"Error: {e.message}")
    print(f"Status: {e.status_code}")
```

Common error codes:
- `401` - Invalid or missing API key
- `429` - Rate limit exceeded
- `400` - Invalid request payload
- `500` - Internal server error

## Best Practices

### 1. Use Environment Variables

```python
import os
from memorystack import MemoryStackClient

client = MemoryStackClient(
    api_key=os.environ["MEMORYSTACK_API_KEY"],
    base_url="https://memorystack.app/api/v1"
)
```

### 2. Handle Errors Gracefully

```python
from memorystack import MemoryStackError

try:
    result = client.add_message("Important fact")
    print(f"Success: {result.memories_created}")
except MemoryStackError as e:
    print(f"Failed to save memory: {e.message}")
    # Implement retry logic or fallback
```

### 3. Use User IDs for B2B

```python
# Always pass user_id for multi-tenant applications
client.add_conversation(
    user_message,
    assistant_response,
    user_id=request.user.id  # your application's user ID
)
```

### 4. Add Metadata for Context

```python
from datetime import datetime

client.create_memory(
    messages=[...],
    user_id="alice_123",
    metadata={
        "session_id": "sess_xyz",
        "source": "mobile_app",
        "version": "1.2.3",
        "timestamp": datetime.utcnow().isoformat()
    }
)
```

## Examples

### Chatbot Integration

```python
from memorystack import MemoryStackClient
import os

memory_client = MemoryStackClient(
    api_key=os.environ["MEMORYSTACK_API_KEY"],
    base_url="https://memorystack.app/api/v1"
)

def handle_chat_message(user_id: str, message: str):
    # Your AI logic here
    response = generate_ai_response(message)
    
    # Save to memory
    memory_client.add_conversation(
        message,
        response,
        user_id=user_id
    )
    
    return response
```

### Personal Assistant

```python
# Store user preferences
client.add_message(
    "I prefer meetings in the morning",
    metadata={"category": "preference", "type": "scheduling"}
)

# Retrieve preferences later
preferences = client.list_memories(
    user_id="self",
    memory_type="preference",
    limit=100
)
```

## Support

- Documentation: https://memorystack.app/docs
- Issues: https://github.com/memorystack-labs/memorystack-python/issues
- Email: support@memorystack.app

## License

MIT

