#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Vulnerability scanning utilities for Metaport Python Agent.

This module provides functionality to scan for vulnerabilities in Python
dependencies using poetry audit with the ko-poetry-audit-plugin.

Supports Python 3.10+ with appropriate type hints and error handling.
"""

import json
import subprocess
import sys
import os


class VulnerabilityScanner(object):
    """
    Vulnerability scanner for Python dependencies.

    Provides methods to scan for vulnerabilities using poetry audit with
    the ko-poetry-audit-plugin. Returns structured vulnerability information
    for SBOM generation.

    Features:
        - Poetry audit vulnerability scanning
        - Structured vulnerability data extraction
        - Error handling for failed scans
        - Timeout handling for slow scans
    """

    def __init__(self, timeout=60):
        # type: (int) -> None
        """
        Initialize vulnerability scanner.

        Args:
            timeout: Command execution timeout in seconds (default: 60)
        """
        self._timeout = timeout
        self._available_scanners = []  # type: List[str]
        self._scan_results = {}  # type: Dict[str, Any]

    def scan_vulnerabilities(self, project_path="."):
        # type: (str) -> List[Dict[str, str]]
        """
        Scan for vulnerabilities using available tools.

        Uses poetry audit for Poetry projects and pip-audit for other projects
        to scan for vulnerabilities in Python dependencies.

        Args:
            project_path: Path to project directory to scan (default: current directory)

        Returns:
            List of vulnerability dictionaries with keys:
            - PackageName: Name of vulnerable package
            - Identifier: Vulnerability identifier (CVE, GHSA, etc.)
            - Source: Source of vulnerability data
            - Severity: Vulnerability severity level
            - Cvss: CVSS score if available
            - Reference: Reference URL for vulnerability
            - Status: None
        """
        vulnerabilities = []

        # Check available scanners
        self._detect_available_scanners()

        if not self._available_scanners:
            print("Warning: No vulnerability scanners available", file=sys.stderr)
            return vulnerabilities

        # Try poetry audit first (for Poetry projects)
        if "poetry" in self._available_scanners:
            try:
                vulns = self._scan_with_poetry_audit(project_path)
                if vulns:
                    vulnerabilities.extend(vulns)
                    return vulnerabilities  # If poetry audit works, use it
            except Exception as e:
                print(
                    "Info: poetry audit not applicable: {}".format(str(e)),
                    file=sys.stderr,
                )

        # Fall back to pip-audit (for requirements.txt, Pipfile, etc.)
        if "pip-audit" in self._available_scanners:
            try:
                vulns = self._scan_with_pip_audit(project_path)
                if vulns:
                    vulnerabilities.extend(vulns)
            except Exception as e:
                print(
                    "Warning: pip-audit scan failed: {}".format(str(e)), file=sys.stderr
                )

        return vulnerabilities

    def _detect_available_scanners(self):
        # type: () -> None
        """
        Detect which vulnerability scanners are available on the system.
        """
        self._available_scanners = []

        # Check for poetry (and poetry audit command)
        if self._command_exists("poetry"):
            self._available_scanners.append("poetry")

        # Check for pip-audit
        if self._command_exists("pip-audit"):
            self._available_scanners.append("pip-audit")

    def _command_exists(self, command):
        # type: (str) -> bool
        """
        Check if a command exists and is executable.

        Args:
            command: Command name to check

        Returns:
            True if command exists, False otherwise
        """
        try:
            # Use 'which' on Unix-like systems, 'where' on Windows
            check_cmd = "where" if os.name == "nt" else "which"
            result = subprocess.run(
                [check_cmd, command],
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL,
                timeout=5,
            )
            return result.returncode == 0
        except (subprocess.TimeoutExpired, FileNotFoundError, OSError):
            return False

    def _scan_with_poetry_audit(self, project_path):
        # type: (str) -> List[Dict[str, str]]
        """
        Scan vulnerabilities using poetry audit.

        Args:
            project_path: Path to project directory

        Returns:
            List of vulnerability dictionaries
        """
        try:
            # Check if pyproject.toml exists
            pyproject_path = os.path.join(project_path, "pyproject.toml")
            if not os.path.exists(pyproject_path):
                return []

            # Run poetry audit with JSON output
            result = subprocess.run(
                ["poetry", "audit"],
                capture_output=True,
                text=True,
                timeout=self._timeout,
                cwd=project_path,
            )

            # poetry audit returns non-zero when vulnerabilities found
            if result.returncode not in [0, 1]:
                print(
                    "Warning: poetry audit failed with exit code {}".format(
                        result.returncode
                    ),
                    file=sys.stderr,
                )
                return []

            if not result.stdout.strip():
                return []

            # Parse JSON output
            scan_data = json.loads(result.stdout)
            vulnerabilities = []

            # Extract vulnerabilities from poetry audit output
            if isinstance(scan_data, list):
                for vuln_item in scan_data:
                    advisory = vuln_item.get("advisory", {})
                    severity = advisory.get("severity", "unknown")

                    # Look for CVSS score in advisory data
                    cvss_score = advisory.get("cvss", None)
                    if cvss_score == "unknown":
                        cvss_score = advisory.get("cvss_score", None)

                    vuln_dict = {
                        "PackageName": vuln_item.get("package", "unknown"),
                        "Identifier": advisory.get("id", "unknown"),
                        "Source": "poetry-audit",
                        "Severity": severity,
                        "Cvss": self._calculate_cvss(severity, cvss_score),
                        "Reference": advisory.get("url", None),
                        "Status": None,
                    }
                    vulnerabilities.append(vuln_dict)

            return vulnerabilities

        except subprocess.TimeoutExpired:
            print("Warning: poetry audit timed out", file=sys.stderr)
            return []
        except json.JSONDecodeError as e:
            print(
                "Warning: Failed to parse poetry audit output: {}".format(str(e)),
                file=sys.stderr,
            )
            return []
        except Exception as e:
            print(
                "Warning: poetry audit execution failed: {}".format(str(e)),
                file=sys.stderr,
            )
            return []

    def _scan_with_pip_audit(self, project_path):
        # type: (str) -> List[Dict[str, str]]
        """
        Scan vulnerabilities using pip-audit.

        Args:
            project_path: Path to project directory

        Returns:
            List of vulnerability dictionaries
        """
        try:
            # Run pip-audit with JSON output on the current environment
            result = subprocess.run(
                ["pip-audit", "--format=json", "--desc"],
                capture_output=True,
                text=True,
                timeout=self._timeout,
                cwd=project_path,
            )

            # pip-audit returns non-zero when vulnerabilities found
            if result.returncode not in [0, 1]:
                print(
                    "Warning: pip-audit failed with exit code {}".format(
                        result.returncode
                    ),
                    file=sys.stderr,
                )
                return []

            if not result.stdout.strip():
                return []

            # Parse JSON output
            scan_data = json.loads(result.stdout)
            vulnerabilities = []

            # Extract vulnerabilities from pip-audit output
            # pip-audit does not yet support a CVSS or Severity determination
            if "dependencies" in scan_data:
                for dep_item in scan_data["dependencies"]:
                    package_name = dep_item.get("name", "unknown")
                    if "vulns" in dep_item and dep_item["vulns"]:
                        for vuln_item in dep_item["vulns"]:
                            vuln_dict = {
                                "PackageName": package_name,
                                "Identifier": vuln_item.get("id", "unknown"),
                                "Source": "pip-audit",
                                "Severity": None,
                                "Cvss": None,
                                "Reference": self._extract_reference_from_pip_audit(
                                    vuln_item
                                )
                                or None,
                                "Status": "open",
                            }
                            vulnerabilities.append(vuln_dict)

            return vulnerabilities

        except subprocess.TimeoutExpired:
            print("Warning: pip-audit timed out", file=sys.stderr)
            return []
        except json.JSONDecodeError as e:
            print(
                "Warning: Failed to parse pip-audit output: {}".format(str(e)),
                file=sys.stderr,
            )
            return []
        except Exception as e:
            print(
                "Warning: pip-audit execution failed: {}".format(str(e)),
                file=sys.stderr,
            )
            return []

    def _extract_reference_from_pip_audit(self, vuln):
        # type: (Dict[str, Any]) -> str
        """
        Extract reference URL from pip-audit vulnerability data.

        Args:
            vuln: Vulnerability dict from pip-audit

        Returns:
            Reference URL string or 'unknown'
        """
        try:
            # Check for fix information
            if "fix" in vuln and isinstance(vuln["fix"], dict):
                if "versions" in vuln["fix"] and vuln["fix"]["versions"]:
                    return "Fix available: upgrade to {}".format(
                        ", ".join(vuln["fix"]["versions"])
                    )

            # Fallback to vulnerability ID
            vuln_id = vuln.get("id", "")
            if vuln_id.startswith("GHSA-"):
                return "https://github.com/advisories/{}".format(vuln_id)
            elif vuln_id.startswith("CVE-"):
                return "https://cve.mitre.org/cgi-bin/cvename.cgi?name={}".format(
                    vuln_id
                )
            else:
                return "unknown"
        except Exception:
            return "unknown"

    def _calculate_cvss(self, severity_text, cvss_score=None):
        # type: (str, Optional[str]) -> str
        """
        Calculate CVSS score range following Node.js agent pattern.

        If a numeric CVSS score is provided, use it directly.
        Otherwise, convert text severity to numeric ranges compatible with other agents.

        Args:
            severity_text: Text severity level (e.g., 'critical', 'high', 'medium', 'low')
            cvss_score: Optional numeric CVSS score as string

        Returns:
            CVSS score or range as string, or 'unknown' if cannot determine
        """
        # If we have a numeric CVSS score, use it directly
        if cvss_score and cvss_score != "unknown":
            try:
                # Validate it's a number
                float(cvss_score)
                return cvss_score
            except (ValueError, TypeError):
                pass

        # Convert text severity to CVSS ranges (following Node.js agent pattern)
        if not severity_text or not isinstance(severity_text, str):
            return "unknown"

        severity = severity_text.strip().upper()

        if severity == "NONE":
            return "0.0"
        elif severity == "LOW":
            return "0.1 - 3.9"
        elif severity == "MEDIUM":
            return "4.0 - 6.9"
        elif severity == "HIGH":
            return "7.0 - 8.9"
        elif severity == "CRITICAL":
            return "9.0 - 10.0"
        else:
            return "unknown"

    def _cvss_to_severity(self, cvss_score):
        # type: (float) -> str
        """
        Convert CVSS score to severity level.

        Args:
            cvss_score: CVSS score (0.0-10.0)

        Returns:
            Severity level string
        """
        if cvss_score >= 9.0:
            return "critical"
        elif cvss_score >= 7.0:
            return "high"
        elif cvss_score >= 4.0:
            return "medium"
        elif cvss_score > 0.0:
            return "low"
        else:
            return "unknown"

    def get_available_scanners(self):
        # type: () -> List[str]
        """
        Get list of available vulnerability scanners.

        Returns:
            List of available scanner names
        """
        if not self._available_scanners:
            self._detect_available_scanners()
        return self._available_scanners.copy()
