import os
import re
import shlex
import json
import tempfile
import getpass
from modaic import AutoProgram

# --- Modaic ---

MODAIC_REPO_PATH = "farouk1/nanocode"

# --- ANSI colors ---

RESET = "\033[0m"
BOLD = "\033[1m"
DIM = "\033[2m"
BLUE = "\033[34m"
CYAN = "\033[36m"
GREEN = "\033[32m"
YELLOW = "\033[33m"
RED = "\033[31m"
MAGENTA = "\033[35m"

# --- Display utilities ---

LONG_PASTE_THRESHOLD = int(os.environ.get("NANOCODE_LONG_PASTE_THRESHOLD", "4000"))


def save_long_paste(text: str) -> str:
    fd, path = tempfile.mkstemp(prefix="nanocode_paste_", suffix=".txt")
    with os.fdopen(fd, "w") as handle:
        handle.write(text)
    return path


def separator():
    """Return a horizontal separator line that fits the terminal width."""
    return f"{DIM}{'─' * min(os.get_terminal_size().columns, 80)}{RESET}"


def render_markdown(text):
    """Convert basic markdown bold syntax to ANSI bold."""
    return re.sub(r"\*\*(.+?)\*\*", f"{BOLD}\\1{RESET}", text)


# --- Cache utilities ---

CACHE_DIR = os.path.join(
    os.getenv("XDG_CACHE_HOME", os.path.expanduser("~/.cache")),
    "nanocode",
)
OPENROUTER_KEY_PATH = os.path.join(CACHE_DIR, "openrouter_key.json")


def load_openrouter_key():
    if not os.path.exists(OPENROUTER_KEY_PATH):
        return None
    try:
        with open(OPENROUTER_KEY_PATH, "r", encoding="utf-8") as handle:
            data = json.load(handle)
        key = data.get("openrouter_api_key")
        return key if key else None
    except (OSError, json.JSONDecodeError):
        return None


def save_openrouter_key(key: str) -> None:
    os.makedirs(CACHE_DIR, exist_ok=True)
    tmp_path = None
    try:
        fd, tmp_path = tempfile.mkstemp(prefix="nanocode_key_", suffix=".json", dir=CACHE_DIR)
        with os.fdopen(fd, "w", encoding="utf-8") as handle:
            json.dump({"openrouter_api_key": key}, handle)
        os.replace(tmp_path, OPENROUTER_KEY_PATH)
        try:
            os.chmod(OPENROUTER_KEY_PATH, 0o600)
        except OSError:
            pass
    finally:
        if tmp_path and os.path.exists(tmp_path):
            try:
                os.remove(tmp_path)
            except OSError:
                pass


def clear_openrouter_key() -> None:
    try:
        os.remove(OPENROUTER_KEY_PATH)
    except OSError:
        pass


# --- Model selection ---

AVAILABLE_MODELS = {
    "1": ("GPT-5.2 Codex", "openai/gpt-5.2-codex"),
    "2": ("GPT-5.2", "openai/gpt-5.2"),
    "3": ("Claude Opus 4.5", "anthropic/claude-opus-4.5"),
    "4": ("Claude Opus 4", "anthropic/claude-opus-4"),
    "5": ("Qwen 3 Coder", "qwen/qwen3-coder"),
    "6": ("Gemini 3 Flash Preview", "google/gemini-3-flash-preview"),
    "7": ("Kimi K2 0905", "moonshotai/kimi-k2-0905"),
    "8": ("Minimax M2.1", "minimax/minimax-m2.1"),
}


def select_model():
    """Interactive model selection or use environment variable."""
    model_env = os.getenv("MODEL")
    if model_env:
        print(f"{GREEN}⏺ Using model from environment: {model_env}{RESET}")
        return model_env

    print(f"\n{BOLD}Select a model:{RESET}")
    for key, (name, model_id) in AVAILABLE_MODELS.items():
        print(f"  {BLUE}{key}{RESET}. {name} ({DIM}{model_id}{RESET})")
    print(f"  {BLUE}c{RESET}. Custom model (enter manually)")

    while True:
        try:
            choice = (
                input(f"\n{BOLD}{BLUE}❯{RESET} Enter choice (1-8 or c): ")
                .strip()
                .lower()
            )

            if choice in AVAILABLE_MODELS:
                name, model_id = AVAILABLE_MODELS[choice]
                print(f"{GREEN}⏺ Selected: {name}{RESET}")
                return model_id
            elif choice == "c":
                custom_model = input(
                    f"{BOLD}{BLUE}❯{RESET} Enter model ID (e.g., openai/gpt-4): "
                ).strip()
                if custom_model:
                    print(f"{GREEN}⏺ Selected custom model: {custom_model}{RESET}")
                    return custom_model
                else:
                    print(f"{RED}⏺ Invalid model ID{RESET}")
            else:
                print(f"{RED}⏺ Invalid choice. Please enter 1-8 or c{RESET}")
        except (KeyboardInterrupt, EOFError):
            print(f"\n{RED}⏺ Model selection cancelled{RESET}")
            exit(1)

def main():
    cached_key = load_openrouter_key()
    if cached_key and not os.getenv("OPENROUTER_API_KEY"):
        os.environ["OPENROUTER_API_KEY"] = cached_key
        print(f"{GREEN}⏺ Loaded OpenRouter key from cache{RESET}")

    model = select_model()

    # Add openrouter/ prefix if not already present
    if not model.startswith("openrouter/"):
        model = f"openrouter/{model}"

    agent = AutoProgram.from_precompiled(MODAIC_REPO_PATH, config={"lm": model})

    print(
        f"{BOLD}NANOCODE DSPY{RESET} | {DIM}{model}{RESET} | {os.getcwd()}{RESET}\n"
    )

    # Conversation history for context
    history = []

    # MCP servers registry
    mcp_servers = {}

    def register_mcp_server(name, server):
        tool_names = []
        for tool in server.tools:
            tool_name = f"{name}_{tool.__name__}"
            agent.set_tool(tool_name, tool)
            tool_names.append(tool_name)
        return tool_names

    while True:
        try:
            print(separator())
            user_input = input(f"{BOLD}{BLUE}❯{RESET} ").strip()
            print(separator())

            tmp_paste_path = None
            if len(user_input) > LONG_PASTE_THRESHOLD:
                tmp_paste_path = save_long_paste(user_input)
                print(
                    f"{YELLOW}⏺ Long paste detected ({len(user_input)} chars). Saved to {tmp_paste_path}{RESET}"
                )
                user_input = (
                    f"The user pasted a long input ({len(user_input)} chars). "
                    f"It has been saved to {tmp_paste_path}. "
                    "Use read_file to view it. The file will be deleted after this response."
                )

            if not user_input:
                continue
            if user_input in ("/q", "exit"):
                break
            if user_input.startswith("/openrouter-key"):
                parts = shlex.split(user_input)
                args = parts[1:]
                if args and args[0] in ("clear", "unset", "remove"):
                    clear_openrouter_key()
                    os.environ.pop("OPENROUTER_API_KEY", None)
                    print(f"{GREEN}⏺ OpenRouter key cleared{RESET}")
                    continue

                key = args[0] if args else getpass.getpass(
                    f"{BOLD}{BLUE}❯{RESET} Enter OpenRouter API key (input hidden): "
                ).strip()
                if not key:
                    print(f"{RED}⏺ OpenRouter key not set (empty input){RESET}")
                    continue
                save_openrouter_key(key)
                os.environ["OPENROUTER_API_KEY"] = key
                print(f"{GREEN}⏺ OpenRouter key saved to cache{RESET}")
                continue
            if user_input == "/c":
                history = []
                print(f"{GREEN}⏺ Cleared conversation{RESET}")
                continue
            if user_input == "/model":
                print(f"\n{BOLD}Current model: {agent.config.lm}{RESET}")
                print(f"\n{BOLD}Select a new model:{RESET}")
                for key, (name, model_id) in AVAILABLE_MODELS.items():
                    print(f"  {BLUE}{key}{RESET}. {name} ({DIM}{model_id}{RESET})")
                print(f"  {BLUE}c{RESET}. Custom model (enter manually)")
                print(f"  {BLUE}k{RESET}. Keep current model")

                choice = input(f"\n{BOLD}{BLUE}❯{RESET} Enter choice: ").strip().lower()

                if choice == "k":
                    print(f"{GREEN}⏺ Keeping current model: {agent.config.lm}{RESET}")
                    continue
                elif choice in AVAILABLE_MODELS:
                    name, model_id = AVAILABLE_MODELS[choice]
                    new_model = (
                        model_id
                        if model_id.startswith("openrouter/")
                        else f"openrouter/{model_id}"
                    )
                    agent = AutoProgram.from_precompiled(MODAIC_REPO_PATH, config={"lm": new_model})
                    for server_name, info in mcp_servers.items():
                        info["tools"] = register_mcp_server(server_name, info["server"])
                    print(f"{GREEN}⏺ Switched to: {name} ({new_model}){RESET}")
                elif choice == "c":
                    custom_model = input(
                        f"{BOLD}{BLUE}❯{RESET} Enter model ID: "
                    ).strip()
                    if custom_model:
                        new_model = (
                            custom_model
                            if custom_model.startswith("openrouter/")
                            else f"openrouter/{custom_model}"
                        )
                        agent = AutoProgram.from_precompiled(MODAIC_REPO_PATH, config={"lm": new_model})
                        for server_name, info in mcp_servers.items():
                            info["tools"] = register_mcp_server(
                                server_name, info["server"]
                            )
                        print(f"{GREEN}⏺ Switched to custom model: {new_model}{RESET}")
                    else:
                        print(f"{RED}⏺ Invalid model ID, keeping current model{RESET}")
                else:
                    print(f"{RED}⏺ Invalid choice, keeping current model{RESET}")
                continue

            if user_input.startswith("/add-mcp"):
                parts = shlex.split(user_input)
                args = parts[1:]
                if not args:
                    print(
                        f"{YELLOW}⏺ Usage: /add-mcp <name> <server> [--auth <auth>|--oauth] [--headers '<json>'] [--auto-auth|--no-auto-auth]{RESET}"
                    )
                    continue

                name = None
                auth = None
                headers = None
                auto_auth = None
                positional = []
                i = 0
                while i < len(args):
                    if args[i] in ("--name", "-n") and i + 1 < len(args):
                        name = args[i + 1]
                        i += 2
                    elif args[i].startswith("--auth="):
                        auth = args[i].split("=", 1)[1]
                        i += 1
                    elif args[i] == "--auth" and i + 1 < len(args):
                        auth = args[i + 1]
                        i += 2
                    elif args[i] == "--oauth":
                        auth = "oauth"
                        i += 1
                    elif args[i] == "--auto-auth":
                        auto_auth = True
                        i += 1
                    elif args[i] == "--no-auto-auth":
                        auto_auth = False
                        i += 1
                    elif args[i].startswith("--headers="):
                        headers = json.loads(args[i].split("=", 1)[1])
                        i += 1
                    elif args[i] == "--headers" and i + 1 < len(args):
                        headers = json.loads(args[i + 1])
                        i += 2
                    else:
                        positional.append(args[i])
                        i += 1

                server_cmd = None
                if positional:
                    if name is None and len(positional) >= 2:
                        name = positional[0]
                        server_cmd = " ".join(positional[1:])
                    else:
                        server_cmd = " ".join(positional)

                if not server_cmd:
                    print(
                        f"{YELLOW}⏺ Usage: /add-mcp <name> <server> [--auth <auth>|--oauth] [--headers '<json>'] [--auto-auth|--no-auto-auth]{RESET}"
                    )
                    continue

                if not name:
                    name = re.sub(r"[^a-zA-Z0-9_]+", "_", server_cmd).strip("_")
                    if not name:
                        name = f"mcp_{len(mcp_servers) + 1}"

                if name in mcp_servers:
                    for tool_name in mcp_servers[name]["tools"]:
                        agent.remove_tool(tool_name)

                try:
                    from mcp2py import load

                    kwargs = {}
                    if auth is not None:
                        kwargs["auth"] = auth
                    if headers:
                        kwargs["headers"] = headers
                    if auto_auth is not None:
                        kwargs["auto_auth"] = auto_auth

                    server = load(server_cmd, **kwargs)
                    tool_names = register_mcp_server(name, server)
                    mcp_servers[name] = {"server": server, "tools": tool_names}

                    print(
                        f"{GREEN}⏺ Added MCP server '{name}' with {len(tool_names)} tools{RESET}"
                    )
                    print(f"{GREEN}⏺ Tools: {list(agent.tools.keys())}{RESET}")
                except Exception as err:
                    print(f"{RED}⏺ Failed to add MCP server: {err}{RESET}")

                continue

            # Build context from history
            context = f"Working directory: {os.getcwd()}\n"
            if history:
                context += "\nPrevious conversation:\n"
                for h in history[-5:]:  # Keep last 5 exchanges
                    context += f"User: {h['user']}\nAssistant: {h['assistant']}\n\n"

            task = f"{context}\nCurrent task: {user_input}"

            print(f"\n{CYAN}⏺{RESET} Thinking...", flush=True)

            # Run the RLM agent
            try:
                result = agent(task=task)
            finally:
                if tmp_paste_path:
                    try:
                        os.remove(tmp_paste_path)
                    except OSError:
                        pass

            # Display the answer
            print(f"\n{CYAN}⏺{RESET} {render_markdown(result.answer)}")

            # Display affected files
            print(f"\n{MAGENTA}⏺ Affected files: {result.affected_files}{RESET}")

            # Save to history
            history.append({"user": user_input, "assistant": result.answer})

            print()

        except (KeyboardInterrupt, EOFError):
            break
        except Exception as err:
            import traceback

            traceback.print_exc()
            print(f"{RED}⏺ Error: {err}{RESET}")


if __name__ == "__main__":
    main()
