import os
import shlex
import getpass
from modaic import AutoProgram

from utils.cache import clear_openrouter_key, load_openrouter_key, save_openrouter_key
from utils.display import (
    BOLD,
    BLUE,
    CYAN,
    DIM,
    GREEN,
    MAGENTA,
    RED,
    RESET,
    YELLOW,
    render_markdown,
    separator,
)
from utils.models import handle_model_command, resolve_startup_models
from utils.mcp import handle_add_mcp_command, register_mcp_server
from utils.paste import LONG_PASTE_THRESHOLD, save_long_paste

MODAIC_REPO_PATH = "farouk1/nanocode"


def format_auth_error(err: Exception) -> str | None:
    message = str(err)
    if (
        "OpenrouterException" in message
        and "No cookie auth credentials found" in message
    ):
        return (
            "OpenRouter authentication failed: no credentials found. "
            "Set `OPENROUTER_API_KEY` or run `/openrouter-key` to cache it."
        )
    if "AuthenticationError" in message and "OpenrouterException" in message:
        return (
            "OpenRouter authentication failed. "
            "Set `OPENROUTER_API_KEY` or run `/openrouter-key` to cache it."
        )
    return None


def main():
    cached_key = load_openrouter_key()
    if cached_key and not os.getenv("OPENROUTER_API_KEY"):
        os.environ["OPENROUTER_API_KEY"] = cached_key
        print(f"{GREEN}⏺ Loaded OpenRouter key from cache{RESET}")

    model, sub_lm = resolve_startup_models()
    if os.getenv("MODAIC_ENV"):
        print(f"Using env: {os.getenv('MODAIC_ENV', 'prod')}")

    agent = AutoProgram.from_precompiled(
        MODAIC_REPO_PATH,
        rev=os.getenv("MODAIC_ENV", "prod"),
        config={"lm": model, "sub_lm": sub_lm},
    )

    print(
        f"{BOLD}NANOCODE DSPY{RESET} | {DIM}{model}{RESET} | {DIM}sub_lm:{sub_lm}{RESET} | {os.getcwd()}{RESET}\n"
    )

    history = []

    mcp_servers = {}  # mcp registry

    while True:
        try:
            print(separator())
            user_input = input(f"{BOLD}{BLUE}❯{RESET} ").strip()
            print(separator())

            tmp_paste_path = None
            if len(user_input) > LONG_PASTE_THRESHOLD:
                tmp_paste_path = save_long_paste(user_input)
                print(
                    f"{YELLOW}⏺ Long paste detected ({len(user_input)} chars). Saved to {tmp_paste_path}{RESET}"
                )
                user_input = (
                    f"The user pasted a long input ({len(user_input)} chars). "
                    f"It has been saved to {tmp_paste_path}. "
                    "Use read_file to view it. The file will be deleted after this response."
                )

            if not user_input:
                continue

            if user_input in ("/q", "exit"):
                break

            if user_input.startswith("/openrouter-key"):
                parts = shlex.split(user_input)
                args = parts[1:]

                if args and args[0] in ("clear", "unset", "remove"):
                    clear_openrouter_key()
                    os.environ.pop("OPENROUTER_API_KEY", None)
                    print(f"{GREEN}⏺ OpenRouter key cleared{RESET}")
                    continue

                key = (
                    args[0]
                    if args
                    else getpass.getpass(
                        f"{BOLD}{BLUE}❯{RESET} Enter OpenRouter API key (input hidden): "
                    ).strip()
                )

                if not key:
                    print(f"{RED}⏺ OpenRouter key not set (empty input){RESET}")
                    continue

                save_openrouter_key(key)
                os.environ["OPENROUTER_API_KEY"] = key
                print(f"{GREEN}⏺ OpenRouter key saved to cache{RESET}")
                continue

            if user_input == "/c":
                history = []
                print(f"{GREEN}⏺ Cleared conversation{RESET}")
                continue

            handled, agent, new_sub_lm = handle_model_command(
                user_input, agent, mcp_servers, register_mcp_server, MODAIC_REPO_PATH
            )

            if handled:
                if new_sub_lm:
                    sub_lm = new_sub_lm
                continue

            if handle_add_mcp_command(user_input, agent, mcp_servers):
                continue

            # build from history
            context = f"Working directory: {os.getcwd()}\n"
            if history:
                context += "\nPrevious conversation:\n"
                for h in history[-5:]:  # should be configurable
                    context += f"User: {h['user']}\nAssistant: {h['assistant']}\n\n"

            task = f"{context}\nCurrent task: {user_input}"

            print(f"\n{CYAN}⏺{RESET} Thinking...", flush=True)

            # run the RLM
            try:
                result = agent(task=task)
            finally:
                if tmp_paste_path:
                    try:
                        os.remove(tmp_paste_path)
                    except OSError:
                        pass

            # final output
            print(f"\n{CYAN}⏺{RESET} {render_markdown(result.answer)}")
            print(f"\n{MAGENTA}⏺ Affected files: {result.affected_files}{RESET}")

            # save to history
            history.append({"user": user_input, "assistant": result.answer})
            print()

        except (KeyboardInterrupt, EOFError):
            break
        except Exception as err:
            auth_message = format_auth_error(err)
            if auth_message:
                print(f"{RED}⏺ {auth_message}{RESET}")
                continue
            import traceback

            traceback.print_exc()
            print(f"{RED}⏺ Error: {err}{RESET}")


if __name__ == "__main__":
    main()
