import os
from typing import Any, Callable

from modaic import PrecompiledProgram

from .cache import load_model_config, save_model_config
from .display import BOLD, BLUE, DIM, GREEN, RED, RESET

AVAILABLE_MODELS = {
    "1": ("GPT-5.2 Codex", "openai/gpt-5.2-codex"),
    "2": ("GPT-5.2", "openai/gpt-5.2"),
    "3": ("Claude Opus 4.5", "anthropic/claude-opus-4.5"),
    "4": ("Claude Opus 4", "anthropic/claude-opus-4"),
    "5": ("Qwen 3 Coder", "qwen/qwen3-coder"),
    "6": ("Gemini 3 Flash Preview", "google/gemini-3-flash-preview"),
    "7": ("Kimi K2 0905", "moonshotai/kimi-k2-0905"),
    "8": ("Minimax M2.1", "minimax/minimax-m2.1"),
}


def normalize_model_id(model_id: str) -> str:
    assert isinstance(model_id, str), "model_id must be a str"
    return model_id if model_id.startswith("openrouter/") else f"openrouter/{model_id}"


def select_model() -> str:
    """Interactive model selection."""
    assert isinstance(AVAILABLE_MODELS, dict), "AVAILABLE_MODELS must be a dict"

    print(f"\n{BOLD}Select a model:{RESET}")
    for key, (name, model_id) in AVAILABLE_MODELS.items():
        print(f"  {BLUE}{key}{RESET}. {name} ({DIM}{model_id}{RESET})")
    print(f"  {BLUE}c{RESET}. Custom model (enter manually)")

    while True:
        try:
            choice = (
                input(f"\n{BOLD}{BLUE}❯{RESET} Enter choice (1-8 or c): ")
                .strip()
                .lower()
            )

            if choice in AVAILABLE_MODELS:
                name, model_id = AVAILABLE_MODELS[choice]
                print(f"{GREEN}⏺ Selected: {name}{RESET}")
                return model_id
            elif choice == "c":
                custom_model = input(
                    f"{BOLD}{BLUE}❯{RESET} Enter model ID (e.g., openai/gpt-4): "
                ).strip()
                if custom_model:
                    print(f"{GREEN}⏺ Selected custom model: {custom_model}{RESET}")
                    return custom_model
                else:
                    print(f"{RED}⏺ Invalid model ID{RESET}")
            else:
                print(f"{RED}⏺ Invalid choice. Please enter 1-8 or c{RESET}")
        except (KeyboardInterrupt, EOFError):
            print(f"\n{RED}⏺ Model selection cancelled{RESET}")
            exit(1)


def resolve_startup_models() -> tuple[str, str]:
    assert isinstance(AVAILABLE_MODELS, dict), "AVAILABLE_MODELS must be a dict"
    model_env = os.getenv("MODEL")
    sub_env = os.getenv("SUB_LM")
    cached_model, cached_sub_lm = load_model_config()

    if model_env:
        assert isinstance(model_env, str), "MODEL must be a str"
        model = model_env
    elif cached_model:
        print(f"{GREEN}⏺ Using cached model: {cached_model}{RESET}")
        model = cached_model
    else:
        model = select_model()
        normalized_model = normalize_model_id(model)
        save_model_config(normalized_model, normalized_model)
        cached_sub_lm = normalized_model

    if sub_env:
        assert isinstance(sub_env, str), "SUB_LM must be a str"
        sub_lm = sub_env
    elif cached_sub_lm:
        if cached_model and not model_env:
            print(f"{GREEN}⏺ Using cached sub_lm: {cached_sub_lm}{RESET}")
        sub_lm = cached_sub_lm
    else:
        sub_lm = model

    if cached_model and not cached_sub_lm:
        save_model_config(normalize_model_id(model), normalize_model_id(sub_lm))

    return normalize_model_id(model), normalize_model_id(sub_lm)


def handle_model_command(
    user_input: str,
    agent: PrecompiledProgram,
    mcp_servers: dict[str, dict[str, Any]],
    register_mcp_server: Callable[[PrecompiledProgram, str, Any], list[str]],
    repo_path: str,
) -> tuple[bool, PrecompiledProgram, str]:
    assert isinstance(user_input, str), "user_input must be a str"
    assert isinstance(agent, PrecompiledProgram), "agent must be PrecompiledProgram"
    assert isinstance(mcp_servers, dict), "mcp_servers must be a dict"
    assert callable(register_mcp_server), "register_mcp_server must be callable"
    assert isinstance(repo_path, str), "repo_path must be a str"

    if user_input != "/model":
        return False, agent, ""

    print(f"\n{BOLD}Current model: {agent.config.lm}{RESET}")
    print(f"{BOLD}Current sub_lm: {agent.config.sub_lm}{RESET}")
    print(f"\n{BOLD}Select a new model:{RESET}")
    for key, (name, model_id) in AVAILABLE_MODELS.items():
        print(f"  {BLUE}{key}{RESET}. {name} ({DIM}{model_id}{RESET})")
    print(f"  {BLUE}c{RESET}. Custom model (enter manually)")
    print(f"  {BLUE}k{RESET}. Keep current model")

    choice = input(f"\n{BOLD}{BLUE}❯{RESET} Enter choice: ").strip().lower()

    new_model = agent.config.lm
    if choice == "k":
        print(f"{GREEN}⏺ Keeping current model: {agent.config.lm}{RESET}")
    elif choice in AVAILABLE_MODELS:
        name, model_id = AVAILABLE_MODELS[choice]
        new_model = normalize_model_id(model_id)
        print(f"{GREEN}⏺ Selected: {name} ({new_model}){RESET}")
    elif choice == "c":
        custom_model = input(f"{BOLD}{BLUE}❯{RESET} Enter model ID: ").strip()
        if not custom_model:
            print(f"{RED}⏺ Invalid model ID, keeping current model{RESET}")
            return True, agent, normalize_model_id(agent.config.sub_lm)
        new_model = normalize_model_id(custom_model)
        print(f"{GREEN}⏺ Selected custom model: {new_model}{RESET}")
    else:
        print(f"{RED}⏺ Invalid choice, keeping current model{RESET}")
        return True, agent, normalize_model_id(agent.config.sub_lm)

    print(f"\n{BOLD}Select a sub_lm:{RESET}")
    for key, (name, model_id) in AVAILABLE_MODELS.items():
        print(f"  {BLUE}{key}{RESET}. {name} ({DIM}{model_id}{RESET})")
    print(f"  {BLUE}m{RESET}. Use primary model")
    print(f"  {BLUE}c{RESET}. Custom model (enter manually)")
    print(f"  {BLUE}k{RESET}. Keep current sub_lm")

    sub_choice = input(f"\n{BOLD}{BLUE}❯{RESET} Enter choice: ").strip().lower()
    if sub_choice == "k":
        new_sub_lm = normalize_model_id(agent.config.sub_lm)
        print(f"{GREEN}⏺ Keeping current sub_lm: {new_sub_lm}{RESET}")
    elif sub_choice == "m":
        new_sub_lm = normalize_model_id(new_model)
        print(f"{GREEN}⏺ sub_lm set to primary model: {new_sub_lm}{RESET}")
    elif sub_choice in AVAILABLE_MODELS:
        _, model_id = AVAILABLE_MODELS[sub_choice]
        new_sub_lm = normalize_model_id(model_id)
        print(f"{GREEN}⏺ Selected sub_lm: {new_sub_lm}{RESET}")
    elif sub_choice == "c":
        custom_sub = input(f"{BOLD}{BLUE}❯{RESET} Enter sub_lm ID: ").strip()
        if not custom_sub:
            print(f"{RED}⏺ Invalid sub_lm ID, keeping current sub_lm{RESET}")
            new_sub_lm = normalize_model_id(agent.config.sub_lm)
        else:
            new_sub_lm = normalize_model_id(custom_sub)
            print(f"{GREEN}⏺ Selected custom sub_lm: {new_sub_lm}{RESET}")
    else:
        print(f"{RED}⏺ Invalid choice, keeping current sub_lm{RESET}")
        new_sub_lm = normalize_model_id(agent.config.sub_lm)

    agent = agent.__class__.from_precompiled(
        repo_path, config={"lm": normalize_model_id(new_model), "sub_lm": new_sub_lm}
    )
    for server_name, info in mcp_servers.items():
        info["tools"] = register_mcp_server(agent, server_name, info["server"])

    save_model_config(normalize_model_id(new_model), new_sub_lm)
    print(
        f"{GREEN}⏺ Switched to: {normalize_model_id(new_model)} | sub_lm: {new_sub_lm}{RESET}"
    )
    return True, agent, new_sub_lm
