# 安装配置

本章节详细介绍 Midscene Python 的安装步骤、环境配置和依赖管理。

## 📋 系统要求

### 基础要求
- **Python**: 3.9 或更高版本
- **操作系统**: Windows 10+, macOS 10.14+, Linux (Ubuntu 18.04+)
- **内存**: 最少 4GB RAM（推荐 8GB+）
- **网络**: 稳定的互联网连接（用于 AI 模型调用）

### AI 模型要求
至少需要以下 AI 服务之一的 API 访问权限：
- OpenAI GPT-4V
- Anthropic Claude 3
- 阿里云通义千问 VL
- Google Gemini Pro Vision

## 🚀 快速安装

### 方式一：使用 pip 安装（推荐）
```bash
# 安装最新版本
pip install midscene-python

# 或指定版本
pip install midscene-python==0.1.0
```

### 方式二：从源码安装
```bash
# 克隆仓库
git clone https://gitee.com/Python51888/midscene-python.git
cd midscene-python

# 安装依赖并安装
pip install -e .
```

### 方式三：开发者安装
```bash
# 克隆仓库
git clone https://gitee.com/Python51888/midscene-python.git
cd midscene-python

# 安装开发依赖
pip install -e ".[dev,docs]"

# 安装 pre-commit hooks
pre-commit install
```

## 🔧 平台特定配置

### Web 自动化配置

#### Selenium 配置
```bash
# 安装 Selenium 和 WebDriver 管理器
pip install selenium webdriver-manager

# Python 代码中自动管理驱动
from selenium import webdriver
from webdriver_manager.chrome import ChromeDriverManager
from selenium.webdriver.chrome.service import Service

service = Service(ChromeDriverManager().install())
driver = webdriver.Chrome(service=service)
```

#### Playwright 配置
```bash
# 安装 Playwright
pip install playwright

# 安装浏览器
playwright install

# 仅安装 Chromium（节省空间）
playwright install chromium
```

### Android 自动化配置

#### ADB 设置
```bash
# 安装 ADB（Ubuntu/Debian）
sudo apt-get install android-tools-adb

# 安装 ADB（macOS）
brew install android-platform-tools

# 安装 ADB（Windows）
# 下载 Android SDK Platform Tools
# 添加到系统 PATH
```

#### 设备连接
```bash
# 启用开发者选项和 USB 调试
# 连接设备后验证
adb devices

# 预期输出
List of devices attached
DEVICE_ID    device
```

## 🔑 AI 模型配置

### 环境变量配置
创建 `.env` 文件：

```bash
# OpenAI 配置
OPENAI_API_KEY=sk-your-openai-api-key
OPENAI_BASE_URL=https://api.openai.com/v1  # 可选

# Anthropic 配置
ANTHROPIC_API_KEY=sk-ant-your-anthropic-key

# 通义千问配置
DASHSCOPE_API_KEY=sk-your-dashscope-key

# Gemini 配置
GOOGLE_API_KEY=AIza-your-google-api-key

# 默认模型配置
MIDSCENE_AI_PROVIDER=openai
MIDSCENE_AI_MODEL=gpt-4-vision-preview
```

### 代码配置
```python
from midscene.core.ai_model import AIModelConfig

# 多个 AI 提供商配置
configs = {
    "openai": AIModelConfig(
        provider="openai",
        model="gpt-4-vision-preview",
        api_key="your-openai-key",
        temperature=0.1
    ),
    "claude": AIModelConfig(
        provider="anthropic", 
        model="claude-3-sonnet-20240229",
        api_key="your-claude-key",
        temperature=0.1
    )
}
```

## 📦 依赖管理

### 核心依赖
```toml
# pyproject.toml 中的核心依赖
[project]
dependencies = [
    "pydantic>=2.0,<3.0",
    "selenium>=4.15.0,<5.0", 
    "playwright>=1.40.0,<2.0",
    "opencv-python>=4.8.0,<5.0",
    "pillow>=10.0.0,<11.0",
    "aiohttp>=3.9.0,<4.0",
    "loguru>=0.7.0,<1.0",
    "typer>=0.9.0,<1.0",
    "httpx>=0.25.0,<1.0",
    "openai>=1.3.0,<2.0",
    "anthropic>=0.7.0,<1.0"
]
```

### 可选依赖
```bash
# 开发工具
pip install "midscene-python[dev]"

# 文档工具
pip install "midscene-python[docs]"

# 全部依赖
pip install "midscene-python[dev,docs]"
```

## 🔍 验证安装

### 基础验证
```python
# test_installation.py
import asyncio
from midscene import Agent
from midscene.core.ai_model import AIModelService

async def test_installation():
    """测试安装是否成功"""
    
    # 测试导入
    print("✓ 导入模块成功")
    
    # 测试 AI 服务配置
    try:
        ai_service = AIModelService()
        print("✓ AI 服务初始化成功")
    except Exception as e:
        print(f"✗ AI 服务初始化失败: {e}")
    
    print("🎉 安装验证完成！")

# 运行测试
asyncio.run(test_installation())
```

### Web 平台验证
```python
# test_web.py
import asyncio
from midscene import Agent
from midscene.web import SeleniumWebPage

async def test_web():
    """测试 Web 平台功能"""
    try:
        with SeleniumWebPage.create() as page:
            agent = Agent(page)
            await page.goto("https://www.example.com")
            print("✓ Web 自动化测试成功")
    except Exception as e:
        print(f"✗ Web 自动化测试失败: {e}")

asyncio.run(test_web())
```

### Android 平台验证
```python
# test_android.py
import asyncio
from midscene import Agent
from midscene.android import AndroidDevice

async def test_android():
    """测试 Android 平台功能"""
    try:
        device = AndroidDevice()
        await device.connect()
        agent = Agent(device)
        print("✓ Android 自动化测试成功")
    except Exception as e:
        print(f"✗ Android 自动化测试失败: {e}")

asyncio.run(test_android())
```

## 🔧 常见问题解决

### Python 版本问题
```bash
# 检查 Python 版本
python --version

# 如果版本低于 3.9，安装新版本
# Ubuntu/Debian
sudo apt-get install python3.9

# macOS
brew install python@3.9

# Windows
# 从 python.org 下载安装
```

### 依赖冲突解决
```bash
# 创建虚拟环境（推荐）
python -m venv midscene-env
source midscene-env/bin/activate  # Linux/macOS
# 或
midscene-env\Scripts\activate     # Windows

# 在虚拟环境中安装
pip install midscene-python
```

### 网络连接问题
```bash
# 使用国内镜像源
pip install -i https://pypi.tuna.tsinghua.edu.cn/simple midscene-python

# 或配置永久镜像源
pip config set global.index-url https://pypi.tuna.tsinghua.edu.cn/simple
```

### AI API 连接问题
```python
# 测试 API 连接
import os
import httpx

async def test_openai_connection():
    api_key = os.getenv("OPENAI_API_KEY")
    if not api_key:
        print("❌ 未设置 OPENAI_API_KEY")
        return
    
    async with httpx.AsyncClient() as client:
        try:
            response = await client.get(
                "https://api.openai.com/v1/models",
                headers={"Authorization": f"Bearer {api_key}"}
            )
            if response.status_code == 200:
                print("✅ OpenAI API 连接正常")
            else:
                print(f"❌ OpenAI API 连接失败: {response.status_code}")
        except Exception as e:
            print(f"❌ 网络连接错误: {e}")
```

## 🚀 性能优化配置

### 系统级优化
```bash
# 增加文件描述符限制（Linux/macOS）
ulimit -n 65536

# 设置环境变量优化
export PYTHONUNBUFFERED=1
export PYTHONDONTWRITEBYTECODE=1
```

### Python 配置优化
```python
# config.py
import asyncio

# 设置异步事件循环策略
if hasattr(asyncio, 'WindowsSelectorEventLoopPolicy'):
    asyncio.set_event_loop_policy(asyncio.WindowsSelectorEventLoopPolicy())

# 配置日志级别
import logging
logging.getLogger("httpx").setLevel(logging.WARNING)
logging.getLogger("selenium").setLevel(logging.WARNING)
```

## 📋 配置检查清单

### 安装完成检查
- [ ] Python 3.9+ 已安装
- [ ] midscene-python 包已安装
- [ ] 至少一个 AI 提供商已配置
- [ ] Web 驱动程序已安装（如果使用 Web 自动化）
- [ ] ADB 已安装并设备已连接（如果使用 Android 自动化）

### 环境配置检查
- [ ] 环境变量已设置
- [ ] API 密钥有效且有足够额度
- [ ] 网络连接正常
- [ ] 防火墙和代理配置正确

### 功能测试检查
- [ ] 基础导入测试通过
- [ ] AI 服务初始化成功
- [ ] 选择的平台（Web/Android）测试通过
- [ ] 示例代码可以正常运行

## 🔄 升级和维护

### 版本升级
```bash
# 检查当前版本
pip show midscene-python

# 升级到最新版本
pip install --upgrade midscene-python

# 升级特定版本
pip install midscene-python==0.2.0
```

### 配置备份
```bash
# 备份配置文件
cp .env .env.backup
cp pyproject.toml pyproject.toml.backup

# 导出依赖列表
pip freeze > requirements.txt
```

### 清理和重装
```bash
# 卸载当前版本
pip uninstall midscene-python

# 清理缓存
pip cache purge

# 重新安装
pip install midscene-python
```

---

完成配置后，您就可以开始使用 Midscene Python 进行 AI 驱动的自动化了！接下来推荐阅读 [快速开始](快速开始.md) 指南。