# 快速开始

欢迎使用 Midscene Python！本指南将帮助你在 5 分钟内上手 AI 驱动的自动化操作。

## 📋 前置要求

在开始之前，请确保你的环境满足以下要求：

- **Python 3.9+**
- **pip** 包管理器
- **浏览器** (Chrome/Firefox/Edge，用于 Web 自动化)
- **AI 模型 API Key** (OpenAI、Claude、Qwen 或 Gemini 任选其一)

## 🚀 快速安装

### 1. 安装 Midscene Python

```bash
pip install midscene-python
```

### 2. 安装浏览器驱动（可选）

如果你计划进行 Web 自动化，需要安装对应的浏览器驱动：

```bash
# Selenium WebDriver
pip install webdriver-manager

# 或者 Playwright
pip install playwright
playwright install
```

### 3. 配置 AI 模型

创建 `.env` 文件配置 AI 模型（以 OpenAI 为例）：

```bash
# .env
OPENAI_API_KEY=your_openai_api_key_here
OPENAI_BASE_URL=https://api.openai.com/v1  # 可选，默认官方 API
```

## 🎯 第一个示例

让我们从一个简单的 Web 自动化示例开始：

### 示例 1: 搜索操作

```python
import asyncio
from midscene import Agent
from midscene.web import SeleniumWebPage

async def search_example():
    """在百度搜索 Python 教程"""
    
    # 创建 Web 页面实例
    with SeleniumWebPage.create() as page:
        # 创建 Agent
        agent = Agent(page)
        
        # 导航到网站
        await page.goto("https://www.baidu.com")
        
        # 使用自然语言进行搜索
        await agent.ai_action("在搜索框输入'Python 教程'")
        await agent.ai_action("点击搜索按钮")
        
        # 验证搜索结果
        await agent.ai_assert("页面显示了 Python 教程的搜索结果")
        
        print("✅ 搜索操作完成！")

# 运行示例
asyncio.run(search_example())
```

### 示例 2: 数据提取

```python
import asyncio
from midscene import Agent
from midscene.web import SeleniumWebPage

async def extract_example():
    """提取新闻标题"""
    
    with SeleniumWebPage.create() as page:
        agent = Agent(page)
        
        # 访问新闻网站
        await page.goto("https://news.example.com")
        
        # 提取结构化数据
        news_data = await agent.ai_extract({
            "articles": [
                {
                    "title": "新闻标题",
                    "time": "发布时间",
                    "summary": "新闻摘要"
                }
            ]
        })
        
        # 输出结果
        for article in news_data["articles"]:
            print(f"📰 {article['title']}")
            print(f"⏰ {article['time']}")
            print(f"📄 {article['summary']}\n")

# 运行示例
asyncio.run(extract_example())
```

## 📱 Android 自动化示例

```python
import asyncio
from midscene import Agent
from midscene.android import AndroidDevice

async def android_example():
    """Android 应用自动化"""
    
    # 连接 Android 设备
    device = AndroidDevice()
    await device.connect()
    
    # 创建 Agent
    agent = Agent(device)
    
    # 启动应用
    await device.start_app("com.example.app")
    
    # 自然语言操作
    await agent.ai_action("点击登录按钮")
    await agent.ai_action("输入用户名 'testuser'")
    await agent.ai_action("输入密码 'password123'")
    await agent.ai_action("点击确认登录")
    
    # 验证登录状态
    await agent.ai_assert("显示用户已登录")
    
    print("✅ Android 自动化完成！")

# 运行示例
asyncio.run(android_example())
```

## 🎛️ 配置选项

### AI 模型配置

```python
from midscene.core.ai_model import AIModelConfig

# 自定义 AI 配置
config = AIModelConfig(
    provider="openai",  # 或 "claude", "qwen", "gemini"
    model="gpt-4-vision-preview",
    temperature=0.1,
    max_tokens=1000
)

agent = Agent(page, ai_config=config)
```

### Agent 选项

```python
from midscene.core import AgentOptions

# 自定义 Agent 选项
options = AgentOptions(
    timeout=30,  # 操作超时时间（秒）
    retry_count=3,  # 重试次数
    screenshot_on_error=True,  # 错误时自动截图
    cache_enabled=True  # 启用智能缓存
)

agent = Agent(page, options=options)
```

## 🔧 常用操作

### 基础交互

```python
# 点击操作
await agent.ai_action("点击提交按钮")
await agent.ai_action("点击页面右上角的用户头像")

# 输入操作
await agent.ai_action("在用户名框输入 'admin'")
await agent.ai_action("在密码框输入密码")

# 滚动操作
await agent.ai_action("向下滚动查看更多内容")
await agent.ai_action("滚动到页面底部")

# 等待操作
await agent.ai_action("等待页面加载完成")
```

### 元素定位

```python
# 精确定位元素
element = await agent.ai_locate("登录按钮")
await element.click()

# 定位多个元素
elements = await agent.ai_locate_all("商品卡片")
for element in elements:
    await element.hover()
```

### 条件断言

```python
# 页面状态验证
await agent.ai_assert("用户已成功登录")
await agent.ai_assert("购物车显示 3 件商品")
await agent.ai_assert("页面不包含错误信息")

# 元素存在性验证
await agent.ai_assert("页面包含搜索结果")
await agent.ai_assert("显示用户个人信息")
```

## 📊 查看执行报告

Midscene Python 自动生成详细的执行报告：

```python
# 运行后，检查生成的报告文件
# 报告位置: ./midscene_reports/
# - execution_report.html  # 可视化报告
# - screenshots/           # 执行截图
# - logs/                  # 详细日志
```

## 🔍 调试技巧

### 启用详细日志

```python
import logging
from midscene.shared.logger import setup_logger

# 启用调试日志
setup_logger(level=logging.DEBUG)
```

### 截图调试

```python
# 手动截图
screenshot = await page.screenshot()
with open("debug.png", "wb") as f:
    f.write(screenshot)

# 获取页面信息
context = await page.get_context()
print(f"页面标题: {context.page_title}")
print(f"页面 URL: {context.url}")
```

## 🚨 常见问题

### 1. AI 模型调用失败
```python
# 检查 API Key 配置
import os
print(f"API Key: {os.getenv('OPENAI_API_KEY')[:10]}...")
```

### 2. 元素定位失败
```python
# 使用更具体的描述
await agent.ai_action("点击页面左上角的蓝色登录按钮")
```

### 3. 页面加载问题
```python
# 添加等待时间
await page.wait_for_page_load()
await agent.ai_action("等待 3 秒让页面完全加载")
```

## 🎓 下一步

恭喜！你已经掌握了 Midscene Python 的基础用法。接下来可以：

1. 📖 深入学习 [核心概念](核心概念/Agent核心控制器.md)
2. 🔧 查看 [API 参考](API参考/Agent-API.md)
3. 🌐 了解 [Web 自动化](平台集成/Web自动化/README.md) 高级特性
4. 📱 探索 [Android 自动化](平台集成/Android自动化.md)
5. 🎯 参考 [最佳实践](示例和教程/最佳实践.md)

## 💡 小贴士

- 使用具体、清晰的自然语言描述能获得更好的执行效果
- 定期查看执行报告来优化自动化脚本
- 善用缓存机制来提升执行效率
- 为不同环境配置不同的 AI 模型

---

*准备好探索更多功能了吗？查看我们的 [示例集合](示例和教程/基础示例.md)！*