"""Configuration and options dataclasses."""

from __future__ import annotations

import sys
from typing import Literal

from mkdocstrings import get_logger
from pydantic import BaseModel, Field

# YORE: EOL 3.10: Replace block with line 2.
if sys.version_info >= (3, 11):
    pass
else:
    pass


logger = get_logger(__name__)


SIGNATURE_VERSION = Literal["ref", "major", "semver", "string"]
PARAMETERS_ORDER = Literal["alphabetical", "source"]
PARAMETERS_SECTION_STYLE = Literal["table", "list"]


class GitHubOptions(BaseModel):
    """The following are the available options for the GitHub handler.
    The options are loaded as a pydantic model, so they are type-checked and validated.
    """

    # General options
    show_description: bool = Field(
        default=True,
        description="Whether to show the description in the documentation.",
    )

    description: str = Field(
        default="",
        description="A custom string to override the autogenerated description of the object.",
    )

    show_source: bool = Field(
        default=True,
        description="Whether to show the source link in the documentation.",
    )

    # Heading options
    show_heading: bool = Field(
        default=True,
        description="Whether to show the heading in the documentation.",
    )

    heading: str = Field(
        default="",
        description="A custom string to override the autogenerated heading of the object.",
    )

    heading_level: int = Field(
        default=2,
        description="The initial heading level to use.",
    )

    show_branding: bool = Field(
        default=True,
        description="Whether to show the action branding (logo and color) in the documentation.",
    )

    branding_icon: str = Field(
        default="",
        description="Custom icon from https://feathericons.com/ to use for the branding.",
    )

    branding_icon_color: str = Field(
        default="",
        description="Custom icon color for the feather icon.",
    )

    show_toc_entry: bool = Field(
        default=True,
        description="If the heading is not shown, at least add a ToC entry for it.",
    )

    toc_label: str = Field(
        default="",
        description="A custom string to override the autogenerated toc label of the root object.",
    )

    # Signature options
    show_signature: bool = Field(
        default=True,
        description="Whether to show the signature in the documentation.",
    )

    signature_repository: str = Field(
        default="",
        description="""The GitHub repository in the format *owner/repo*.

        By default, the repository is inferred from the current git repository using the default origin remote.
        If it cannot be inferred, it must be set manually.
        """,
    )

    signature_show_secrets: bool = Field(
        default=False,
        description="Whether to show secrets in the signature.",
    )

    signature_show_permissions: bool = Field(
        default=True,
        description="Whether to show permissions in the workflow signature.",
    )

    signature_version: SIGNATURE_VERSION = Field(
        default="ref",
        description="""The versioning scheme to use for the signature.

        - `ref`: use the git ref (branch or tag) from which the workflow or action is run,
        - `major`: use the latest release tag matching `vX` (e.g. `v1`, `v2`),
        - `semver`: use the latest release tag matching `vX.X.X` (e.g. `v1.0.0`, `v2.1.3`),
        - `string`: use the string provided in the [`signature_version_string`][mkdocstrings_handlers.github.config.GitHubOptions.signature_version_string] option.
        """,
    )

    signature_version_string: str = Field(
        default="latest",
        description="The version string to use if [`signature_version`][mkdocstrings_handlers.github.config.GitHubOptions.signature_version] is set to `string`.",
    )

    # Parameter options
    show_inputs: bool = Field(
        default=True,
        description="Whether to show inputs in the documentation.",
    )

    show_inputs_only_required: bool = Field(
        default=False,
        description="Whether to show only required inputs in the documentation.",
    )

    show_outputs: bool = Field(
        default=False,
        description="Whether to show outputs in the documentation.",
    )

    show_secrets: bool = Field(
        default=True,
        description="Whether to show secrets in the documentation.",
    )

    show_secrets_only_required: bool = Field(
        default=False,
        description="Whether to show only required secrets in the documentation.",
    )

    parameters_order: PARAMETERS_ORDER = Field(
        default="source",
        description="""The parameters ordering to use.

        - `alphabetical`: order by the parameters names,
        - `source`: order parameters as they appear in the source file.
        """,
    )

    parameters_section_style: PARAMETERS_SECTION_STYLE = Field(
        default="table",
        description="""The style used to render docstring sections.

        - `table`: render parameters in a table,
        - `list`: render parameters in a list.
        """,
    )

    parameters_anchors: bool = Field(
        default=True,
        description="Whether to add anchors to parameters in the documentation.",
    )


class GitHubConfig(BaseModel):
    """Configuration options for the GitHub handler."""

    feather_icons_source: str = Field(
        default="https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js",
        description="""The source URL for Feather icons.

        In certain enterprise environments, external CDN access may be restricted.
        In such cases, you can host the `feather.min.js` file locally and set this option to the local path or URL.
        See more information at https://github.com/feathericons/feather.
        """,
    )

    options: GitHubOptions = Field(
        default_factory=GitHubOptions,
        description="Options for the GitHub handler.",
    )
