---
hide:
    - navigation
---

# Usage

## Relative cross-references

As already mentioned before, `mkdocstrings-python-betterrefs` allows you to use a custom improved syntax when specifying
your cross-references, to make your references shorter and easier to comprehend. The most important feature is the
relative cross-references support. Check the example below:

=== "Absolute"

    ```python title="mypkg/mymod.py"
    class OtherClass: ...

    class MyClass:
        def other_method(self): ...

        def this_method(self):
            """
            See [other_method][mypkg.mymod.MyClass.other_method] and
            the [OtherClass][mypkg.mymod.OtherClass].
            """
    ```

=== "Relative"

    ```python title="mypkg/mymod.py"
    class OtherClass: ...

    class MyClass:
        def other_method(self): ...

        def this_method(self):
            """
            See [other_method][..other_method] and
            the [OtherClass][(m).OtherClass]
            """
    ```

The relative path specifier works as follows:

<!-- prettier-ignore-start -->

- `(c)`: Replaced by the path of the **class** that contains the docstring.
- `(m)`: Replaced by the path of the **module** that contains the docstring.
- `(p)`: Replaced by the path of the **package** that contains the docstring. (If this is a stand-alone module, this
module will be treated as a package.)

- **One or more `.` characters:** Expanded to the path of the current docstring (or its parent elements).

    For example, in a method's docstring:

    - `.` is replaced by the method name,
    - `..` is replaced by the class name, and
    - `...` is replaced by the module name.

- **One or more `^` characters:** Replaced by the path of the parent element. This is a shorthand for .., which is
  commonly used.

    For instance, in a method's docstring:

    - `^` is replaced by the class name, and
    - `^^` is replaced by the module name.

<!-- prettier-ignore-end -->

!!! note

    When using either `^` or `..` we have found that going up more than one or two levels makes cross-references
    difficult to read and should be avoided

## Avoiding repetition

In addition to relative reference support, there's special handling to reduce repetitive reference declarations in the
title and the cross-reference target. For instance, instead of writing `[MyClass][..MyClass]`, you can simply write
`[MyClass][..]`, resulting in a much cleaner and more compact syntax.

This rule applies when the cross-reference path ends with a period (`.`); in such cases, the title text is automatically
appended to the path (ignoring any bold, italic, or code markup).

## Demonstration

Quick demonstration:

=== "mkdocstrings-python-betterrefs"

    ```python title="this_package/this_module.py"
    def some_func(): ...

    class MyClass:
        def that_method(self): ...

        def this_method(self):
            """
            [MyClass][^]
            Also [MyClass][(c)]
            And [`MyClass`][(m).] yet again
            [`that_method`][^.]
            Also [`that_method`][..]
            [init method][(c).__init__]
            [this module][(m)]
            [this package][(p)]
            [that module][(p).that_module] or [that module][(p).]
            [OtherClass][(m).]
            [some_func][^^.] or [some_func][...]
            """
    ```

=== "mkdocstrings-python"

    ```python title="this_package/this_module.py"
    def some_func(): ...

    class MyClass:
        def that_method(self): ...

        def this_method(self):
            """
            [MyClass][this_package.this_module.MyClass]
            Also [MyClass][this_package.this_module.MyClass]
            And [`MyClass`][this_package.this_module.MyClass] yet again
            [`that_method`][this_package.this_module.MyClass.that_method]
            Also [`that_method`][this_package.this_module.MyClass.that_method]
            [init method][this_package.this_module.MyClass.__init__]
            [this module][this_package.this_module]
            [this package][this_package]
            [that module][this_package.that_module] or [that module][this_package.that_module]
            [OtherClass][this_package.this_module.OtherClass]
            [some_func][this_package.this_module.some_func] or [some_func][this_package.this_module]
            """
    ```

## Cross-reference checking

If `check_crossrefs` is enabled (default), then all cross-reference expressions will be validated to ensure that they
exist. Failures will be reported with the source location information.

If disabled, missing cross-references will still be reported by `mkdocstrings` directly, but these reports lack source
location details, which can make it challenging to locate the problematic docstring.

Note that the errors generated by this feature are in addition to the errors from `mkdocstrings`, which will mean you
will see 2 errors for each invalid reference.

!!! warning

    The current implementation of this feature can produce false errors for definitions from the python standard
    library, or external imported libraries. You can disable the check on a case-by-case basis by prefixing the
    reference expression with a `?`, for example:

    ```python
    def foo() -> pathlib.Path:
        """
        This function returns a [Path][?pathlib.] instance.
        """
    ```
