import logging
from dataclasses import dataclass
from functools import cached_property
from pathlib import Path
from typing import NamedTuple

from mkdocs.structure.files import File
from mkdocs.structure.nav import Page

log = logging.getLogger(__name__)


@dataclass(frozen=True)
class ModuleRef:
    """Module Reference

    The module ``foo.bar.baz`` will have id ``("foo", "bar", "baz")``

    Module references are created by searching through source directories
    """
    base_path: Path
    """Base path containing this package tree"""
    module_path: Path
    """Fully qualified path inside the base path"""

    def __post_init__(self):
        # Validate that module_path is inside base_path
        _ = self.ref_path

    @property
    def module_id(self) -> str:
        ref_path = self.ref_path
        if ref_path[-1] == "__init__":
            ref_path = ref_path[:-1]
        return ".".join(ref_path)

    @property
    def printable_module_id(self) -> str:
        """Printable module id

        This is the same as ``module_name`` except ``__init__`` is stripped."""
        ref_path = self.ref_path
        if ref_path[-1] == "__init__":
            ref_path = ref_path[:-1]
        return ".".join(ref_path)

    @property
    def module_name(self) -> str:
        """Name of the module for internal reference only"""
        ref_path = self.ref_path
        return ref_path[-1] if ref_path[-1] != "__init__" else ref_path[-2]

    @cached_property
    def ref_path(self) -> tuple[str, ...]:
        """Abstract representation of a python module broken into parts"""
        return self.module_path.relative_to(self.base_path).with_suffix("").parts


@dataclass(frozen=True)
class GeneratedFileRef:
    """Generated markdown file reference.

    Stores information about markdown files generated by this plugin."""
    module_ref: ModuleRef
    """Representation of a python module"""
    doc_file_path: Path
    """Absolute path to markdown file on disk"""
    file: File
    """Mkdocs File object for use by mkdocs only"""


class PageRef(NamedTuple):
    """Page reference.

    Reference to a page that was rendered from a generated markdown file"""
    page: Page
    file: GeneratedFileRef
