import pathlib
import unittest

import pytest

from mmif import Mmif, Document, AnnotationTypes
from mmif.utils import sequence_helper as sqh
from mmif.utils import text_document_helper as tdh
from mmif.utils import timeunit_helper as tuh
from mmif.utils import video_document_helper as vdh
from tests.mmif_examples import *
from hypothesis import given, strategies as st


class TestTimeunitHelper(unittest.TestCase):
    FPS = 100  # 1 frame == 10 ms

    @given(st.integers(min_value=300, max_value=10000))
    def test_convert(self, value):
        # using hypothesis library, test time unit conversions 
        # four-way between second, millisecond, framenum, and isoformat 
        for tu1 in ['s', 'm', 'f', 'i']:
            for tu2 in ['s', 'm', 'f', 'i']:
                if tu1 == tu2:
                    continue
                # since the input (generated by hypothesis) is always integer/millisecond
                # I'm going to dual round-trip conversion
                # ms > tu1 > tu2 > tu1 > ms 
                # then see the ends are equal (enough)
                in_t = tuh.convert(value, 'm', tu1, self.FPS)
                converted = tuh.convert(in_t, tu1, tu2, self.FPS)
                converted_back = tuh.convert(converted, tu2, tu1, self.FPS)
                out_t = tuh.convert(converted_back, tu1, 'm', self.FPS)
                self.assertAlmostEqual(value, out_t, delta=1000/self.FPS)  # up to 1 frame error


class TestVideoDocumentHelper(unittest.TestCase):
    def setUp(self):
        self.fps = 29.97
        self.mmif_obj = Mmif(validate=False)
        self.a_view = self.mmif_obj.new_view()
        self.video_doc = Document({
            "@type": "http://mmif.clams.ai/vocabulary/VideoDocument/v1",
            "properties": {
                "mime": "video",
                "id": "d1",
                "location": f"file://{pathlib.Path(__file__).parent}/black-2997fps.mp4"
            }
        })
        self.video_doc.add_property('fps', self.fps)
        self.mmif_obj.add_document(self.video_doc)

    def test_extract_mid_frame(self):
        tf = self.a_view.new_annotation(AnnotationTypes.TimeFrame, start=100, end=200, timeUnit='frame', document=self.video_doc.id)
        self.assertEqual(150, vdh.get_mid_framenum(self.mmif_obj, tf))
        tf = self.a_view.new_annotation(AnnotationTypes.TimeFrame, start=0, end=200, timeUnit='frame', document=self.video_doc.id)
        self.assertEqual(100, vdh.get_mid_framenum(self.mmif_obj, tf))
        tf = self.a_view.new_annotation(AnnotationTypes.TimeFrame, start=0, end=3, timeUnit='seconds', document=self.video_doc.id)
        self.assertEqual(vdh.convert(1.5, 's', 'f', self.fps), vdh.get_mid_framenum(self.mmif_obj, tf))

    def test_extract_representative_frame(self):
        tp = self.a_view.new_annotation(AnnotationTypes.TimePoint, timePoint=1500, timeUnit='milliseconds', document=self.video_doc.id)
        tf = self.a_view.new_annotation(AnnotationTypes.TimeFrame, start=1000, end=2000, timeUnit='milliseconds', document=self.video_doc.id)
        tf.add_property('representatives', [tp.id])
        rep_frame_num = vdh.get_representative_framenum(self.mmif_obj, tf)
        expected_frame_num = vdh.millisecond_to_framenum(self.video_doc, tp.get_property('timePoint'))
        self.assertEqual(expected_frame_num, rep_frame_num)
        # and should work even if no representatives are provided
        tf = self.a_view.new_annotation(AnnotationTypes.TimeFrame, start=1000, end=2000, timeUnit='milliseconds', document=self.video_doc.id)
        self.assertEqual(vdh.get_representative_framenum(self.mmif_obj, tf), vdh.get_mid_framenum(self.mmif_obj, tf))
        # check there is an error if there is a representative referencing a timepoint that does not exist
        tf.add_property('representatives', ['fake_tp_id'])
        with pytest.raises(ValueError):
            vdh.get_representative_framenum(self.mmif_obj, tf)

    def test_get_framerate(self):
        self.assertAlmostEqual(29.97, vdh.get_framerate(self.video_doc), places=0)

    def test_frames_to_seconds(self):
        self.assertAlmostEqual(3.337, vdh.framenum_to_second(self.video_doc, 100), places=0)

    def test_frames_to_milliseconds(self):
        self.assertAlmostEqual(3337.0, vdh.framenum_to_millisecond(self.video_doc, 100), places=0)

    def test_seconds_to_frames(self):
        self.assertAlmostEqual(100, vdh.second_to_framenum(self.video_doc, 3.337), places=0)

    def test_milliseconds_to_frames(self):
        self.assertAlmostEqual(100, vdh.millisecond_to_framenum(self.video_doc, 3337.0), places=0)

    def test_convert_roundtrip(self):
        # ms for 1 frame
        tolerance = 1000 / self.video_doc.get_property('fps')
        for ms in [1000, 1234, 4321, 44444, 789789]:
            m2f = vdh.millisecond_to_framenum(self.video_doc, ms)
            m2f2m = vdh.framenum_to_millisecond(self.video_doc, m2f)
            self.assertAlmostEqual(ms, m2f2m, delta=tolerance)

    def test_sample_frames(self):
        s_frame = vdh.second_to_framenum(self.video_doc, 3)
        e_frame = vdh.second_to_framenum(self.video_doc, 5.5)
        # note that int(29.97) = 29
        self.assertEqual(3, len(vdh.sample_frames(s_frame, e_frame, self.fps)))
        s_frame = vdh.second_to_framenum(self.video_doc, 3)
        e_frame = vdh.second_to_framenum(self.video_doc, 5)
        self.assertEqual(1, len(vdh.sample_frames(s_frame, e_frame, 60)))

    def test_convert_timepoint(self):
        timepoint_ann = self.a_view.new_annotation(AnnotationTypes.BoundingBox, timePoint=3, timeUnit='second',
                                                   document='d1')
        self.assertEqual(vdh.convert(3, 's', 'f', self.fps), vdh.convert_timepoint(self.mmif_obj, timepoint_ann, 'f'))

    def test_convert_timeframe(self):
        self.a_view.metadata.new_contain(AnnotationTypes.TimeFrame, timeUnit='frame', document=self.video_doc.id)
        timeframe_ann = self.a_view.new_annotation(AnnotationTypes.TimeFrame, start=100, end=200)
        for times in zip((3.337, 6.674), vdh.convert_timeframe(self.mmif_obj, timeframe_ann, 's')):
            self.assertAlmostEqual(*times, places=0)

    def test_extract_frames_as_images(self):
        frame_list = [5, 10, 15]
        target_images = vdh.extract_frames_as_images(self.video_doc, frame_list, as_PIL=False)
        self.assertEqual(3, len(target_images))
        # check if the extract_frames_as_images destroy the input frame list
        self.assertEqual(3, len(frame_list))
        # check return empty list if the frame list is empty
        empty_flist = []
        empty_target_images = vdh.extract_frames_as_images(self.video_doc, empty_flist, as_PIL=False)
        self.assertEqual([], empty_target_images)
        # check there is an error if there is a frame in the list that does not exist
        tot_fcount = self.video_doc.get_property('frameCount')
        frame_list.append(tot_fcount + 1)
        new_target_images = vdh.extract_frames_as_images(self.video_doc, frame_list, as_PIL=False)
        self.assertEqual(4, len(frame_list))
        self.assertEqual(3, len(new_target_images))


class TestSequenceHelper(unittest.TestCase):

    def test_validate_labelset(self):
        mmif_obj = Mmif(validate=False)
        view = mmif_obj.new_view()
        anns = [view.new_annotation(AnnotationTypes.TimePoint, labelset=['a', 'b', 'c']) for _ in range(3)]
        self.assertTrue(sqh.validate_labelset(anns))
        anns.append(view.new_annotation(AnnotationTypes.TimePoint, labelset=['a', 'b', 'c', 'd']))
        with pytest.raises(ValueError):
            self.assertFalse(sqh.validate_labelset(anns))
        anns.pop()
        anns.append(view.new_annotation(AnnotationTypes.TimePoint))
        with pytest.raises(KeyError):
            self.assertFalse(sqh.validate_labelset(anns))

    def test_build_remapper(self):
        self.assertEqual({'a': 'a', 'b': 'b', 'c': 'c', 'd': 'd', 'e': 'e', 'f': 'f'},
                         sqh.build_label_remapper(list('abcdef'), {}))
        self.assertEqual({'a': 1, 'b': 2, 'c': 1, 'd': '-', 'e': '-', 'f': '-'},
                         sqh.build_label_remapper(list('abcdef'), {'a': 1, 'b': 2, 'c': 1}))

    def test_build_score_lists(self):
        c1 = {'a': 0.1, 'b': 0.2, 'c': 0.3}
        c2 = {'a': 0.6, 'b': 0.5, 'c': 0.4}
        remap = {'a': 'x', 'b': 'y', 'c': 'x'}
        lblmap, scores = sqh.build_score_lists([c1, c2], remap)
        self.assertEqual(2, len(scores.shape))
        self.assertEqual(2, scores.shape[0])
        self.assertEqual(2, scores.shape[1])
        self.assertEqual({'x': [0.3, 0.6], 'y': [0.2, 0.5]}, {lbl: list(scores[idx]) for lbl, idx in lblmap.items()})
        _, scores = sqh.build_score_lists([c1, c2], remap, score_remap_op=min)
        self.assertEqual({'x': [0.1, 0.4], 'y': [0.2, 0.5]}, {lbl: list(scores[idx]) for lbl, idx in lblmap.items()})

    def test_width_based_smoothing(self):
        scores = [0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1]
        # idx = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9]
        # res = [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1],
        # cannot pass non-positive threshold for sequence sizes
        with pytest.raises(ValueError):
            sqh.smooth_outlying_short_intervals(scores, 1, 0)
        with pytest.raises(ValueError):
            sqh.smooth_outlying_short_intervals(scores, 0, 1)
        with pytest.raises(ValueError):
            sqh.smooth_outlying_short_intervals(scores, 0, 0)
        self.assertEqual([(1, 13), (19, 20)],
                         sqh.smooth_outlying_short_intervals(scores, 1, 4))
        # idx = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9]
        # res = [0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        self.assertEqual([(1, 7)],
                         sqh.smooth_outlying_short_intervals(scores, 4, 2))
        # idx = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9]
        # res = [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0],
        self.assertEqual([(1, 13)],
                         sqh.smooth_outlying_short_intervals(scores, 4, 4))
        # special test case for not trimming short end peaks adjacent to short gaps
        scores = [1, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1]
        # idx = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9]
        # res = [1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1],
        self.assertEqual([(0, 7), (11, 20)],
                         sqh.smooth_outlying_short_intervals(scores, 4, 4))
        # special test case for stitching only mode
        scores = [0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1]
        # idx = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9]
        # res = [0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1],
        self.assertEqual([(3, 7), (11, 15), (18, 20)],
                         sqh.smooth_outlying_short_intervals(scores, 1, 1))


class TestTextDocHelper(unittest.TestCase):
    mmif_obj = Mmif(MMIF_EXAMPLES['everything'])

    @pytest.mark.skip("The only valid test cases come from kaldi app which annotates wrong property")
    def test_slice_text(self):
        sliced_text_full_overlap = tdh.slice_text(self.mmif_obj, 11500, 14600)
        sliced_text_partial_overlap = tdh.slice_text(self.mmif_obj, 7, 10, unit="seconds")
        no_sliced_text = tdh.slice_text(self.mmif_obj, 0, 5000)
        full_sliced_text = tdh.slice_text(self.mmif_obj, 0, 22, unit="seconds")
        self.assertEqual("In the nineteen eighties ,", sliced_text_full_overlap)
        self.assertEqual("is Jim Lehrer with the NewsHour", sliced_text_partial_overlap)
        self.assertEqual("", no_sliced_text)
        self.assertEqual(
            "Hello , this is Jim Lehrer with the NewsHour on PBS . "
            "In the nineteen eighties , barking dogs have increasingly become a problem in urban areas .",
            full_sliced_text)


if __name__ == '__main__':
    unittest.main()
