# This file is auto-generated by setup.py
import warnings
from typing import ClassVar


class TypesBase(object):
    """
    Base class for arbitrary vocabulary type. 
    This class provides bisic initializer, comparators, 
    and (de-)serialization methods. 
    """
    _prefixes = {}
    base_uri: str
    shortname: str
    fuzzy_eq: bool
    # these names are migrated from the old LAPPSgrid vocab 
    old_lapps_type_shortnames = {'Token', 'Sentence', 'Paragraph', 'Markable', 'NamedEntity', 'NounChunk', 'VerbChunk'}

    def __init__(self, type_uri: str):
        """
        Initialize a vocabulary type.

        :param type_uri: full URI or short name of the vocabulary type
        """
        self.fuzzy_eq = False
        self.parse_names(type_uri)
        if self.__repr__() not in self.__class__._prefixes:
            # prepare auto-inferred prefix for Annotations.[].id field
            self.__class__._prefixes[self.__repr__()] \
                = self.__class__._create_prefix(self.shortname,
                                                self.__class__._prefixes.values()
                                                )
    
    def parse_names(self, type_uri: str):
        if '/' in type_uri:
            self.base_uri, self.shortname = type_uri.rsplit('/', 1)
        else:
            self.base_uri = ""
            self.shortname = type_uri

    @classmethod
    def _create_prefix(cls, shortname, existing_prefixes):
        
        def extract_chars(string, indices):
            return "".join(string[i].lower() for i in sorted(set(indices)) if i < len(string))
        
        if shortname.isalnum():
            
            cap_letters = [i for i, c in enumerate(shortname) if c.isupper()] + [len(shortname)]
            if len(cap_letters) > 1:
                max_diff = max(cap_letters[i] - cap_letters[i-1] for i in range(1, len(cap_letters)))
            else:
                max_diff = len(shortname)
            more_letters = []
            prefix = extract_chars(shortname, cap_letters)
            if prefix in existing_prefixes:
                for i in range(1, max_diff):
                    for j in cap_letters:
                        more_letters.append(j + i)
                        prefix = extract_chars(shortname, cap_letters + more_letters)
                        if prefix not in existing_prefixes:
                            break
                    else:
                        continue
                    break
        else:
            prefix = None
        return prefix

    @staticmethod
    def attype_iri_isdocument(iri):
        return 'Document/' in iri or iri.endswith('Document')

    @classmethod
    def from_str(cls, string: str):
        if 'mmif.clams.ai' in string:
            if cls.attype_iri_isdocument(string):
                return DocumentTypesBase(string)
            else:
                return AnnotationTypesBase(string)
        # quirks for legacy LAPPSgrid types
        elif 'vocab.lappsgrid.org' in string and string.split('/')[-1] in cls.old_lapps_type_shortnames:
            # legacy LAPPSgrid types
            shortname = string.split('/')[-1]
            # when migrated, their version was set to 'v1'
            return AnnotationTypesBase(f'http://mmif.clams.ai/vocabulary/{shortname}/v1')
        else:
            return cls(string)
    
    def __hash__(self):
        return hash(str(self))
        
    def __eq__(self, other):
        if isinstance(other, str):
            other = self.from_str(other)
        return isinstance(other, TypesBase) and self.base_uri == other.base_uri and self.shortname == other.shortname
            
    def __repr__(self):
        if self.base_uri:
            return f'{self.base_uri}/{self.shortname}'
        else:
            return self.shortname
    
    # aliases
    def __str__(self):
        return self.__repr__()
    
    def _serialize(self):
        return self.__repr__()

    def get_prefix(self):
        return self.__class__._prefixes[self.__repr__()]

    def set_prefix(self, prefix):
        self.__class__._prefixes[self.__repr__()] = prefix

ThingTypesBase = TypesBase


class ClamsTypesBase(TypesBase):
    """
    Base class for CLAMS vocabulary types. Main 
    This class adds handling of MMIF specification versions 
    in initializer and comparators. 
    """
    version: str
    dev_version: ClassVar[str] = 'develop'
    
    def __init__(self, type_uri, fuzzymode=True):
        """
        Initialize a CLAMS vocabulary type.

        :param type_uri: full URI or short name of the CLAMS vocabulary type
        :param fuzzymode: if True, enables fuzzy equality comparison that ignores version differences
        """
        super().__init__(type_uri)
        self.fuzzy_eq = fuzzymode
    
    def parse_names(self, type_uri:str):
        if not type_uri:
            self.base_uri, self.shortname, self.version = "", "", ""
        elif 'mmif.clams.ai' not in type_uri:
            raise ValueError(f'Cannot process "{type_uri}"!: not a CLAMS vocabulary URI')
        elif type_uri[-1].isdigit():  # new versioning
            self.base_uri, _, self.shortname, self.version = type_uri.rsplit('/', 3)
        else:  # legacy versioning
            self.base_uri, self.version, _, self.shortname = type_uri.rsplit('/', 3)
            self._check_legacy_version(self.version)

    def _check_legacy_version(self, legacy_ver):
        if legacy_ver < '0.4.0':
            raise ValueError(f'Cannot process "{self}"!: CLAMS annotation types from old MMIF (older than 0.4.0) are no '
                             f'longer supported with this app.')
        if legacy_ver > '0.4.2':
            raise ValueError(f'Cannot process "{self}"!: Invalid CLAMS vocabulary type URI.')
        return True

    def __hash__(self):
        return hash(f'{self.base_uri}::{self.shortname}')
    
    def normalize_attype_vers(self):
        if '.' in self.version and self._check_legacy_version(self.version):
            # legacy mapping: see https://github.com/clamsproject/mmif/issues/14#issuecomment-1504439497
            if self.version == '0.4.2' and self.shortname == 'Annotation':
                return 'v2'
            elif self.version.startswith('0.4.'):
                return 'v1'
        else:
            return self.version

    def _eq_internal(self, other, fuzzy):
        if isinstance(other, ClamsTypesBase):
            if fuzzy:
                vers = [x.normalize_attype_vers() for x in (self, other)]
                if self.base_uri == other.base_uri and self.shortname == other.shortname:
                    if vers[0] != vers[1]:
                        warnings.warn(f'version difference detected: {self.version}&{other.version}@{self.shortname}')
                    return True
                return False
            else:
                return self.base_uri == other.base_uri \
                    and self.shortname == other.shortname \
                    and self.version == other.version
        else:
            # quirks for legacy LAPPSgrid types
            if self.shortname in self.old_lapps_type_shortnames and str(other) == f'http://vocab.lappsgrid.org/{self.shortname}':
                return True
            return False
    
    def __eq__(self, other):
        if isinstance(other, str):
            other = ThingTypesBase.from_str(other)
        return self._eq_internal(other, self.fuzzy_eq and other.fuzzy_eq)
    
    def __repr__(self):
        if self.version[0] == 'v':
            return f'{self.base_uri}/vocabulary/{self.shortname}/{self.version}'
        else:
            return f'{self.base_uri}/{self.version}/vocabulary/{self.shortname}'


class AnnotationTypesBase(ClamsTypesBase):
    """
    Inherit from this class to build your own custom annotation
    vocabularies. 
    """
    ...


class DocumentTypesBase(ClamsTypesBase):
    """
    Inherit from this class to build your own custom document
    vocabularies. 
    """
    ...


class ThingType(ThingTypesBase):
    """
    This class contains the topmost CLAMS thing type 
    defined in the spec version 1.1.0 as a class variable. 
    """
    Thing = ClamsTypesBase('http://mmif.clams.ai/vocabulary/Thing/v1')
    _typevers = {'Thing': 'v1'}
