import datetime
import pprint
from collections.abc import MutableMapping
from typing import Any, Iterator, Optional

from bson.objectid import ObjectId

STATUS_NEW = 'new'
STATUS_PENDING = 'pending'
STATUS_FAILED = 'failed'
STATUS_SUCCESSFUL = 'successful'


class Task(MutableMapping):
    """
    Task objects are building blocks of their corresponding TaskQueue.

    Task is a mapping class which encapsulates attributes and data
    pertaining to a certain job.
    """

    def __init__(
        self,
        _id: Optional[ObjectId] = None,
        assignedTo: Optional[str] = None,
        createdAt: Optional[float] = None,
        modifiedAt: Optional[float] = None,
        scheduledAt: Optional[float] = None,
        leaseId: Optional[str] = None,
        leaseExpiresAt: Optional[float] = None,
        status: Optional[str] = None,
        errorMessage: Optional[str] = None,
        retries: int = 0,
        priority: int = 0,
        payload: Any = None,
        dedupeKey: Optional[str] = None,
        rateLimitKey: Optional[str] = None,
    ):
        """
        Instantiates a Task instance.

        :param _id: the Task ObjectId generated by MongoDB
        :param assignedTo: the consumer assigned to the Task
        :param createdAt: the timestamp when the Task object was first created
        :param modifiedAt: the timestamp of the most recent change to the Task
        :param scheduledAt: earliest timestamp that a Task can be leased
        :param leaseId: lease identifier for the current Task assignment
        :param leaseExpiresAt: timestamp when the lease expires
        :param status: the Task status. It can take values from the following
                       list: ['new', 'pending', 'failed', 'successful']
        :param errorMessage: last error message occurred while
                               processing the Task
        :param retries: number of attempts to process the Task
        :param priority: Task priority based on which Tasks are pulled
                         from the TaskQueue
        :param payload: data payload which can be used to store data
                        with the Task
        :param dedupeKey: optional idempotency key for avoiding duplicates
        :param rateLimitKey: optional key for rate-limited consumption
        """
        datetime_now = datetime.datetime.now()
        self._id = _id or ObjectId()
        self.assignedTo = assignedTo
        self.createdAt = createdAt or datetime_now.timestamp()
        self.modifiedAt = modifiedAt or datetime_now.timestamp()
        self.scheduledAt = scheduledAt
        self.leaseId = leaseId
        self.leaseExpiresAt = leaseExpiresAt
        self.status = status or STATUS_NEW
        self.retries = retries
        self.errorMessage = errorMessage
        self.priority = priority
        self.payload = payload
        self.dedupeKey = dedupeKey
        self.rateLimitKey = rateLimitKey

    @property
    def object_id_(self):
        return self._id

    @property
    def assigned_to_(self):
        return self.assignedTo

    @property
    def created_at_(self):
        return self.createdAt

    @property
    def modified_at_(self):
        return self.modifiedAt

    @property
    def scheduled_at_(self):
        return self.scheduledAt

    @property
    def lease_id_(self):
        return self.leaseId

    @property
    def lease_expires_at_(self):
        return self.leaseExpiresAt

    @property
    def status_(self):
        return self.status

    @property
    def retries_(self):
        return self.retries

    @property
    def error_message_(self):
        return self.errorMessage

    @property
    def priority_(self):
        return self.priority

    @property
    def payload_(self):
        return self.payload

    @property
    def dedupe_key_(self):
        return self.dedupeKey

    @property
    def rate_limit_key_(self):
        return self.rateLimitKey

    def __setitem__(self, key, value) -> None:
        self.__setattr__(key, value)

    def __getitem__(self, key) -> Any:
        return self.__getattribute__(key)

    def __delitem__(self, key: str) -> None:
        self.__delattr__(key)

    def __len__(self) -> int:
        return len(self.__dict__)

    def __iter__(self) -> Iterator[str]:
        return iter(self.__dict__)

    def __repr__(self):
        attrs_ = dict(self.__dict__)
        created_at = attrs_.get('createdAt')
        if isinstance(created_at, (int, float)):
            attrs_['createdAt'] = datetime.datetime.fromtimestamp(created_at)
        modified_at = attrs_.get('modifiedAt')
        if isinstance(modified_at, (int, float)):
            attrs_['modifiedAt'] = datetime.datetime.fromtimestamp(modified_at)
        scheduled_at = attrs_.get('scheduledAt')
        if isinstance(scheduled_at, (int, float)):
            attrs_['scheduledAt'] = datetime.datetime.fromtimestamp(scheduled_at)
        lease_expires_at = attrs_.get('leaseExpiresAt')
        if isinstance(lease_expires_at, (int, float)):
            attrs_['leaseExpiresAt'] = datetime.datetime.fromtimestamp(
                lease_expires_at
            )
        return pprint.pformat(attrs_)
