# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mongodantic']

package_data = \
{'': ['*']}

install_requires = \
['motor>=3.1.1,<4.0.0', 'pydantic>=1.10.4,<2.0.0', 'pymongo>=4.3.3,<5.0.0']

setup_kwargs = {
    'name': 'mongodantic-python',
    'version': '0.2.0',
    'description': 'Pydantic models for MongoDB',
    'long_description': '# Mongodantic\n\n[![GitHub Workflow Status](https://img.shields.io/github/actions/workflow/status/cocreators-ee/mongodantic/publish.yaml)](https://github.com/cocreators-ee/mongodantic/actions/workflows/publish.yaml)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Security: bandit](https://img.shields.io/badge/security-bandit-green.svg)](https://github.com/PyCQA/bandit)\n[![Pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/cocreators-ee/mongodantic/blob/master/.pre-commit-config.yaml)\n[![PyPI](https://img.shields.io/pypi/v/mongodantic-python)](https://pypi.org/project/mongodantic-python/)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/mongodantic-python)](https://pypi.org/project/mongodantic-python/)\n[![License: BSD 3-Clause](https://img.shields.io/badge/License-BSD%203--Clause-blue.svg)](https://opensource.org/licenses/BSD-3-Clause)\n\nPython async database models for MongoDB using Pydantic base models. It should work on Python 3.9+, maybe 3.8, not quite\nsure.\n\n## Motivation\n\nIt\'s usually a good idea to have a simple layer on your DB that doesn\'t try to do too much, but takes care of the basic\nthings like validating your data and mapping database records to class instances, and overall providing basic database\naccess helpers. [Pydantic](https://docs.pydantic.dev) does a great job at the typing of models and validating data, and\njust needs a bit of database logic around it to provide all the capabilities commonly needed.\n\nThere are similar libraries already for other databases that serve as inspiration for this,\ne.g. [firedantic](http://github.com/ioxiocom/firedantic) for Firestore,\nand [arangodantic](https://github.com/ioxiocom/arangodantic) for ArangoDB.\n\n## Installation\n\nIt\'s a Python library, what do you expect?\n\n```bash\npip install mongodantic-python\n# OR\npoetry add mongodantic-python\n```\n\n## Usage\n\nSmall example of how you can use this library (also in [readme_example.py](./readme_example.py)).\n\n```python\nimport asyncio\nfrom datetime import datetime\nfrom typing import Optional, Sequence\n\nfrom motor.motor_asyncio import AsyncIOMotorClient\nfrom pydantic import Field\n\n# IndexModel and ASCENDING are just passed through from pymongo\nfrom mongodantic import ASCENDING, IndexModel, Model, ModelNotFoundError, set_database\n\nMONGODB_CONNECT_STR = "mongodb://localhost:27017"  # Point to your MongoDB server\n\n\nclass User(Model):\n  # Indexes are automatically created when model is accessed\n  indexes: Sequence[IndexModel] = [\n    IndexModel(\n      keys=[\n        ("name", ASCENDING),\n      ]\n    ),\n  ]\n\n  # id properly is automatically added - stored as _id in MongoDB\n\n  # Pydantic typing + Field usage works great\n  created: datetime = Field(default_factory=datetime.now)\n  name: Optional[str]\n\n  # You can of course add methods\n  def greet(self):\n    print(f"Hello, {self.name} from {self.created}")\n\n  async def rename(self):\n    self.name = f"Another {self.name}"\n    await self.save()\n\n  # You can also run code after loading objects from DB\n  async def after_load(self) -> None:\n    self.greet()\n\n\nasync def main():\n  # Configure the DB connection at the start of your application\n  print("Connecting to DB")\n  client = AsyncIOMotorClient(MONGODB_CONNECT_STR)\n  db = client["my_test_db"]\n  set_database(db)\n\n  # You can use this for cleanup\n  # for user in await User.find({}):\n  #     await user.delete()\n\n  # And just use the models\n  print("Creating user")\n  user = User()\n  await user.save()\n\n  print("Updating user")\n  user.name = "Test"\n  await user.save()\n\n  print("Renaming user")\n  await user.rename()\n\n  # Load up a specific one if you know the str representation of its id\n  print("Searching by ID")\n  user_again = await User.get_by_id(user.id)\n  assert user_again.name == "Another Test"\n\n  # Find many\n  # {} is a Pymongo filter, if filtering by id make sure you use "_id" key and ObjectId() for value\n  print("Finding all users")\n  users = await User.find({})\n  assert len(users) == 1\n\n  # Counting\n  for idx in range(0, 10):\n    u = User(\n      name=f"user-{idx + 1}"\n    )\n    await u.save()\n\n  assert await User.count() == 11\n  assert await User.count({"name": user.name}) == 1\n\n  # Pagination\n  users = await User.find({"name": {"$ne": user.name}}, skip=3, limit=3)\n  assert len(users) == 3\n  for u in users:\n    print(u.name)\n\n  # Load up the first matching entry\n  print("Finding a user by name")\n  test_user = await User.find_one({"name": "Another Test"})\n  assert test_user.id == user.id\n\n  print("Deleting user")\n  await user.delete()\n\n  try:\n    print("Attempting reload")\n    await user.reload()\n    raise Exception("User was supposed to be deleted")\n  except ModelNotFoundError:\n    print("User not found")\n\n\nif __name__ == "__main__":\n  asyncio.run(main())\n```\n\n## Development\n\nIssues and PRs are welcome!\n\nPlease open an issue first to discuss the idea before sending a PR so that you know if it would be wanted or needs\nre-thinking or if you should just make a fork for yourself.\n\nFor local development, make sure you install [pre-commit](https://pre-commit.com/#install), then run:\n\n```bash\npre-commit install\npoetry install\npoetry run pytest-watch  # Hit Ctrl+C when done with your changes\npoetry run python readme_example.py\n```\n\n## License\n\nThe code is released under the BSD 3-Clause license. Details in the [LICENSE.md](./LICENSE.md) file.\n\n# Financial support\n\nThis project has been made possible thanks to [Cocreators](https://cocreators.ee) and [Lietu](https://lietu.net). You\ncan help us continue our open source work by supporting us\non [Buy me a coffee](https://www.buymeacoffee.com/cocreators).\n\n[!["Buy Me A Coffee"](https://www.buymeacoffee.com/assets/img/custom_images/orange_img.png)](https://www.buymeacoffee.com/cocreators)\n',
    'author': 'Janne Enberg',
    'author_email': 'janne.enberg@lietu.net',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/cocreators-ee/mongodantic/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
