#!/usr/bin/python3

#
#   Developer: Alexey Zakharov (alexey.zakharov@vectioneer.com)
#   All rights reserved. Copyright (c) 2016-2025 VECTIONEER.
#

"""
motorcortex.request

Provides the Request class for managing request connections to a Motorcortex server,
including login, parameter retrieval, parameter updates, and group management.
"""

import base64
import hashlib
import json
import tempfile
import os
from threading import Event
from concurrent.futures import ThreadPoolExecutor, Future
from typing import Any, Callable, List, Optional, Union
from pynng import Req0, TLSConfig
from enum import Enum

from motorcortex.reply import Reply
from motorcortex.setup_logger import logger
from motorcortex.state_callback_handler import StateCallbackHandler
from motorcortex.parameter_tree import ParameterTree
from motorcortex.message_types import MessageTypes
from motorcortex.nng_url import NngUrl


class ConnectionState(Enum):
    """Enumeration of connection states.
    
        - CONNECTING:           Connection is being established.
        - CONNECTION_OK:        Connection is successfully established.
        - CONNECTION_LOST:      Connection was lost.
        - CONNECTION_FAILED:    Connection attempt failed.
        - DISCONNECTING:        Connection is being closed.
        - DISCONNECTED:         Connection is closed.
    """
    CONNECTING = 0
    CONNECTION_OK = 1
    CONNECTION_LOST = 2
    CONNECTION_FAILED = 3
    DISCONNECTING = 4
    DISCONNECTED = 5


class Request:
    """
    Represents a request connection to a Motorcortex server.

    The Request class allows you to:
    - Establish and manage a connection to a Motorcortex server.
    - Perform login authentication.
    - Retrieve, set, and overwrite parameter values.
    - Manage parameter groups for efficient batch operations.
    - Save and load parameter trees.
    - Chain asynchronous operations using a promise-like interface (`Reply`).

    Methods:
        url() -> Optional[str]
            Returns the current connection URL.

        connect(url: str, **kwargs) -> Reply
            Establishes a connection to the server.

        close() -> None
            Closes the connection and cleans up resources.

        send(encoded_msg: Any, do_not_decode_reply: bool = False) -> Optional[Reply]
            Sends an encoded message to the server.

        login(login: str, password: str) -> Reply
            Sends a login request.

        connectionState() -> ConnectionState
            Returns the current connection state.

        getParameterTreeHash() -> Reply
            Requests the parameter tree hash from the server.

        getParameterTree() -> Reply
            Requests the parameter tree from the server.

        save(path: str, file_name: str) -> Reply
            Requests the server to save the parameter tree to a file.

        setParameter(path: str, value: Any, type_name: Optional[str] = None, offset: int = 0, length: int = 0) -> Reply
            Sets a new value for a parameter.

        setParameterList(param_list: List[dict]) -> Reply
            Sets new values for a list of parameters.

        getParameter(path: str) -> Reply
            Requests a parameter value and description.

        getParameterList(path_list: List[str]) -> Reply
            Requests values and descriptions for a list of parameters.

        overwriteParameter(path: str, value: Any, force_activate: bool = False, type_name: Optional[str] = None) -> Reply
            Overwrites a parameter value and optionally forces it to stay active.

        releaseParameter(path: str) -> Reply
            Releases the overwrite operation for a parameter.

        createGroup(path_list: List[str], group_alias: str, frq_divider: int = 1) -> Reply
            Creates a subscription group for a list of parameters.

        removeGroup(group_alias: str) -> Reply
            Unsubscribes from a group.

    Examples:
        >>> # Establish a connection
        >>> req = motorcortex.Request(protobuf_types, parameter_tree)
        >>> reply = req.connect("tls+tcp://localhost:6501", certificate="path/to/ca.crt")
        >>> if reply.get():
        ...     print("Connected!")
        >>> # Login
        >>> login_reply = req.login("user", "password")
        >>> if login_reply.get().status == motorcortex.OK:
        ...     print("Login successful")
        >>> # Get a parameter
        >>> param_reply = req.getParameter("MyDevice.MyParam")
        >>> param = param_reply.get()
        >>> print("Value:", param.value)
        >>> # Set a parameter
        >>> req.setParameter("MyDevice.MyParam", 42)
        >>> # Clean up
        >>> req.close()
    """

    def __init__(self, protobuf_types: "MessageTypes", parameter_tree: "ParameterTree", number_of_threads=2) -> None:
        """
        Initialize a Request object.

        Args:
            protobuf_types: Motorcortex message types module.
            parameter_tree: ParameterTree instance.
            number_of_threads (int): Thread pool size (minimum 1, None - use default (CPU-based)).
        """
        self.__socket: Optional[Req0] = None
        self.__url: Optional[str] = None
        self.__connected_event: Optional[Event] = None
        self.__connected: bool = False
        self.__protobuf_types: "MessageTypes" = protobuf_types
        self.__parameter_tree: "ParameterTree" = parameter_tree
        self.__connection_state: ConnectionState = ConnectionState.DISCONNECTED
        self.__pool: ThreadPoolExecutor = ThreadPoolExecutor(max_workers=number_of_threads,
                                                             thread_name_prefix="mcx_req")
        self.__callback_handler: StateCallbackHandler = StateCallbackHandler()
        logger.debug("Request object initialized with state: DISCONNECTED")

    def url(self) -> Optional[str]:
        """Return the current connection URL."""
        return self.__url

    def connect(self, url: str, **kwargs) -> Reply:
        """
        Establish a connection to the Motorcortex server.

        Args:
            url: Connection URL.
            **kwargs: Additional connection parameters. Supported keys include:
                certificate (str, optional): Path to a TLS certificate file for secure connections.
                conn_timeout_ms (int, optional): Connection timeout in milliseconds (default: 1000).
                recv_timeout_ms (int, optional): Receive timeout in milliseconds (default: 500).
                login (str, optional): Username for authentication (if required by server).
                password (str, optional): Password for authentication (if required by server).
                state_update (Callable, optional): Callback function to be called on connection state changes.
                timeout_ms (int, optional): Alternative timeout in milliseconds (used by Request.parse).

        Returns:
            Reply: A promise that resolves when the connection is established.
        """
        logger.debug(f"[CONNECT] Starting connection to {url}")
        logger.debug(f"[CONNECT] Current state: {self.__connection_state.name}")

        self.__connection_state = ConnectionState.CONNECTING
        conn_timeout_ms, recv_timeout_ms, certificate, state_update = self.parse(**kwargs)

        logger.debug(f"[CONNECT] Parameters - timeout: {conn_timeout_ms} ms, recv_timeout: {recv_timeout_ms} ms, "
                     f"has_cert: {certificate is not None}, has_state_update: {state_update is not None}")

        if state_update:
            logger.debug("[CONNECT] Starting state callback handler")
            self.__callback_handler.start(state_update)

        self.__url = url
        tls_config = None
        if certificate:
            parsed = NngUrl(url)
            logger.debug(f"[CONNECT] Using TLS with certificate: {certificate}")
            tls_config = TLSConfig(TLSConfig.MODE_CLIENT, ca_files=certificate, server_name=parsed.hostname)

        logger.debug(f"[CONNECT] Creating Req0 socket with recv_timeout={recv_timeout_ms}ms")
        self.__socket = Req0(recv_timeout=recv_timeout_ms, tls_config=tls_config)
        self.__connected_event = Event()  # Create a fresh event for each connection
        self.__connected = False  # Reset state

        logger.debug("[CONNECT] Socket created, registering callbacks")

        def pre_connect_cb(_pipe):
            logger.debug(f"[CALLBACK] PRE_CONNECT fired - Connection established")
            old_state = self.__connection_state.name
            self.__connected = True
            self.__connection_state = ConnectionState.CONNECTION_OK
            logger.debug(f"[CALLBACK] State transition: {old_state} -> {self.__connection_state.name}")
            self.__callback_handler.notify(self, self.connectionState())
            self.__connected_event.set()
            logger.debug("[CALLBACK] Connection event set, connected=True")

        def post_remove_cb(_pipe):
            logger.debug(f"[CALLBACK] POST_REMOVE fired - Connection lost/failed")
            old_state = self.__connection_state.name

            if self.__connection_state == ConnectionState.DISCONNECTING:
                self.__connection_state = ConnectionState.DISCONNECTED
                logger.debug(f"[CALLBACK] Clean disconnect: {old_state} -> DISCONNECTED")
            elif self.__connection_state == ConnectionState.CONNECTING:
                self.__connection_state = ConnectionState.CONNECTION_FAILED
                logger.debug(f"[CALLBACK] Connection failed: {old_state} -> CONNECTION_FAILED")
            elif self.__connection_state == ConnectionState.CONNECTION_OK:
                self.__connection_state = ConnectionState.CONNECTION_LOST
                logger.debug(f"[CALLBACK] Connection lost: {old_state} -> CONNECTION_LOST")

            self.__connected = False
            self.__callback_handler.notify(self, self.connectionState())
            self.__connected_event.set()
            logger.debug("[CALLBACK] Connection event set, connected=False")

        self.__socket.add_pre_pipe_connect_cb(pre_connect_cb)
        self.__socket.add_post_pipe_remove_cb(post_remove_cb)
        logger.debug("[CONNECT] Callbacks registered successfully")

        logger.debug(f"[CONNECT] Starting dial to {url} (non-blocking)")
        self.__socket.dial(url, block=False)

        logger.debug(f"[CONNECT] Submitting waitForConnection to thread pool with timeout={conn_timeout_ms / 1000.0}s")
        return Reply(self.__pool.submit(self.waitForConnection, self.__connected_event,
                                        conn_timeout_ms / 1000.0, lambda: self.__connected))

    def close(self) -> None:
        """
        Close the request connection and clean up resources.
        """
        logger.debug("[CLOSE] Closing connection")
        logger.debug(f"[CLOSE] Current state: {self.__connection_state.name}, connected: {self.__connected}")

        self.__connection_state = ConnectionState.DISCONNECTING
        if self.__connected_event:
            self.__connected = False
            self.__connected_event.set()
            logger.debug("[CLOSE] Connection event set for shutdown")
        else:
            logger.debug("[CLOSE] No connection event to set")

        if self.__socket:
            logger.debug("[CLOSE] Closing socket")
            self.__socket.close()
        else:
            logger.debug("[CLOSE] No socket to close")

        logger.debug("[CLOSE] Stopping callback handler")
        self.__callback_handler.stop()

        logger.debug("[CLOSE] Shutting down thread pool (blocking)")
        self.__pool.shutdown(wait=True)
        logger.debug("[CLOSE] Connection closed successfully")

    def send(self, encoded_msg: Any, do_not_decode_reply: bool = False) -> Optional[Reply]:
        """
        Send an encoded message to the server.

        Args:
            encoded_msg: Encoded protobuf message.
            do_not_decode_reply: If True, do not decode the reply.

        Returns:
            Reply or None: A promise for the reply, or None if not connected.
        """
        if self.__socket is not None:
            return Reply(self.__pool.submit(self.__send, self.__socket, encoded_msg,
                                            None if do_not_decode_reply else self.__protobuf_types))
        logger.debug("[SEND] Attempted to send on null socket - connection not established?")
        return None

    def login(self, login: str, password: str) -> Reply:
        """
        Send a login request to the server.

        Args:
            login: User login.
            password: User password.

        Returns:
            Reply: A promise for the login reply.
        """

        login_msg = self.__protobuf_types.createType('motorcortex.LoginMsg')
        login_msg.password = password
        login_msg.login = login

        return self.send(self.__protobuf_types.encode(login_msg))

    def connectionState(self) -> ConnectionState:
        """
        Get the current connection state.

        Returns:
            ConnectionState: The current state.
        """
        return self.__connection_state

    def getParameterTreeHash(self) -> Reply:
        """
        Request a parameter tree hash from the server.

        Returns:
            Reply: A promise for the parameter tree hash.
        """

        # getting and instantiating data type from the loaded dict
        param_tree_hash_msg = self.__protobuf_types.createType('motorcortex.GetParameterTreeHashMsg')

        # encoding and sending data
        return self.send(self.__protobuf_types.encode(param_tree_hash_msg))

    def getParameterTree(self) -> Reply:
        """
        Request a parameter tree from the server.

        Returns:
            Reply: A promise for the parameter tree.
        """

        return Reply(self.__pool.submit(self.__getParameterTree,
                                        self.getParameterTreeHash(), self.__protobuf_types, self.__socket))

    def save(self, path: str, file_name: str) -> Reply:
        """
        Request the server to save a parameter tree to a file.

        Args:
            path: Path to save the file.
            file_name: Name of the file.

        Returns:
            Reply: A promise for the save operation.
        """

        param_save_msg = self.__protobuf_types.createType('motorcortex.SaveMsg')
        param_save_msg.path = path
        param_save_msg.file_name = file_name

        return self.send(self.__protobuf_types.encode(param_save_msg))

    def setParameter(self, path: str, value: Any, type_name: Optional[str] = None, offset: int = 0,
                     length: int = 0) -> Reply:
        """
        Set a new value for a parameter.

        Args:
            path: Parameter path.
            value: New value.
            type_name: Type name (optional).
            offset: Offset in array (optional).
            length: Number of elements to update (optional).

        Returns:
            Reply: A promise for the set operation.
        """

        if (offset == 0) and (length == 0):
            return self.send(self.__protobuf_types.encode(self.__buildSetParameterMsg(path, value,
                                                                                      type_name, self.__protobuf_types,
                                                                                      self.__parameter_tree)))
        else:
            return self.send(
                self.__protobuf_types.encode(self.__buildSetParameterWithOffsetMsg(offset, length, path, value,
                                                                                   type_name, self.__protobuf_types,
                                                                                   self.__parameter_tree)))

    def setParameterList(self, param_list: List[dict]):
        """
        Set new values to a parameter list

        Args:
                param_list([{'path'-`str`,'value'-`any`, 'offset', 'length'}]): a list of the parameters which values update

        Returns:
            Reply(StatusMsg): A Promise, which resolves when parameters from the list are updated,
            otherwise fails.

        Examples:
                >>>  req.setParameterList([
                >>>   {'path': 'root/Control/generator/enable', 'value': False},
                >>>   {'path': 'root/Control/generator/amplitude', 'value': 1.4}])
                >>>   {'path': 'root/Control/myArray6', 'value': [1.4, 1.5], 'offset': 1, 'length': 2}])
        """

        # instantiating message type
        set_param_list_msg = self.__protobuf_types.createType("motorcortex.SetParameterListMsg")
        # filling with sub messages
        for param in param_list:
            type_name = param.get("type_name", None)
            offset = param.get("offset", 0)
            length = param.get("length", 0)
            if (offset == 0) and (length == 0):
                set_param_list_msg.params.extend([self.__buildSetParameterMsg(param["path"], param["value"],
                                                                              type_name, self.__protobuf_types,
                                                                              self.__parameter_tree)])
            else:
                set_param_list_msg.params.extend(
                    [self.__buildSetParameterWithOffsetMsg(offset, length, param["path"],
                                                           param["value"],
                                                           type_name, self.__protobuf_types,
                                                           self.__parameter_tree)])

        # encoding and sending data
        return self.send(self.__protobuf_types.encode(set_param_list_msg))

    def getParameter(self, path: str) -> Reply:
        """
        Request a parameter value and description from the server.

        Args:
            path: Parameter path.

        Returns:
            Reply: A promise for the parameter value.
        """

        return self.send(self.__protobuf_types.encode(self.__buildGetParameterMsg(path, self.__protobuf_types)))

    def getParameterList(self, path_list: List[str]) -> Reply:
        """
        Request values and descriptions for a list of parameters.

        Args:
            path_list: List of parameter paths.

        Returns:
            Reply: A promise for the parameter list.
        """

        # instantiating message type
        get_param_list_msg = self.__protobuf_types.createType('motorcortex.GetParameterListMsg')
        # filling with sub messages
        for path in path_list:
            get_param_list_msg.params.extend([self.__buildGetParameterMsg(path, self.__protobuf_types)])

        # encoding and sending data
        return self.send(self.__protobuf_types.encode(get_param_list_msg))

    def overwriteParameter(self, path: str, value: Any, force_activate: bool = False,
                           type_name: Optional[str] = None) -> Reply:
        """
        Overwrite a parameter value and optionally force it to stay active.

        Args:
            path: Parameter path.
            value: New value.
            force_activate: Force value to stay active.
            type_name: Type name (optional).

        Returns:
            Reply: A promise for the overwrite operation.
        """

        return self.send(self.__protobuf_types.encode(self.__buildOverwriteParameterMsg(path, value, force_activate,
                                                                                        type_name,
                                                                                        self.__protobuf_types,
                                                                                        self.__parameter_tree)))

    def releaseParameter(self, path: str) -> Reply:
        """
        Release the overwrite operation for a parameter.

        Args:
            path: Parameter path.

        Returns:
            Reply: A promise for the release operation.
        """

        return self.send(self.__protobuf_types.encode(self.__buildReleaseParameterMsg(path, self.__protobuf_types)))

    def createGroup(self, path_list: List[str], group_alias: str, frq_divider: int = 1) -> Reply:
        """
        Create a subscription group for a list of parameters.

        Args:
            path_list: List of parameter paths.
            group_alias: Group alias.
            frq_divider: Frequency divider.

        Returns:
            Reply: A promise for the group creation.
        """

        # instantiating message type
        create_group_msg = self.__protobuf_types.createType('motorcortex.CreateGroupMsg')
        create_group_msg.alias = group_alias
        create_group_msg.paths.extend(path_list if type(path_list) is list else [path_list])
        create_group_msg.frq_divider = frq_divider if frq_divider > 1 else 1
        # encoding and sending data
        return self.send(self.__protobuf_types.encode(create_group_msg))

    def removeGroup(self, group_alias: str) -> Reply:
        """
        Unsubscribe from a group.

        Args:
            group_alias: Group alias.

        Returns:
            Reply: A promise for the unsubscribe operation.
        """

        # instantiating message type
        remove_group_msg = self.__protobuf_types.createType('motorcortex.RemoveGroupMsg')
        remove_group_msg.alias = group_alias
        # encoding and sending data
        return self.send(self.__protobuf_types.encode(remove_group_msg))

    @staticmethod
    def __buildSetParameterMsg(
            path: str,
            value: Any,
            type_name: Optional[str],
            protobuf_types: MessageTypes,
            parameter_tree: ParameterTree
    ) -> Any:
        """
        Builds a SetParameterMsg for a single parameter.

        Args:
            path: Parameter path.
            value: New value to set.
            type_name: Optional type name for the parameter.
            protobuf_types: Protobuf type definitions.
            parameter_tree: ParameterTree instance.

        Returns:
            The constructed SetParameterMsg.
        """
        param_value = None
        if not type_name:
            type_id = parameter_tree.getDataType(path)
            if type_id:
                param_value = protobuf_types.getTypeByHash(type_id)
        else:
            param_value = protobuf_types.createType(type_name)

        if not param_value:
            logger.error("Failed to find encoder for the path: %s type: %s" % (path, type_name))

        # creating type instance
        set_param_msg = protobuf_types.createType("motorcortex.SetParameterMsg")
        set_param_msg.path = path
        # encoding parameter value
        set_param_msg.value = param_value.encode(value)

        return set_param_msg

    @staticmethod
    def __buildSetParameterWithOffsetMsg(
            offset: int,
            length: int,
            path: str,
            value: Any,
            type_name: Optional[str],
            protobuf_types: MessageTypes,
            parameter_tree: ParameterTree
    ) -> Any:
        """
        Builds a SetParameterMsg for a parameter with offset and length.

        Args:
            offset: Offset in the parameter array.
            length: Number of elements to update.
            path: Parameter path.
            value: New value(s) to set.
            type_name: Optional type name for the parameter.
            protobuf_types: Protobuf type definitions.
            parameter_tree: ParameterTree instance.

        Returns:
            The constructed SetParameterMsg.
        """
        param_value = None
        if not type_name:
            type_id = parameter_tree.getDataType(path)
            if type_id:
                param_value = protobuf_types.getTypeByHash(type_id)
        else:
            param_value = protobuf_types.createType(type_name)

        if not param_value:
            logger.error("Failed to find encoder for the path: %s type: %s" % (path, type_name))

        # check an offset
        if offset < 0:
            offset = 0

        # check length, if == 0 assign length of the value
        if length < 0:
            length = 0
        if length == 0:
            if hasattr(value, '__len__'):
                length = len(value)
            else:
                length = 1

        # creating type instance
        set_param_msg = protobuf_types.createType("motorcortex.SetParameterMsg")
        set_param_msg.offset.type = 1
        set_param_msg.offset.offset = offset
        set_param_msg.offset.length = length
        set_param_msg.path = path
        # encoding parameter value
        set_param_msg.value = param_value.encode(value)

        return set_param_msg

    @staticmethod
    def __buildGetParameterMsg(
            path: str,
            protobuf_types: MessageTypes
    ) -> Any:
        """
        Builds a GetParameterMsg for a single parameter.

        Args:
            path: Parameter path.
            protobuf_types: Protobuf type definitions.

        Returns:
            The constructed GetParameterMsg.
        """
        get_param_msg = protobuf_types.createType('motorcortex.GetParameterMsg')
        get_param_msg.path = path

        return get_param_msg

    @staticmethod
    def __buildOverwriteParameterMsg(
            path: str,
            value: Any,
            activate: bool,
            type_name: Optional[str],
            protobuf_types: MessageTypes,
            parameter_tree: ParameterTree
    ) -> Any:
        """
        Builds an OverwriteParameterMsg for a parameter.

        Args:
            path: Parameter path.
            value: New value to overwrite.
            activate: Whether to force the value to stay active.
            type_name: Optional type name for the parameter.
            protobuf_types: Protobuf type definitions.
            parameter_tree: ParameterTree instance.

        Returns:
            The constructed OverwriteParameterMsg.
        """
        param_value = None
        if not type_name:
            type_id = parameter_tree.getDataType(path)
            if type_id:
                param_value = protobuf_types.getTypeByHash(type_id)
        else:
            param_value = protobuf_types.createType(type_name)

        if not param_value:
            logger.error("Failed to find encoder for the path: %s type: %s" % (path, type_name))

        # creating type instance
        overwrite_param_msg = protobuf_types.createType("motorcortex.OverwriteParameterMsg")
        overwrite_param_msg.path = path
        overwrite_param_msg.activate = activate
        # encoding parameter value
        overwrite_param_msg.value = param_value.encode(value)

        return overwrite_param_msg

    @staticmethod
    def __buildReleaseParameterMsg(
            path: str,
            protobuf_types: MessageTypes
    ) -> Any:
        """
        Builds a ReleaseParameterMsg for a parameter.

        Args:
            path: Parameter path.
            protobuf_types: Protobuf type definitions.

        Returns:
            The constructed ReleaseParameterMsg.
        """
        release_param_msg = protobuf_types.createType('motorcortex.ReleaseParameterMsg')
        release_param_msg.path = path

        return release_param_msg

    @staticmethod
    def parse(
            conn_timeout_ms: int = 0,
            timeout_ms: Optional[int] = None,
            recv_timeout_ms: Optional[int] = None,
            certificate: Optional[str] = None,
            login: Optional[str] = None,
            password: Optional[str] = None,
            state_update: Optional[Callable] = None,
            **kwargs
    ) -> tuple[int, Optional[int], Optional[str], Optional[Callable]]:
        """
        Parses connection parameters for the connect method.

        Args:
            conn_timeout_ms: Connection timeout in milliseconds.
            timeout_ms: Alternative timeout in milliseconds.
            recv_timeout_ms: Receive timeout in milliseconds.
            certificate: Path to the TLS certificate.
            login: Optional login name.
            password: Optional password.
            state_update: Optional state update callback.
        Returns:
            Tuple of (conn_timeout_ms, recv_timeout_ms, certificate, state_update.
        """
        if timeout_ms and not conn_timeout_ms:
            conn_timeout_ms = timeout_ms

        return conn_timeout_ms, recv_timeout_ms, certificate, state_update

    @staticmethod
    def __send(
            req: Req0,
            encoded_msg: Any,
            protobuf_types: Optional[MessageTypes]
    ) -> Any:
        """
        Sends an encoded message and receives the reply.

        Args:
            req: The Req0 socket.
            encoded_msg: Encoded protobuf message.
            protobuf_types: Protobuf type definitions, or None to skip decoding.

        Returns:
            Decoded reply or raw buffer.
        """
        ctx = req.new_context()
        try:
            ctx.send(encoded_msg)
            buffer = ctx.recv()
            if buffer:
                if protobuf_types:
                    return protobuf_types.decode(buffer)
                else:
                    return buffer
        except Exception as e:
            logger.error(f"[__SEND] Error during send/recv: {type(e).__name__}: {e}")
            raise
        finally:
            ctx.close()  # Always close the context!

        return None

    @staticmethod
    def waitForConnection(
            event: Event,
            timeout_sec: float,
            is_connected_fn: Optional[Callable[[], bool]] = None
    ) -> bool:
        """
        Waits for the connection event or times out.

        Args:
            event: Event to wait on.
            timeout_sec: Timeout in seconds.
            is_connected_fn: Optional function to check connection status.

        Returns:
            True if connection is established, raises on failure or timeout.
        """
        logger.debug(f"[WAIT] waitForConnection started with timeout={timeout_sec}s")

        # Wait for the condition to be set or timeout
        if timeout_sec <= 0:
            logger.debug("[WAIT] Waiting indefinitely for connection event")
            result = event.wait()
        else:
            logger.debug(f"[WAIT] Waiting up to {timeout_sec}s for connection event")
            result = event.wait(timeout_sec)

        if not result:
            logger.error(f"[WAIT] Connection timeout after {timeout_sec}s - no event received")
            raise TimeoutError(f"Connection timeout after {timeout_sec}s")

        logger.debug("[WAIT] Event received - checking connection status")

        # Check if we actually connected (event could be set by close() or failure)
        if is_connected_fn:
            connected = is_connected_fn()
            logger.debug(f"[WAIT] Connection status check: connected={connected}")
            if not connected:
                logger.error("[WAIT] Event was set but connection failed or was closed")
                raise ConnectionError("Connection failed or was closed before completion")
        else:
            logger.debug("[WAIT] No connection status check function provided")

        logger.debug("[WAIT] Connection successfully established")
        return True

    @staticmethod
    def __getParameterTree(
            hash_reply: Reply,
            protobuf_types: MessageTypes,
            socket: Req0
    ) -> Any:
        """
        Retrieves the parameter tree, using cache if available.

        Args:
            hash_reply: Reply object containing the parameter tree hash.
            protobuf_types: Protobuf type definitions.
            socket: Req0 socket.

        Returns:
            The parameter tree object.
        """
        tree_hash = hash_reply.get()
        path = os.sep.join([tempfile.gettempdir(), "mcx-python-pt-" + str(tree_hash.hash)])
        tree = Request.loadParameterTreeFile(path, protobuf_types)
        if tree:
            logger.debug('Found parameter tree in the cache')
            return tree
        else:
            logger.debug('Failed to find parameter tree in the cache')

        # getting and instantiating data type from the loaded dict
        param_tree_msg = protobuf_types.createType('motorcortex.GetParameterTreeMsg')
        handle = Request.__send(socket, protobuf_types.encode(param_tree_msg), protobuf_types)

        # encoding and sending data
        return Request.saveParameterTreeFile(path, handle)

    @staticmethod
    def saveParameterTreeFile(
            path: str,
            parameter_tree: ParameterTree
    ) -> ParameterTree:
        """
        Saves the parameter tree to a file in the cache.

        Args:
            path: File path to save the parameter tree.
            parameter_tree: ParameterTree instance.

        Returns:
            The saved ParameterTree instance.
        """
        logger.debug('Saved parameter tree to the cache')
        json_data = {}
        base64_data = base64.b64encode(parameter_tree.SerializeToString())
        json_data['md5'] = hashlib.md5(base64_data).hexdigest()
        json_data['data'] = base64_data.decode('utf-8')

        with open(path, "w") as outfile:
            outfile.write(json.dumps(json_data))

        return parameter_tree

    @staticmethod
    def loadParameterTreeFile(
            path: str,
            protobuf_types: MessageTypes
    ) -> Optional[ParameterTree]:
        """
        Loads the parameter tree from a cached file if available and valid.

        Args:
            path: File path to load the parameter tree from.
            protobuf_types: Protobuf type definitions.

        Returns:
            The loaded ParameterTree instance, or None if not found/invalid.
        """
        logger.debug('Loaded parameter tree from the cache')
        param_tree_hash_msg = None
        if os.path.exists(path):
            with open(path, "r") as outfile:
                json_data = json.load(outfile)

            if json_data:
                if "md5" in json_data and "data" in json_data:
                    if hashlib.md5(json_data['data'].encode()).hexdigest() == json_data['md5']:
                        param_tree_hash_msg = protobuf_types.createType('motorcortex.ParameterTreeMsg')
                        tree_raw = base64.b64decode(json_data['data'])
                        param_tree_hash_msg.ParseFromString(tree_raw)

        return param_tree_hash_msg
