class Timespec:
    """
    Represents a timestamp with seconds and nanoseconds.

    Provides conversion properties for seconds, milliseconds, microseconds, and nanoseconds.
    Prefer using this class for new code instead of standalone functions.
    """

    def __init__(self, sec: int, nsec: int):
        self._sec = sec
        self._nsec = nsec

    def __eq__(self, other: object) -> bool:
        """
        Compare two Timespec objects for equality.

        Args:
            other (object): Another Timespec instance.

        Returns:
            bool: True if both timestamps are equal, False otherwise.
        """
        if not isinstance(other, Timespec):
            return NotImplemented
        return self._sec == other._sec and self._nsec == other._nsec

    def __lt__(self, other: "Timespec") -> bool:
        return (self._sec, self._nsec) < (other._sec, other._nsec)

    def __le__(self, other: "Timespec") -> bool:
        return (self._sec, self._nsec) <= (other._sec, other._nsec)

    def __add__(self, other: "Timespec") -> "Timespec":
        total_nsec = self._nsec + other._nsec
        total_sec = self._sec + other._sec + total_nsec // 1_000_000_000
        total_nsec = total_nsec % 1_000_000_000
        return Timespec(total_sec, total_nsec)

    def __str__(self) -> str:
        return f"{self._sec}s {self._nsec}ns"

    @classmethod
    def from_iterable(cls, iterable):
        """
        Create a Timespec instance from an iterable containing seconds and nanoseconds.
        
        Args:
            iterable: An iterable with two elements: (sec, nsec).
            
        Returns:
            Timespec: A new Timespec instance.
        """
        sec, nsec = iterable
        return cls(sec, nsec)

    @staticmethod
    def from_msec(msec: float) -> "Timespec":
        """
        Create a Timespec instance from milliseconds.
        
        Args:
            msec (float): Time in milliseconds.
            
        Returns:
            Timespec: A new Timespec instance.
        """
        sec = int(msec // 1000)
        nsec = int((msec % 1000) * 1_000_000)
        return Timespec(sec, nsec)

    @property
    def sec(self) -> float:
        """
        Returns the timestamp as seconds (float).
        """
        return self._sec + self._nsec / 1_000_000_000

    @property
    def msec(self) -> float:
        """
        Returns the timestamp as milliseconds (float).
        """
        return self._sec * 1000 + self._nsec / 1_000_000

    @property
    def usec(self) -> float:
        """
        Returns the timestamp as microseconds (float).
        """
        return self._sec * 1_000_000 + self._nsec / 1_000

    @property
    def nsec(self) -> int:
        """
        Returns the timestamp as nanoseconds (int).
        """
        return self._sec * 1_000_000_000 + self._nsec

    def to_tuple(self) -> tuple[int, int]:
        return (self._sec, self._nsec)


def compare_timespec(timestamp1: Timespec, timestamp2: Timespec) -> bool:
    """
    Compare two timestamps.

    Args:
        timestamp1 (timespec): First timestamp to compare.
        timestamp2 (timespec): Second timestamp to compare.

    Returns:
        bool: True if both timestamps are equal, False otherwise.

    Note:
        For new code, use Timespec.__eq__ instead.
    """
    return True if (timestamp1.sec == timestamp2.sec) and (timestamp1.nsec == timestamp2.nsec) else False


def timespec_to_sec(timestamp: Timespec) -> float:
    """
    Convert a timestamp to seconds.

    Args:
        timestamp (timespec): Timestamp to convert.

    Returns:
        float: Time in seconds.

    Note:
        For new code, use Timespec.sec_value instead.
    """
    return timestamp.sec + timestamp.nsec / 1000000000.0


def timespec_to_msec(timestamp: Timespec) -> float:
    """
    Convert a timestamp to milliseconds.

    Args:
        timestamp (timespec): Timestamp to convert.

    Returns:
        float: Time in milliseconds.

    Note:
        For new code, use Timespec.msec instead.
    """
    return timestamp.sec * 1000 + timestamp.nsec / 1000000


def timespec_to_usec(timestamp: Timespec) -> float:
    """
    Convert a timestamp to microseconds.

    Args:
        timestamp (timespec): Timestamp to convert.

    Returns:
        float: Time in microseconds.

    Note:
        For new code, use Timespec.usec instead.
    """
    return timestamp.sec * 1000000 + timestamp.nsec / 1000


def timespec_to_nsec(timestamp: Timespec) -> int:
    """
    Convert a timestamp to nanoseconds.

    Args:
        timestamp (timespec): Timestamp to convert.

    Returns:
        int: Time in nanoseconds.

    Note:
        For new code, use Timespec.nsec_value instead.
    """
    return timestamp.sec * 1000000000 + timestamp.nsec
