#!/usr/bin/python3

#
#   Developer: Alexey Zakharov (alexey.zakharov@vectioneer.com)
#   All rights reserved. Copyright (c) 2024 - 2026 VECTIONEER.
#

from .robot_command import *


def isEqual(list_of_values, objective_value):
    """Checks if all the values in the list are equal to the objective value.

        Args:
            list_of_values(list(any)): list of the values to compare
            objective_value(any): an objective value

        Returns:
            bool: True if all values are equal to objective, False otherwise

    """
    return all(objective_value == value for value in list_of_values)


class MultiRobotCommand(object):
    """Class represents a state machine of the multiple robot arms.

        Args:
            req(motorcortex.Request): reference to a Request instance
            motorcortex_types(motorcortex.MessageTypes): reference to a MessageTypes instance
            systems_id(list(int)): a list of systems id

    """

    def __init__(self, req, motorcortex_types, systems_id=None):
        self.__Motorcortex = motorcortex_types.getNamespace("motorcortex")
        self.__motorcortex_types = motorcortex_types
        self.__req = req
        self.__kinematics_update_counter = 0
        self.__robot_command_list = {}
        if not systems_id:
            self.__systems_id = [None]
            self.__robot_command_list.update({None: RobotCommand(req, motorcortex_types)})
        else:
            self.__systems_id = systems_id
            for system_id in self.__systems_id:
                self.__robot_command_list.update({system_id: RobotCommand(req, motorcortex_types, system_id)})

    def __addSystem(self, name, system):
        if not system:
            return name
        return f"{name}{system:02d}"

    def play(self, systems_id=None, wait_time=1.0):
        """Plays the program from the current trajectory position for multiple robots.

        If any robot has deviated from the trajectory, a guard violation will occur
        and that robot must return to the trajectory start position using moveToStart()
        before motion can continue.

            Args:
                systems_id(list(int)): optional list of systems id, subset of object defined list
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                list(InterpreterStates): actual states of the program interpreters

            See Also:
                moveToStart: Use to return robots to trajectory start after guard violation.
                playFromActual: Use to continue from the current position without returning to trajectory.

        """

        if not systems_id:
            systems_id = self.__systems_id

        paramList = []
        for system_id in systems_id:
            paramList.append({"path": f'root/{self.__addSystem("MotionInterpreter", system_id)}/interpreterCommand',
                              "value": InterpreterEvents.PLAY_PROGRAM_E.value})

        self.__req.setParameterList(paramList)
        time.sleep(wait_time)

        return self.getState(systems_id)

    def playFromActual(self, systems_id=None, wait_time=1.0):
        """Plays the program from the robots' current actual positions.

        Unlike play(), this method allows robots to continue from their current positions
        even if they have deviated from the original trajectory. The interpreter automatically
        moves each robot back to the trajectory and resumes program execution without
        requiring manual intervention (no button hold needed).

            Args:
                systems_id(list(int)): optional list of systems id, subset of object defined list
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                list(InterpreterStates): actual states of the program interpreters

            See Also:
                play: Use when robots are on the trajectory or after manual moveToStart.
                moveToStart: Manual alternative requiring button hold to return to trajectory.

        """

        if not systems_id:
            systems_id = self.__systems_id

        paramList = []
        for system_id in systems_id:
            paramList.append({"path": f'root/{self.__addSystem("MotionInterpreter", system_id)}/interpreterCommand',
                              "value": InterpreterEvents.PLAY_PROGRAM_FROM_ACTUAL_E.value})

        self.__req.setParameterList(paramList)
        time.sleep(wait_time)

        return self.getState(systems_id)

    def pause(self, systems_id=None, wait_time=1.0):
        """Pause the program for multiple robots.

        Stops motion while preserving the current program position. Use play() or
        playFromActual() to resume execution.

            Args:
                systems_id(list(int)): optional list of systems id, subset of object defined list
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                list(InterpreterStates): actual states of the program interpreters

            See Also:
                play: Resume from the trajectory position (may require moveToStart if deviated).
                playFromActual: Resume from the current actual position.
                stop: Stop the program and return to the beginning.

        """

        if not systems_id:
            systems_id = self.__systems_id

        param_list = []
        for system_id in systems_id:
            param_list.append({"path": f'root/{self.__addSystem("MotionInterpreter", system_id)}/interpreterCommand',
                               "value": InterpreterEvents.PAUSE_PROGRAM_E.value})

        self.__req.setParameterList(param_list)
        time.sleep(wait_time)

        return self.getState(systems_id)

    def getState(self, systems_id=None):
        """Get the current interpreter states for multiple robots.

            Args:
                systems_id(list(int)): optional list of systems id, subset of object defined list

            Returns:
                list(InterpreterStates): actual states of the interpreters

        """

        if not systems_id:
            systems_id = self.__systems_id

        param_list = []
        for system_id in systems_id:
            param_list.append(f'root/{self.__addSystem("MotionInterpreter", system_id)}/actualStateOut')

        params = self.__req.getParameterList(param_list).get()

        result = []
        for param in params.params:
            result.append(param.value[0])

        return result

    def engage(self):
        """Switch robot to Engage state.

            Returns:
                bool: True if the operation is completed, False if failed

        """
        return self.__robot_command_list[self.__systems_id[0]].engage()

    def stop(self, systems_id=None, wait_time=1.0):
        """Stop the program for multiple robots.

        Stops motion and resets program execution to the beginning. Unlike pause(),
        the program cannot be resumed from the current position.

            Args:
                systems_id(list(int)): optional list of systems id, subset of object defined list
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                list(InterpreterStates): actual states of the program interpreters

            See Also:
                pause: Pause without resetting the program position.
                reset: Stop and clear the interpreter buffer.

        """

        if not systems_id:
            systems_id = self.__systems_id

        param_list = []
        for system_id in systems_id:
            param_list.append({"path": f'root/{self.__addSystem("MotionInterpreter", system_id)}/interpreterCommand',
                               "value": InterpreterEvents.STOP_PROGRAM_E.value})

        self.__req.setParameterList(param_list).get()

        time.sleep(wait_time)
        return self.getState(systems_id)

    def reset(self, systems_id=None, wait_time=1.0):
        """Stop the program and clear the interpreter buffer for multiple robots.

        Performs a full reset of the interpreters, stopping any motion and clearing
        all buffered commands. Use this to completely restart program execution.

            Args:
                systems_id(list(int)): optional list of systems id, subset of object defined list
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                list(InterpreterStates): actual states of the program interpreters

            See Also:
                stop: Stop without clearing the buffer.

        """

        if not systems_id:
            systems_id = self.__systems_id

        param_list = []
        for system_id in systems_id:
            param_list.append({"path": f'root/{self.__addSystem("MotionInterpreter", system_id)}/interpreterCommand',
                               "value": InterpreterEvents.RESET_INTERPRETER_E.value})

        self.__req.setParameterList(param_list).get()

        time.sleep(wait_time)
        return self.getState(systems_id)

    def moveToStart(self, timeout_s, systems_id=None):
        """Move arms to the start of the trajectory for multiple robots.

        Used to return robots to the trajectory start position after a guard violation
        (when a robot has deviated from the planned trajectory). This is required before
        play() can resume motion.

            Args:
                timeout_s(float): timeout in seconds
                systems_id(list(int)): optional list of systems id, subset of object defined list

            Returns:
                list(InterpreterStates): actual states of the program interpreters

            Raises:
                Exception: if any robot fails to move to start position within timeout

            See Also:
                play: Resumes program execution after robots are at trajectory start.
                playFromActual: Alternative that continues from current position without moveToStart.

        """
        if not systems_id:
            systems_id = self.__systems_id

        for system_id in systems_id:
            robot_system = self.__robot_command_list[system_id]
            if robot_system.play() is InterpreterStates.MOTION_NOT_ALLOWED_S.value:
                print('Robot is not at a start position, moving to the start')
                if robot_system.moveToStart(timeout_s):
                    print('Robot is at the start position')
                    robot_system.pause()
                else:
                    raise Exception('Failed to move to the start position')

        return self.getState()

    def system(self, system):
        """Get the RobotCommand instance for a specific system.

            Args:
                system(int): id of the system

            Returns:
                RobotCommand: robot command instance for specified system id

        """
        return self.__robot_command_list[system]