#!/usr/bin/python3

#
#   Developer: Alexey Zakharov (alexey.zakharov@vectioneer.com)
#   All rights reserved. Copyright (c) 2017 - 2026 VECTIONEER.
#

import time

from .system_defs import States, StateEvents, ModeCommands, Modes, InterpreterEvents, InterpreterStates, \
    MotionGeneratorStates


class RobotCommand(object):
    """Class represents a state machine of the robot arm.

        Args:
            req(motorcortex.Request): reference to a Request instance
            motorcortex_types(motorcortex.MessageTypes): reference to a MessageTypes instance
            system_id(int): system id, for example, for the dual-arm robot

    """

    def __init__(self, req, motorcortex_types, system_id=None):
        self.__Motorcortex = motorcortex_types.getNamespace("motorcortex")
        self.__motorcortex_types = motorcortex_types
        self.__req = req
        self.__kinematics_update_counter = 0
        self.__system_id = system_id

    @staticmethod
    def __addSystem(name, system_id):
        if not system_id:
            return name
        return f"{name}{system_id:02d}"

    def off(self):
        """Switch robot to Off state.

            Returns:
                bool: True if the operation is completed, False if failed

        """

        self.__req.setParameter("root/Logic/stateCommand", StateEvents.GOTO_OFF_E.value).get()
        actual_state = self.__req.getParameter("root/Logic/state").get().value[0]

        while actual_state != States.OFF_S.value:
            actual_state = self.__req.getParameter("root/Logic/state").get().value[0]
            time.sleep(0.1)

        return True

    def disengage(self):
        """Switch robot to Disengage state.

            Returns:
                bool: True if the operation is completed, False if failed

        """

        self.__req.setParameter("root/Logic/stateCommand", StateEvents.GOTO_DISENGAGED_E.value).get()
        actual_state = self.__req.getParameter("root/Logic/state").get().value[0]

        while actual_state != States.DISENGAGED_S.value:
            actual_state = self.__req.getParameter("root/Logic/state").get().value[0]
            time.sleep(0.1)

        return True

    def engage(self):
        """Switch robot to Engage state.

            Returns:
                bool: True if the operation is completed, False if failed

        """

        self.__req.setParameter("root/Logic/stateCommand", StateEvents.GOTO_ENGAGED_E.value).get()
        actual_state = self.__req.getParameter("root/Logic/state").get().value[0]

        while actual_state != States.ENGAGED_S.value:
            actual_state = self.__req.getParameter("root/Logic/state").get().value[0]
            time.sleep(0.1)

        return True

    def acknowledge(self, timeout_s=20.0):
        """Acknowledge the errors and warnings. If the robot is in EStop state brings it to the Off state,
        if the robot is in ForceDisengaged/ForceIdle state, it brings it to the Disengaged/Idle state.

            Returns:
                bool: True if the operation is completed, False if failed
        """

        actual_state = self.__req.getParameter("root/Logic/state").get().value[0]
        self.__req.setParameter("root/Logic/stateCommand", StateEvents.ACKNOWLEDGE_ERROR.value).get()
        if actual_state == States.ESTOP_OFF_S.value or actual_state == States.RESET_ESTOP_T.value:
            return self.__waitForTheParameterValuet("root/Logic/state", States.OFF_S.value, timeout_s)
        elif actual_state == States.FORCEDISENGAGED_S.value or actual_state == States.RESET_FORCEDDISENGAGE_T.value:
            return self.__waitForTheParameterValuet("root/Logic/state", States.DISENGAGED_S.value, timeout_s)

        return True

    def __waitForTheParameterValuet(self, parameter_name, target_value, timeout_s):
        timer = 0
        actual_vale = None
        while timer <= timeout_s:
            actual_vale = self.__req.getParameter(parameter_name).get().value[0]
            if actual_vale == target_value:
                return True
            time.sleep(0.1)
            timer += 0.1
        return False

    def manualCartMode(self):
        """Switch the robot to a manual Cartesian motion.

            Returns:
                bool: True if the operation is completed, False if failed

        """

        actual_mode = self.__req.getParameter("root/Logic/mode").get().value[0]
        while actual_mode != Modes.MANUAL_CART_MODE_M.value:
            self.__req.setParameter("root/Logic/modeCommand", ModeCommands.GOTO_MANUAL_CART_MODE_E.value).get()
            time.sleep(0.1)
            actual_mode = self.__req.getParameter("root/Logic/mode").get().value[0]
        return True

    def manualJointMode(self):
        """Switch the robot to a manual joint motion.

            Returns:
                bool: True if the operation is completed, False if failed

        """

        actual_mode = self.__req.getParameter("root/Logic/mode").get().value[0]
        while actual_mode != Modes.MANUAL_JOINT_MODE_M.value:
            self.__req.setParameter("root/Logic/modeCommand", ModeCommands.GOTO_MANUAL_JOINT_MODE_E.value).get()
            time.sleep(0.1)
            actual_mode = self.__req.getParameter("root/Logic/mode").get().value[0]
        return True

    def semiAutoMode(self):
        """Switch robot to semi-auto mode. Semi-auto moves the arm to the target
        when the user holds a button. Semi-auto is active, for example, during
        the move to the start of the program.

            Returns:
                bool: True if the operation is completed, False if failed

        """

        actual_mode = self.__req.getParameter("root/Logic/mode").get().value[0]
        while actual_mode != Modes.SEMI_AUTO_M.value:
            self.__req.setParameter("root/Logic/modeCommand", ModeCommands.GOTO_SEMI_AUTO_E.value).get()
            time.sleep(0.1)
            actual_mode = self.__req.getParameter("root/Logic/mode").get().value[0]
        return True

    def toolTipOffset(self, tool_tip_offset):
        """Update tool-tip offset. Robot should be manual joint mode.

            Args:
                tool_tip_offset(list(double)): new tool tip offset in the Cartesian frame of the last segment, rotation
                is defined in Euler ZYX angles.

            Returns:
                bool: True if the operation is completed, False if failed

        """

        if self.manualJointMode():
            fade_time = self.__req.getParameter(
                f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/mechanism/tooltipFader/fadeTime").get()
            self.__req.setParameterList([
                {
                    "path": f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/mechanism/ttOffset",
                    "value": tool_tip_offset
                },
                {
                    "path": f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/mechanism/update",
                    "value": self.__kinematics_update_counter
                }]).get()
            self.__kinematics_update_counter = self.__kinematics_update_counter + 1
            time.sleep(fade_time.value[0])
            return True

        return False

    def moveToPoint(self, target_joint_coord_rad, v_max=0.5, a_max=1.0):
        """Move the arm to a specified pose in a joint space.

            Args:
                target_joint_coord_rad(list(double)): target pose in joint space, rad
                v_max(float): maximum velocity, rad/s
                a_max(float): maximum acceleration, rad/s^2

            Returns:
                bool: True if the operation is completed, False if failed

        """

        if self.semiAutoMode():
            self.__req.setParameterList([
                {
                    "path": f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/semiAutoMotionGenerator/maxAcc",
                    "value": a_max
                },
                {
                    "path": f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/semiAutoMotionGenerator/maxVel",
                    "value": v_max
                },
                {
                    "path": f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/semiAutoMotionGenerator/target",
                    "value": target_joint_coord_rad
                }]).get()

            while True:
                self.__req.setParameter(
                    f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/activateSemiAuto",
                    True).get()
                motion_state = \
                    self.__req.getParameter(
                        f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/semiAutoMotionGenerator/motionGenStateOut").get().value[
                        0]
                if motion_state != MotionGeneratorStates.RUNNING_S.value:
                    self.__req.setParameter(
                        f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/activateSemiAuto",
                        False).get()
                    return True
                time.sleep(0.1)

        return False

    def moveToStart(self, timeout_s):
        """Move the arm to the start of the trajectory.

        Used to return the robot to the trajectory start position after a guard violation
        (when the robot has deviated from the planned trajectory). This is required before
        play() can resume motion.

            Args:
                timeout_s(float): timeout in seconds

            Returns:
                bool: True if the operation is completed, False if timeout

            See Also:
                play: Resumes program execution after the robot is at trajectory start.
                playFromActual: Alternative that continues from current position without moveToStart.

        """

        self.__req.setParameter(
            f"root/{self.__addSystem('MotionInterpreter', self.__system_id)}/interpreterCommand",
            InterpreterEvents.MOVE_TO_START.value).get()
        sleep_s = 0.1
        max_number_of_tries = timeout_s / sleep_s
        counter = 0
        while True:
            counter += 1
            self.__req.setParameter(
                f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/activateMoveToStart",
                1).get()
            interpreter_stat = self.getState()
            if interpreter_stat == InterpreterStates.PROGRAM_PAUSE_S.value:
                self.__req.setParameter(
                    f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/activateMoveToStart",
                    0).get()
                return True
            if counter > max_number_of_tries:
                self.__req.setParameter(
                    f"root/{self.__addSystem('ManipulatorControl', self.__system_id)}/activateMoveToStart",
                    0).get()
                return False
            time.sleep(sleep_s)

    def play(self, wait_time=0.1):
        """Plays the program from the current trajectory position.

        If the robot has deviated from the trajectory, a guard violation will occur
        and the robot must return to the trajectory start position using moveToStart()
        before motion can continue.

            Args:
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                InterpreterStates: actual state of the program interpreter

            See Also:
                moveToStart: Use to return robot to trajectory start after guard violation.
                playFromActual: Use to continue from the current position without returning to trajectory.

        """

        self.__req.setParameter(
            f"root/{self.__addSystem('MotionInterpreter', self.__system_id)}/interpreterCommand",
            InterpreterEvents.PLAY_PROGRAM_E.value).get()
        time.sleep(wait_time)
        return self.getState()

    def playFromActual(self, wait_time=0.1):
        """Plays the program from the robot's current actual position.

        Unlike play(), this method allows the robot to continue from its current position
        even if it has deviated from the original trajectory. The interpreter automatically
        moves the robot back to the trajectory and resumes program execution without
        requiring manual intervention (no button hold needed).

            Args:
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                InterpreterStates: actual state of the program interpreter

            See Also:
                play: Use when the robot is on the trajectory or after manual moveToStart.
                moveToStart: Manual alternative requiring button hold to return to trajectory.

        """

        self.__req.setParameter(
            f"root/{self.__addSystem('MotionInterpreter', self.__system_id)}/interpreterCommand",
            InterpreterEvents.PLAY_PROGRAM_FROM_ACTUAL_E.value).get()
        time.sleep(wait_time)
        return self.getState()

    def pause(self, wait_time=0.1):
        """Pause the program.

        Stops motion while preserving the current program position. Use play() or
        playFromActual() to resume execution.

            Args:
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                InterpreterStates: actual state of the program interpreter

            See Also:
                play: Resume from the trajectory position (may require moveToStart if deviated).
                playFromActual: Resume from the current actual position.
                stop: Stop the program and return to the beginning.

        """

        self.__req.setParameter(
            f"root/{self.__addSystem('MotionInterpreter', self.__system_id)}/interpreterCommand",
            InterpreterEvents.PAUSE_PROGRAM_E.value).get()
        time.sleep(wait_time)
        return self.getState()

    def stop(self, wait_time=0.1):
        """Stop the program.

        Stops motion and resets program execution to the beginning. Unlike pause(),
        the program cannot be resumed from the current position.

            Args:
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                InterpreterStates: actual state of the program interpreter

            See Also:
                pause: Pause without resetting the program position.
                reset: Stop and clear the interpreter buffer.

        """

        self.__req.setParameter(
            f"root/{self.__addSystem('MotionInterpreter', self.__system_id)}/interpreterCommand",
            InterpreterEvents.STOP_PROGRAM_E.value).get()
        time.sleep(wait_time)
        return self.getState()

    def reset(self, wait_time=0.1):
        """Stop the program and clear the interpreter buffer.

        Performs a full reset of the interpreter, stopping any motion and clearing
        all buffered commands. Use this to completely restart program execution.

            Args:
                wait_time(float): short delay after which the actual state of the interpreter is requested

            Returns:
                InterpreterStates: actual state of the program interpreter

            See Also:
                stop: Stop without clearing the buffer.

        """

        self.__req.setParameter(
            f"root/{self.__addSystem('MotionInterpreter', self.__system_id)}/interpreterCommand",
            InterpreterEvents.RESET_INTERPRETER_E.value).get()
        time.sleep(wait_time)
        return self.getState()

    def getState(self):
        """Get the current interpreter state.

            Returns:
                InterpreterStates: actual state of the interpreter

        """
        return self.__req.getParameter(
            f"root/{self.__addSystem('MotionInterpreter', self.__system_id)}/actualStateOut").get().value[0]
