import typer
import asyncio
import requests

llm = typer.Typer(short_help="The LLM management.")


@llm.command(short_help="Start a server for the LLM.")
def serve(
    model: str = typer.Argument("", help="The ID of the LLM model to use."),
    model_name: str = typer.Argument("", help="The name of the LLM model to use."),
    parallal_size: int = typer.Option(1, help="The number of GPUs to use."),
    host: str = typer.Option("0.0.0.0", help="The host of the server."),
    port: int = typer.Option(8000, help="The port of the server."),
    quantization: str = typer.Option(None, help="The quantization method to use."),
    load_format: str = typer.Option(None, help="The load format to use."),
    enforce_eager: bool = typer.Option(
        False, help="Whether to enforce eager execution."
    ),
    max_num_seq: int = typer.Option(2, help="The maximum number of sequences."),
    max_model_len: int = typer.Option(4096, help="The maximum model length."),
):
    from mw_python_sdk.llm.inference import serve

    serve(
        model,
        model_name=model_name,
        host=host,
        port=port,
        max_num_seq=max_num_seq,
        max_model_len=max_model_len,
        tensor_parallel_size=parallal_size,
        quantization=quantization,
        load_format=load_format,
        enforce_eager=enforce_eager,
    )

@llm.command(short_help="Bench an online serving model.")
def bench(
    base_url: str = typer.Option("http://127.0.0.1:8000", help="The base URL of the model."),
    model: str = typer.Option(None, help="The ID of the model to use."),
    tokenizer: str = typer.Option("", help="The tokenizer path to use."),
    dataset_name: str = typer.Option("sharegpt", help="The dataset name to use."),
    dataset_path: str = typer.Option("", help="The dataset path to use."),
    backend: str = typer.Option("vllm", help="The backend to use."),
    share_gpt_context_length: int = typer.Option(4096, help="The shared GPT context length."),
    num_prompts: int = typer.Option(1000, help="The number of prompts to use."),
    output_len: int = typer.Option(None, help="The output length."),
    request_rate: float = typer.Option(float("inf"), help="The request rate."),
    max_concurrency: int = typer.Option(None, help="The maximum concurrency."),
    disable_tqdm: bool = typer.Option(False, help="Whether to disable tqdm."),
):
    from mw_python_sdk.llm.bench import benchmark, set_ulimit, get_tokenizer,get_dataset
    set_ulimit()
    api_url = f"{base_url}/v1/completions"
    model_url = f"{base_url}/v1/models"
    if model is None:
        response = requests.get(model_url, timeout=10)
        model_list = response.json().get("data", [])
        model = model_list[0]["id"] if model_list else None
    print(f"Using tokenzier: {tokenizer}")
    print(f"Get dataset {dataset_name} from {dataset_path}")
    tkzr = get_tokenizer(tokenizer)
    input_requests = get_dataset(dataset_name, dataset_path, num_prompts, share_gpt_context_length, output_len, tkzr)
    asyncio.run(
            benchmark(
                backend=backend,
                api_url=api_url,
                base_url=base_url,
                model_id=model,
                tokenizer=tkzr,
                input_requests=input_requests,
                request_rate=request_rate,
                max_concurrency=max_concurrency,
                disable_tqdm=disable_tqdm,
                lora_name="",
                extra_request_body=dict(),
                profile=False,
            )
        )

datasets = typer.Typer(short_help="The datasets management.")


@datasets.command(short_help="Create a new dataset by uploading a directory.")
def create(
    name: str = typer.Argument(..., help="The name of the dataset to create."),
    source: str = typer.Argument(..., help="The path to the directory to upload."),
    description: str = typer.Option(None, help="The description of the dataset."),
):
    from mw_python_sdk import create_dataset

    create_dataset(name, source, "", description)
    return


@datasets.command(short_help="Upload a file or directory to the dataset.")
def upload(
    source: str = typer.Argument(
        help="The path to the file or the directory to upload."
    ),
    destination: str = typer.Argument(
        help="The destination of the file or the directory in the dataset."
    ),
    dataset: str = typer.Argument(help="The ID of the dataset to upload in."),
    overwrite: bool = typer.Option(
        False, help="Whether to overwrite the file if it already exists."
    ),
    recursive: bool = typer.Option(
        False, "-r", help="Whether to recursively upload all files in the directory."
    ),
):
    from mw_python_sdk import upload_file, upload_folder
    import os

    if recursive:
        if not os.path.isdir(source):
            raise ValueError("Source must be a directory when using --recursive.")
        upload_folder(source, destination, dataset, overwrite)
        return
    else:
        upload_file(source, destination, dataset, overwrite)
        return


@datasets.command(short_help="Download a file or directory from the dataset.")
def download(
    dataset: str = typer.Argument(..., help="The ID of the dataset to download from."),
    source: str = typer.Argument(
        None, help="The name of the file or the directory to download."
    ),
    destination: str = typer.Option(
        None, "-d", help="The destination directory of the downloaded file."
    ),
    recursive: bool = typer.Option(
        False,
        "-r",
        help="Whether to recursively download all files in the directory.",
    ),
):
    from mw_python_sdk import download_file, download_dir

    if recursive or source is None:
        downloaded_dir = download_dir(dataset, sub_dir=source, local_dir=destination)
        print(f"Successfully downloaded to {downloaded_dir}")
        return
    else:
        downloaded_file = download_file(dataset, source, local_dir=destination)
        print(f"Successfully downloaded to {downloaded_file}")
        return


app = typer.Typer()
app.add_typer(llm, name="llm")
app.add_typer(datasets, name="ds")


@app.command(short_help="Set up a fast reverse proxy.")
def frp(port: int = typer.Argument(..., help="The port of the proxy.")):
    from mw_python_sdk.frp import fast_reverse_proxy

    fast_reverse_proxy(port)


if __name__ == "__main__":
    app()
