
################################################################################
##                                                                            ##
##  This file is part of NCrystal (see https://mctools.github.io/ncrystal/)   ##
##                                                                            ##
##  Copyright 2015-2024 NCrystal developers                                   ##
##                                                                            ##
##  Licensed under the Apache License, Version 2.0 (the "License");           ##
##  you may not use this file except in compliance with the License.          ##
##  You may obtain a copy of the License at                                   ##
##                                                                            ##
##      http://www.apache.org/licenses/LICENSE-2.0                            ##
##                                                                            ##
##  Unless required by applicable law or agreed to in writing, software       ##
##  distributed under the License is distributed on an "AS IS" BASIS,         ##
##  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  ##
##  See the License for the specific language governing permissions and       ##
##  limitations under the License.                                            ##
##                                                                            ##
################################################################################

"""

Data for hfg2ncmat

"""

__all__ = []

_cached_data = [None]
def get_data():
    """Hydrogen VDOS for various functional groups (supplementary material of doi:10.1088/1361-648X/abfc13)"""
    if _cached_data[0] is None:
        _cached_data[0] = _get_data()
    return _cached_data[0]

def _get_data():
    import io
    from ._numpy import _np, _ensure_numpy
    _ensure_numpy()
    thedata="""
#Energy      CHali       CHaro       CH2         CH3         NH          NH2         NH3         OH          SH
 0.00000E+00 2.13554E-05 1.33447E-04 4.19945E-05 6.16412E-05 9.87755E-05 1.58844E-04 1.66414E-05 1.10565E-05 9.18572E-06
 9.01804E-04 4.25611E-05 2.14471E-04 9.06037E-05 1.44290E-04 1.63593E-04 2.14715E-04 3.58153E-05 2.55964E-05 2.65186E-05
 1.80361E-03 7.85363E-05 3.05676E-04 1.74811E-04 2.98485E-04 2.36042E-04 2.64685E-04 6.78767E-05 5.03742E-05 6.49416E-05
 2.70541E-03 1.35340E-04 3.94305E-04 3.06398E-04 5.49869E-04 3.07172E-04 3.15534E-04 1.14922E-04 8.57807E-05 1.36052E-04
 3.60721E-03 2.17858E-04 4.72909E-04 4.94518E-04 9.07625E-04 3.75427E-04 3.81898E-04 1.76387E-04 1.29767E-04 2.46917E-04
 4.50902E-03 3.25605E-04 5.45649E-04 7.41915E-04 1.35003E-03 4.46001E-04 4.74550E-04 2.48511E-04 1.80479E-04 3.95660E-04
 5.41082E-03 4.48735E-04 6.27004E-04 1.03959E-03 1.82126E-03 5.24448E-04 5.93392E-04 3.24323E-04 2.39079E-04 5.75327E-04
 6.31263E-03 5.68995E-04 7.31566E-04 1.36181E-03 2.24894E-03 6.13562E-04 7.28691E-04 3.94879E-04 3.09468E-04 7.85659E-04
 7.21443E-03 6.68072E-04 8.62561E-04 1.66718E-03 2.58112E-03 7.15253E-04 8.63823E-04 4.52960E-04 3.95789E-04 1.04112E-03
 8.11623E-03 7.38792E-04 1.01083E-03 1.91299E-03 2.82301E-03 8.31824E-04 9.77799E-04 4.97925E-04 5.01333E-04 1.36244E-03
 9.01804E-03 7.89848E-04 1.16526E-03 2.08157E-03 3.04246E-03 9.64004E-04 1.05469E-03 5.37190E-04 6.28608E-04 1.75127E-03
 9.91984E-03 8.38334E-04 1.32013E-03 2.19762E-03 3.32835E-03 1.10625E-03 1.09889E-03 5.80584E-04 7.75871E-04 2.16194E-03
 1.08216E-02 8.94610E-04 1.46840E-03 2.30773E-03 3.71962E-03 1.24042E-03 1.13848E-03 6.30132E-04 9.29796E-04 2.49355E-03
 1.17234E-02 9.52165E-04 1.59183E-03 2.42918E-03 4.15102E-03 1.33632E-03 1.20438E-03 6.74345E-04 1.06380E-03 2.62439E-03
 1.26253E-02 9.92401E-04 1.66731E-03 2.52363E-03 4.46588E-03 1.36908E-03 1.30279E-03 6.94751E-04 1.15067E-03 2.48124E-03
 1.35271E-02 1.00056E-03 1.68600E-03 2.53463E-03 4.50492E-03 1.34115E-03 1.41275E-03 6.80982E-04 1.18161E-03 2.09326E-03
 1.44289E-02 9.77866E-04 1.66055E-03 2.44867E-03 4.20991E-03 1.27929E-03 1.51185E-03 6.40946E-04 1.17339E-03 1.58157E-03
 1.53307E-02 9.39376E-04 1.61225E-03 2.30892E-03 3.65939E-03 1.20412E-03 1.59661E-03 5.97357E-04 1.15725E-03 1.09459E-03
 1.62325E-02 9.01295E-04 1.55825E-03 2.17637E-03 3.01643E-03 1.11074E-03 1.67237E-03 5.75495E-04 1.15906E-03 7.44844E-04
 1.71343E-02 8.69515E-04 1.51159E-03 2.09365E-03 2.43893E-03 9.87278E-04 1.72917E-03 5.92606E-04 1.18137E-03 5.84810E-04
 1.80361E-02 8.38573E-04 1.47991E-03 2.08070E-03 2.01605E-03 8.43933E-04 1.73756E-03 6.53383E-04 1.19666E-03 6.14449E-04
 1.89379E-02 8.00728E-04 1.45323E-03 2.13783E-03 1.76553E-03 7.11190E-04 1.67048E-03 7.49362E-04 1.16275E-03 7.93397E-04
 1.98397E-02 7.55884E-04 1.39954E-03 2.23602E-03 1.68583E-03 6.11496E-04 1.52619E-03 8.60650E-04 1.05581E-03 1.04638E-03
 2.07415E-02 7.13469E-04 1.28590E-03 2.31545E-03 1.82084E-03 5.43303E-04 1.33199E-03 9.62313E-04 8.91678E-04 1.27519E-03
 2.16433E-02 6.86955E-04 1.10959E-03 2.31413E-03 2.28323E-03 4.93518E-04 1.13214E-03 1.03645E-03 7.12840E-04 1.39276E-03
 2.25451E-02 6.87416E-04 9.08701E-04 2.20659E-03 3.19980E-03 4.55473E-04 9.72409E-04 1.08320E-03 5.57435E-04 1.36774E-03
 2.34469E-02 7.18738E-04 7.41294E-04 2.01501E-03 4.58794E-03 4.29664E-04 8.85848E-04 1.12034E-03 4.42458E-04 1.24063E-03
 2.43487E-02 7.73095E-04 6.51374E-04 1.78608E-03 6.23742E-03 4.14062E-04 8.80270E-04 1.16981E-03 3.68773E-04 1.08604E-03
 2.52505E-02 8.27750E-04 6.48105E-04 1.55903E-03 7.70355E-03 4.02521E-04 9.31984E-04 1.24119E-03 3.30212E-04 9.49908E-04
 2.61523E-02 8.49231E-04 7.08090E-04 1.34987E-03 8.47163E-03 3.92022E-04 9.93141E-04 1.32286E-03 3.15304E-04 8.23730E-04
 2.70541E-02 8.09382E-04 7.89889E-04 1.15816E-03 8.24483E-03 3.85433E-04 1.01396E-03 1.38467E-03 3.08108E-04 6.77375E-04
 2.79559E-02 7.05591E-04 8.49332E-04 9.86162E-04 7.18470E-03 3.85785E-04 9.68990E-04 1.39174E-03 2.95005E-04 5.05124E-04
 2.88577E-02 5.66499E-04 8.54394E-04 8.52984E-04 5.88959E-03 3.90925E-04 8.71613E-04 1.32560E-03 2.71773E-04 3.33831E-04
 2.97595E-02 4.34229E-04 7.96421E-04 7.90895E-04 5.05748E-03 3.93910E-04 7.70013E-04 1.19953E-03 2.42798E-04 1.96165E-04
 3.06613E-02 3.38742E-04 6.88838E-04 8.24336E-04 5.05232E-03 3.85708E-04 7.28555E-04 1.05444E-03 2.15250E-04 1.07436E-04
 3.15631E-02 2.88064E-04 5.54972E-04 9.46241E-04 5.68543E-03 3.58489E-04 8.01248E-04 9.38347E-04 1.96800E-04 6.67500E-05
 3.24649E-02 2.78124E-04 4.19143E-04 1.11351E-03 6.36498E-03 3.11558E-04 1.00492E-03 8.86085E-04 1.96394E-04 7.22683E-05
 3.33667E-02 3.04305E-04 3.07235E-04 1.27160E-03 6.51443E-03 2.56387E-04 1.30524E-03 9.08228E-04 2.19623E-04 1.33835E-04
 3.42685E-02 3.60346E-04 2.47559E-04 1.38883E-03 5.95284E-03 2.13330E-04 1.63069E-03 9.86749E-04 2.59302E-04 2.76106E-04
 3.51703E-02 4.31041E-04 2.62717E-04 1.46831E-03 4.94070E-03 2.00707E-04 1.91505E-03 1.08017E-03 2.95124E-04 5.30132E-04
 3.60721E-02 4.94324E-04 3.54524E-04 1.53155E-03 3.90719E-03 2.24972E-04 2.14219E-03 1.14605E-03 3.08442E-04 9.10846E-04
 3.69739E-02 5.34379E-04 4.93262E-04 1.59724E-03 3.16289E-03 2.77011E-04 2.35289E-03 1.17163E-03 2.96685E-04 1.38907E-03
 3.78758E-02 5.50813E-04 6.25299E-04 1.67167E-03 2.81448E-03 3.34341E-04 2.59498E-03 1.18441E-03 2.70677E-04 1.89372E-03
 3.87776E-02 5.53208E-04 7.02109E-04 1.74344E-03 2.81752E-03 3.70130E-04 2.84682E-03 1.22708E-03 2.40924E-04 2.38883E-03
 3.96794E-02 5.49637E-04 7.09476E-04 1.77856E-03 3.02674E-03 3.68113E-04 2.98723E-03 1.31809E-03 2.11124E-04 3.00879E-03
 4.05812E-02 5.44232E-04 6.69100E-04 1.73298E-03 3.23655E-03 3.33941E-04 2.86805E-03 1.43721E-03 1.83577E-04 4.11298E-03
 4.14830E-02 5.43652E-04 6.14439E-04 1.59053E-03 3.26752E-03 2.92079E-04 2.44791E-03 1.55212E-03 1.66247E-04 6.07819E-03
 4.23848E-02 5.58161E-04 5.68807E-04 1.39201E-03 3.06042E-03 2.69356E-04 1.85455E-03 1.65401E-03 1.72535E-04 8.86949E-03
 4.32866E-02 5.91105E-04 5.39708E-04 1.21275E-03 2.68175E-03 2.78019E-04 1.29979E-03 1.75353E-03 2.12875E-04 1.18232E-02
 4.41884E-02 6.29002E-04 5.20293E-04 1.10302E-03 2.23959E-03 3.10723E-04 9.35775E-04 1.83912E-03 2.82555E-04 1.40974E-02
 4.50902E-02 6.46842E-04 4.94099E-04 1.05342E-03 1.80961E-03 3.50155E-04 7.95317E-04 1.85926E-03 3.55287E-04 1.55273E-02
 4.59920E-02 6.26101E-04 4.49950E-04 1.01607E-03 1.43064E-03 3.84431E-04 8.35668E-04 1.76546E-03 3.93472E-04 1.68688E-02
 4.68938E-02 5.68155E-04 4.03096E-04 9.51045E-04 1.13026E-03 4.14251E-04 9.94366E-04 1.56974E-03 3.73547E-04 1.88838E-02
 4.77956E-02 4.91595E-04 4.06307E-04 8.55160E-04 9.26704E-04 4.46980E-04 1.20460E-03 1.34763E-03 3.05642E-04 2.12745E-02
 4.86974E-02 4.19702E-04 5.39228E-04 7.61693E-04 8.12765E-04 4.88092E-04 1.40113E-03 1.18357E-03 2.27511E-04 2.29890E-02
 4.95992E-02 3.71259E-04 8.70489E-04 7.23017E-04 7.55689E-04 5.38891E-04 1.54335E-03 1.11493E-03 1.79456E-04 2.36665E-02
 5.05010E-02 3.58327E-04 1.39404E-03 7.88551E-04 7.20043E-04 5.93564E-04 1.62890E-03 1.12038E-03 1.86541E-04 2.41547E-02
 5.14028E-02 3.85499E-04 1.97567E-03 9.82931E-04 6.91270E-04 6.30993E-04 1.67787E-03 1.14986E-03 2.61775E-04 2.51304E-02
 5.23046E-02 4.47087E-04 2.38397E-03 1.28730E-03 6.78677E-04 6.18081E-04 1.71361E-03 1.16197E-03 4.18788E-04 2.56847E-02
 5.32064E-02 5.24661E-04 2.43275E-03 1.63161E-03 6.95999E-04 5.36808E-04 1.75929E-03 1.13822E-03 6.74168E-04 2.40177E-02
 5.41082E-02 5.90135E-04 2.12804E-03 1.91126E-03 7.38209E-04 4.11073E-04 1.82741E-03 1.07414E-03 1.02769E-03 1.94793E-02
 5.50100E-02 6.17487E-04 1.66570E-03 2.03320E-03 7.78483E-04 2.99385E-04 1.89553E-03 9.71655E-04 1.42648E-03 1.33715E-02
 5.59118E-02 5.97876E-04 1.27277E-03 1.96857E-03 7.88362E-04 2.56638E-04 1.90779E-03 8.46142E-04 1.74910E-03 7.73603E-03
 5.68136E-02 5.44497E-04 1.05397E-03 1.76689E-03 7.58400E-04 2.98616E-04 1.81701E-03 7.32579E-04 1.85484E-03 3.80368E-03
 5.77154E-02 4.79775E-04 9.73274E-04 1.51481E-03 6.99160E-04 3.93330E-04 1.62045E-03 6.76823E-04 1.68308E-03 1.61756E-03
 5.86172E-02 4.17208E-04 9.39182E-04 1.27898E-03 6.27257E-04 4.81355E-04 1.35211E-03 7.23944E-04 1.30836E-03 6.16113E-04
 5.95190E-02 3.57990E-04 8.86729E-04 1.08375E-03 5.54107E-04 5.12331E-04 1.05451E-03 9.16492E-04 8.85654E-04 2.41192E-04
 6.04208E-02 3.05618E-04 8.05589E-04 9.28810E-04 4.85958E-04 4.73128E-04 7.65035E-04 1.28874E-03 5.42408E-04 1.58891E-04
 6.13226E-02 2.81373E-04 7.29077E-04 8.14419E-04 4.30056E-04 3.88559E-04 5.17882E-04 1.84622E-03 3.23093E-04 2.22697E-04
 6.22244E-02 3.22254E-04 7.04691E-04 7.49499E-04 3.96482E-04 3.03888E-04 3.46414E-04 2.55925E-03 2.11969E-04 3.77137E-04
 6.31263E-02 4.57138E-04 7.56295E-04 7.42514E-04 3.90339E-04 2.75181E-04 2.82042E-04 3.39882E-03 1.77157E-04 5.80124E-04
 6.40281E-02 6.74445E-04 8.60294E-04 7.88487E-04 4.01547E-04 3.75465E-04 3.56104E-04 4.37985E-03 1.89214E-04 7.63247E-04
 6.49299E-02 9.08143E-04 9.61499E-04 8.65016E-04 4.06452E-04 6.96027E-04 6.07028E-04 5.55023E-03 2.19268E-04 8.44648E-04
 6.58317E-02 1.06424E-03 1.01938E-03 9.41484E-04 3.83923E-04 1.31255E-03 1.07872E-03 6.92833E-03 2.40171E-04 7.81862E-04
 6.67335E-02 1.07689E-03 1.03945E-03 9.95116E-04 3.30365E-04 2.20795E-03 1.78403E-03 8.44081E-03 2.36681E-04 6.07435E-04
 6.76353E-02 9.49004E-04 1.05792E-03 1.02213E-03 2.59006E-04 3.20262E-03 2.63340E-03 9.87339E-03 2.13914E-04 4.02477E-04
 6.85371E-02 7.41801E-04 1.09763E-03 1.03455E-03 1.87494E-04 3.99232E-03 3.39692E-03 1.08536E-02 1.95749E-04 2.35479E-04
 6.94389E-02 5.27914E-04 1.14311E-03 1.04416E-03 1.27878E-04 4.32772E-03 3.78879E-03 1.09742E-02 2.16381E-04 1.28843E-04
 7.03407E-02 3.53624E-04 1.15882E-03 1.04913E-03 8.51873E-05 4.19319E-03 3.65016E-03 1.00783E-02 3.09917E-04 7.20494E-05
 7.12425E-02 2.34574E-04 1.12605E-03 1.03926E-03 6.05668E-05 3.78922E-03 3.06974E-03 8.44482E-03 4.93918E-04 5.04533E-05
 7.21443E-02 1.71306E-04 1.05975E-03 1.01418E-03 5.38557E-05 3.34684E-03 2.32512E-03 6.61756E-03 7.46013E-04 5.98080E-05
 7.30461E-02 1.61466E-04 1.00146E-03 9.91829E-04 6.45296E-05 3.02935E-03 1.70765E-03 5.04272E-03 9.95627E-04 1.02379E-04
 7.39479E-02 2.02123E-04 1.00980E-03 9.96603E-04 9.23836E-05 3.04782E-03 1.37625E-03 3.88819E-03 1.15688E-03 1.73607E-04
 7.48497E-02 2.90021E-04 1.14439E-03 1.04031E-03 1.38498E-04 3.78357E-03 1.31850E-03 3.11783E-03 1.18585E-03 2.51393E-04
 7.57515E-02 4.25679E-04 1.42148E-03 1.11302E-03 2.04337E-04 5.62750E-03 1.41415E-03 2.61626E-03 1.10696E-03 3.01582E-04
 7.66533E-02 6.14669E-04 1.76493E-03 1.18732E-03 2.86141E-04 8.53271E-03 1.54419E-03 2.25266E-03 9.83154E-04 3.02333E-04
 7.75551E-02 8.54817E-04 2.01986E-03 1.23063E-03 3.67561E-04 1.16654E-02 1.65429E-03 1.92662E-03 8.66348E-04 2.67980E-04
 7.84569E-02 1.11725E-03 2.04761E-03 1.22139E-03 4.21341E-04 1.36447E-02 1.72500E-03 1.60584E-03 7.73835E-04 2.47695E-04
 7.93587E-02 1.34859E-03 1.82418E-03 1.16321E-03 4.26104E-04 1.34069E-02 1.71865E-03 1.31351E-03 6.96274E-04 2.96259E-04
 8.02605E-02 1.49930E-03 1.45098E-03 1.08527E-03 3.85081E-04 1.10015E-02 1.60613E-03 1.07789E-03 6.15583E-04 4.35319E-04
 8.11623E-02 1.54402E-03 1.07654E-03 1.02080E-03 3.25637E-04 7.53741E-03 1.46086E-03 9.00522E-04 5.18359E-04 6.27298E-04
 8.20641E-02 1.47488E-03 8.06277E-04 9.78026E-04 2.79194E-04 4.34849E-03 1.49686E-03 7.61559E-04 4.03372E-04 7.83724E-04
 8.29659E-02 1.29661E-03 6.68619E-04 9.34684E-04 2.63888E-04 2.21824E-03 1.96191E-03 6.40050E-04 2.83098E-04 8.18016E-04
 8.38677E-02 1.04225E-03 6.50791E-04 8.65604E-04 2.84624E-04 1.25611E-03 2.92573E-03 5.28966E-04 1.77909E-04 7.13934E-04
 8.47695E-02 7.77392E-04 7.67023E-04 7.77775E-04 3.40953E-04 1.23353E-03 4.12937E-03 4.36578E-04 1.05958E-04 5.58173E-04
 8.56713E-02 5.67888E-04 1.09409E-03 7.23538E-04 4.27993E-04 1.87838E-03 5.09013E-03 3.73555E-04 7.62571E-05 5.31667E-04
 8.65731E-02 4.42793E-04 1.72248E-03 7.88779E-04 5.31546E-04 2.92195E-03 5.46405E-03 3.38535E-04 8.92476E-05 9.06222E-04
 8.74749E-02 3.93320E-04 2.63802E-03 1.06624E-03 6.30578E-04 4.04304E-03 5.35084E-03 3.19270E-04 1.41667E-04 2.03322E-03
 8.83768E-02 4.01817E-04 3.63837E-03 1.61546E-03 7.14475E-04 4.90140E-03 5.20668E-03 3.07450E-04 2.29221E-04 4.18652E-03
 8.92786E-02 4.64305E-04 4.40870E-03 2.41142E-03 8.03435E-04 5.27503E-03 5.44737E-03 3.06423E-04 3.43357E-04 7.18925E-03
 9.01804E-02 5.85337E-04 4.74497E-03 3.30885E-03 9.42705E-04 5.16554E-03 6.15451E-03 3.20849E-04 4.65470E-04 1.01265E-02
 9.10822E-02 7.53872E-04 4.72176E-03 4.06602E-03 1.15455E-03 4.75805E-03 7.10300E-03 3.42639E-04 5.68286E-04 1.16681E-02
 9.19840E-02 9.26794E-04 4.61702E-03 4.44291E-03 1.38793E-03 4.26684E-03 7.94826E-03 3.52610E-04 6.28866E-04 1.09950E-02
 9.28858E-02 1.04334E-03 4.65970E-03 4.33086E-03 1.53891E-03 3.80643E-03 8.37200E-03 3.36567E-04 6.41205E-04 8.48018E-03
 9.37876E-02 1.06635E-03 4.84426E-03 3.82969E-03 1.54449E-03 3.39294E-03 8.19076E-03 2.97479E-04 6.13237E-04 5.37147E-03
 9.46894E-02 1.01225E-03 4.96361E-03 3.19671E-03 1.44674E-03 3.04808E-03 7.46185E-03 2.52114E-04 5.54440E-04 2.84268E-03
 9.55912E-02 9.33412E-04 4.81850E-03 2.68053E-03 1.34652E-03 2.89399E-03 6.48646E-03 2.17930E-04 4.74435E-04 1.38755E-03
 9.64930E-02 8.67134E-04 4.44555E-03 2.37227E-03 1.30547E-03 3.14089E-03 5.63459E-03 2.02936E-04 3.94758E-04 9.20689E-04
 9.73948E-02 8.08755E-04 4.18042E-03 2.20525E-03 1.31218E-03 3.93387E-03 5.11187E-03 2.04981E-04 3.53342E-04 1.14963E-03
 9.82966E-02 7.38522E-04 4.45742E-03 2.07483E-03 1.32957E-03 5.14039E-03 4.86181E-03 2.17735E-04 3.89067E-04 1.75687E-03
 9.91984E-02 6.64521E-04 5.46023E-03 1.93594E-03 1.34499E-03 6.28719E-03 4.66621E-03 2.36230E-04 5.18840E-04 2.38820E-03
 1.00100E-01 6.27400E-04 6.91981E-03 1.80226E-03 1.36873E-03 6.80472E-03 4.34433E-03 2.57969E-04 7.27581E-04 2.69130E-03
 1.01002E-01 6.63567E-04 8.27071E-03 1.69460E-03 1.40274E-03 6.42930E-03 3.88578E-03 2.82190E-04 9.76170E-04 2.48494E-03
 1.01904E-01 7.68506E-04 9.03450E-03 1.61343E-03 1.42599E-03 5.37575E-03 3.41203E-03 3.10996E-04 1.21814E-03 1.87635E-03
 1.02806E-01 8.95144E-04 9.06681E-03 1.55576E-03 1.41484E-03 4.10964E-03 3.03145E-03 3.51295E-04 1.41841E-03 1.16007E-03
 1.03707E-01 9.89014E-04 8.49407E-03 1.54544E-03 1.37636E-03 2.99597E-03 2.76594E-03 4.14110E-04 1.56875E-03 5.92342E-04
 1.04609E-01 1.03061E-03 7.51659E-03 1.63414E-03 1.35243E-03 2.17476E-03 2.61533E-03 5.09635E-04 1.68258E-03 2.60062E-04
 1.05511E-01 1.04435E-03 6.30225E-03 1.86010E-03 1.38061E-03 1.65819E-03 2.63685E-03 6.39272E-04 1.76318E-03 1.14452E-04
 1.06413E-01 1.06667E-03 4.99235E-03 2.20160E-03 1.45572E-03 1.41489E-03 2.91981E-03 7.91113E-04 1.78723E-03 6.84564E-05
 1.07315E-01 1.11586E-03 3.73885E-03 2.57381E-03 1.54504E-03 1.35787E-03 3.49827E-03 9.46685E-04 1.75200E-03 5.80829E-05
 1.08216E-01 1.20101E-03 2.72486E-03 2.87111E-03 1.64208E-03 1.34741E-03 4.31884E-03 1.09367E-03 1.76472E-03 5.49597E-05
 1.09118E-01 1.34653E-03 2.14645E-03 3.01478E-03 1.78948E-03 1.27723E-03 5.28680E-03 1.22639E-03 2.10295E-03 6.32341E-05
 1.10020E-01 1.58336E-03 2.14867E-03 2.97856E-03 2.04128E-03 1.15993E-03 6.29917E-03 1.32958E-03 3.18216E-03 1.32270E-04
 1.10922E-01 1.90596E-03 2.73356E-03 2.79702E-03 2.41291E-03 1.10342E-03 7.20362E-03 1.36960E-03 5.35317E-03 4.03967E-04
 1.11824E-01 2.24950E-03 3.68910E-03 2.56400E-03 2.89048E-03 1.18902E-03 7.76447E-03 1.31679E-03 8.51037E-03 1.17744E-03
 1.12725E-01 2.52840E-03 4.63322E-03 2.40878E-03 3.51333E-03 1.37053E-03 7.75510E-03 1.18183E-03 1.18027E-02 2.88544E-03
 1.13627E-01 2.70442E-03 5.23150E-03 2.43916E-03 4.43743E-03 1.49761E-03 7.13250E-03 1.02042E-03 1.39474E-02 5.82123E-03
 1.14529E-01 2.80429E-03 5.44817E-03 2.67214E-03 5.82704E-03 1.44942E-03 6.10091E-03 8.92326E-04 1.41145E-02 9.65542E-03
 1.15431E-01 2.85866E-03 5.55666E-03 3.00083E-03 7.57113E-03 1.25669E-03 4.97512E-03 8.19330E-04 1.25195E-02 1.32323E-02
 1.16333E-01 2.83810E-03 5.85835E-03 3.24181E-03 9.12056E-03 1.10063E-03 3.98965E-03 7.90911E-04 1.00691E-02 1.52041E-02
 1.17234E-01 2.67945E-03 6.40657E-03 3.25670E-03 9.75855E-03 1.20632E-03 3.24893E-03 8.06299E-04 7.56277E-03 1.52080E-02
 1.18136E-01 2.38580E-03 7.00590E-03 3.05864E-03 9.14516E-03 1.76257E-03 2.81419E-03 8.97531E-04 5.35310E-03 1.43489E-02
 1.19038E-01 2.07822E-03 7.40816E-03 2.79332E-03 7.59256E-03 2.96931E-03 2.75338E-03 1.10225E-03 3.54720E-03 1.42178E-02
 1.19940E-01 1.92043E-03 7.47557E-03 2.60254E-03 5.81419E-03 5.12524E-03 3.05905E-03 1.41015E-03 2.20777E-03 1.53103E-02
 1.20842E-01 1.98645E-03 7.23418E-03 2.51467E-03 4.45354E-03 8.47990E-03 3.54484E-03 1.73650E-03 1.33322E-03 1.64970E-02
 1.21743E-01 2.21246E-03 6.83939E-03 2.46685E-03 3.80057E-03 1.27170E-02 3.88877E-03 1.95944E-03 8.17857E-04 1.61021E-02
 1.22645E-01 2.47881E-03 6.44977E-03 2.40150E-03 3.78399E-03 1.65352E-02 3.82896E-03 2.00202E-03 5.17267E-04 1.34596E-02
 1.23547E-01 2.72492E-03 6.08228E-03 2.31867E-03 4.10330E-03 1.81246E-02 3.34013E-03 1.88572E-03 3.31783E-04 9.41095E-03
 1.24449E-01 2.97738E-03 5.61681E-03 2.25142E-03 4.39153E-03 1.65023E-02 2.63408E-03 1.70201E-03 2.23560E-04 5.45684E-03
 1.25351E-01 3.28144E-03 4.96854E-03 2.21873E-03 4.37313E-03 1.24143E-02 1.99553E-03 1.53405E-03 1.84847E-04 2.62856E-03
 1.26253E-01 3.63024E-03 4.21232E-03 2.21184E-03 3.97116E-03 7.75621E-03 1.59873E-03 1.40889E-03 2.05441E-04 1.10475E-03
 1.27154E-01 3.97091E-03 3.50422E-03 2.22267E-03 3.30291E-03 4.14429E-03 1.44475E-03 1.31420E-03 2.61032E-04 5.69516E-04
 1.28056E-01 4.26621E-03 2.93008E-03 2.27706E-03 2.57435E-03 2.07847E-03 1.44086E-03 1.24225E-03 3.27344E-04 6.83176E-04
 1.28958E-01 4.52970E-03 2.48988E-03 2.42608E-03 1.96520E-03 1.18195E-03 1.52255E-03 1.20971E-03 4.21332E-04 1.24383E-03
 1.29860E-01 4.79251E-03 2.21229E-03 2.69343E-03 1.57546E-03 8.75362E-04 1.70920E-03 1.24109E-03 6.62054E-04 2.06784E-03
 1.30762E-01 5.03742E-03 2.22809E-03 3.03345E-03 1.42418E-03 8.09978E-04 2.06825E-03 1.34459E-03 1.33238E-03 2.84878E-03
 1.31663E-01 5.17127E-03 2.70308E-03 3.34774E-03 1.46471E-03 9.33988E-04 2.62756E-03 1.51100E-03 2.88981E-03 3.21727E-03
 1.32565E-01 5.08124E-03 3.68536E-03 3.54633E-03 1.61077E-03 1.36861E-03 3.29261E-03 1.73068E-03 5.80125E-03 2.98258E-03
 1.33467E-01 4.75189E-03 4.99838E-03 3.60253E-03 1.77823E-03 2.23085E-03 3.84246E-03 2.00038E-03 1.01036E-02 2.30171E-03
 1.34369E-01 4.34793E-03 6.28765E-03 3.56627E-03 1.91653E-03 3.45557E-03 4.05470E-03 2.31593E-03 1.49406E-02 1.58723E-03
 1.35271E-01 4.17322E-03 7.20540E-03 3.53284E-03 1.99816E-03 4.70594E-03 3.88523E-03 2.67765E-03 1.87227E-02 1.25963E-03
 1.36172E-01 4.52141E-03 7.58151E-03 3.58765E-03 1.98728E-03 5.49880E-03 3.52199E-03 3.11074E-03 2.01736E-02 1.55746E-03
 1.37074E-01 5.53325E-03 7.46184E-03 3.75946E-03 1.84443E-03 5.51295E-03 3.22911E-03 3.66544E-03 1.94028E-02 2.44736E-03
 1.37976E-01 7.14931E-03 7.06606E-03 4.01264E-03 1.57499E-03 4.81169E-03 3.13483E-03 4.38600E-03 1.77965E-02 3.58679E-03
 1.38878E-01 9.13769E-03 6.74857E-03 4.28791E-03 1.25893E-03 3.75809E-03 3.19173E-03 5.27873E-03 1.67520E-02 4.41933E-03
 1.39780E-01 1.11482E-02 6.90576E-03 4.55606E-03 1.01397E-03 2.73622E-03 3.32153E-03 6.29508E-03 1.65414E-02 4.48375E-03
 1.40681E-01 1.28139E-02 7.76935E-03 4.82015E-03 9.21220E-04 1.97118E-03 3.52450E-03 7.31771E-03 1.63108E-02 3.73843E-03
 1.41583E-01 1.39050E-02 9.22083E-03 5.05581E-03 9.77020E-04 1.56216E-03 3.81308E-03 8.16001E-03 1.50179E-02 2.56978E-03
 1.42485E-01 1.44067E-02 1.08209E-02 5.17814E-03 1.10207E-03 1.59799E-03 4.09957E-03 8.61111E-03 1.23525E-02 1.46992E-03
 1.43387E-01 1.44001E-02 1.20394E-02 5.11043E-03 1.19374E-03 2.17867E-03 4.22509E-03 8.52102E-03 8.90087E-03 7.21891E-04
 1.44289E-01 1.38777E-02 1.24882E-02 4.87861E-03 1.18315E-03 3.28933E-03 4.09706E-03 7.87248E-03 5.60065E-03 3.44142E-04
 1.45190E-01 1.27321E-02 1.20237E-02 4.58970E-03 1.06159E-03 4.64776E-03 3.75969E-03 6.79021E-03 3.10907E-03 2.21288E-04
 1.46092E-01 1.09355E-02 1.07511E-02 4.30949E-03 8.69081E-04 5.74622E-03 3.33955E-03 5.48441E-03 1.56847E-03 2.28245E-04
 1.46994E-01 8.69388E-03 8.99826E-03 4.01131E-03 6.64817E-04 6.15712E-03 2.95362E-03 4.16854E-03 7.62748E-04 2.72266E-04
 1.47896E-01 6.41968E-03 7.23360E-03 3.67028E-03 5.02800E-04 5.84372E-03 2.65003E-03 3.00305E-03 3.88976E-04 2.93160E-04
 1.48798E-01 4.58414E-03 5.87638E-03 3.36927E-03 4.21696E-04 5.14143E-03 2.39646E-03 2.09091E-03 2.33420E-04 2.65157E-04
 1.49699E-01 3.58367E-03 5.08538E-03 3.27577E-03 4.43113E-04 4.44084E-03 2.12823E-03 1.50601E-03 2.13366E-04 1.98356E-04
 1.50601E-01 3.64923E-03 4.72700E-03 3.52763E-03 5.66797E-04 3.94364E-03 1.82943E-03 1.30679E-03 3.45166E-04 1.24663E-04
 1.51503E-01 4.74835E-03 4.57494E-03 4.16052E-03 7.61927E-04 3.70224E-03 1.56221E-03 1.50571E-03 6.93073E-04 7.68150E-05
 1.52405E-01 6.52274E-03 4.51943E-03 5.13117E-03 9.71631E-04 3.74850E-03 1.40842E-03 2.01995E-03 1.31237E-03 8.24601E-05
 1.53307E-01 8.41129E-03 4.55934E-03 6.37897E-03 1.14480E-03 4.05434E-03 1.39421E-03 2.66575E-03 2.20006E-03 1.75953E-04
 1.54208E-01 9.96595E-03 4.65346E-03 7.83508E-03 1.27739E-03 4.40273E-03 1.47451E-03 3.22562E-03 3.28734E-03 4.02253E-04
 1.55110E-01 1.10799E-02 4.67693E-03 9.35088E-03 1.42699E-03 4.44722E-03 1.57277E-03 3.53986E-03 4.47151E-03 7.83321E-04
 1.56012E-01 1.18756E-02 4.53204E-03 1.06414E-02 1.69158E-03 3.98390E-03 1.62600E-03 3.55011E-03 5.62090E-03 1.25635E-03
 1.56914E-01 1.24256E-02 4.24438E-03 1.13863E-02 2.17357E-03 3.13601E-03 1.60516E-03 3.28741E-03 6.53026E-03 1.65255E-03
 1.57816E-01 1.26940E-02 3.94612E-03 1.14672E-02 2.93297E-03 2.24469E-03 1.51421E-03 2.84881E-03 6.92760E-03 1.78134E-03
 1.58717E-01 1.27313E-02 3.80392E-03 1.11011E-02 3.91210E-03 1.60618E-03 1.38358E-03 2.37398E-03 6.61690E-03 1.57321E-03
 1.59619E-01 1.27875E-02 3.94481E-03 1.06940E-02 4.86919E-03 1.30789E-03 1.26263E-03 1.99882E-03 5.65827E-03 1.13822E-03
 1.60521E-01 1.31861E-02 4.39849E-03 1.05481E-02 5.43937E-03 1.24477E-03 1.19919E-03 1.79916E-03 4.40219E-03 6.74792E-04
 1.61423E-01 1.41611E-02 5.09936E-03 1.06913E-02 5.37125E-03 1.24237E-03 1.20540E-03 1.77182E-03 3.32760E-03 3.28362E-04
 1.62325E-01 1.58087E-02 5.94234E-03 1.09210E-02 4.76019E-03 1.18650E-03 1.23798E-03 1.86350E-03 2.80229E-03 1.32433E-04
 1.63226E-01 1.80493E-02 6.79875E-03 1.09545E-02 4.02661E-03 1.08101E-03 1.22382E-03 2.00791E-03 2.91430E-03 4.66039E-05
 1.64128E-01 2.05032E-02 7.47512E-03 1.05712E-02 3.64628E-03 1.00885E-03 1.12084E-03 2.14044E-03 3.44750E-03 1.77135E-05
 1.65030E-01 2.24197E-02 7.74464E-03 9.70856E-03 3.89493E-03 1.05038E-03 9.61952E-04 2.20006E-03 3.99958E-03 1.05687E-05
 1.65932E-01 2.29007E-02 7.50065E-03 8.49991E-03 4.77599E-03 1.23599E-03 8.39469E-04 2.14247E-03 4.19657E-03 9.16902E-06
 1.66834E-01 2.14352E-02 6.86099E-03 7.23107E-03 6.05684E-03 1.54379E-03 8.45302E-04 1.96566E-03 3.90736E-03 8.58443E-06
 1.67735E-01 1.83585E-02 6.07792E-03 6.22315E-03 7.32131E-03 1.89720E-03 1.01715E-03 1.72150E-03 3.32777E-03 8.56800E-06
 1.68637E-01 1.47539E-02 5.36461E-03 5.70885E-03 8.09879E-03 2.16075E-03 1.31974E-03 1.49022E-03 2.86468E-03 1.00731E-05
 1.69539E-01 1.17885E-02 4.82248E-03 5.77147E-03 8.12122E-03 2.19445E-03 1.66263E-03 1.33205E-03 2.87824E-03 1.34015E-05
 1.70441E-01 1.00179E-02 4.46562E-03 6.34950E-03 7.54862E-03 1.97372E-03 1.95484E-03 1.25506E-03 3.42696E-03 1.75757E-05
 1.71343E-01 9.18113E-03 4.23448E-03 7.27253E-03 6.92956E-03 1.66770E-03 2.18280E-03 1.21982E-03 4.18668E-03 2.09912E-05
 1.72244E-01 8.54470E-03 3.99993E-03 8.32069E-03 6.89297E-03 1.57834E-03 2.45511E-03 1.16957E-03 4.65962E-03 2.26864E-05
 1.73146E-01 7.48257E-03 3.63049E-03 9.31548E-03 7.81626E-03 1.97406E-03 2.94744E-03 1.06424E-03 4.58005E-03 2.28539E-05
 1.74048E-01 5.87215E-03 3.09718E-03 1.02006E-02 9.70935E-03 2.93532E-03 3.74437E-03 9.01215E-04 4.19160E-03 2.20611E-05
 1.74950E-01 4.05960E-03 2.50723E-03 1.10333E-02 1.23554E-02 4.31838E-03 4.68179E-03 7.12658E-04 4.13653E-03 2.03405E-05
 1.75852E-01 2.50566E-03 2.03389E-03 1.18760E-02 1.55142E-02 5.87246E-03 5.35869E-03 5.43593E-04 5.05346E-03 1.73790E-05
 1.76754E-01 1.46137E-03 1.81821E-03 1.26874E-02 1.89127E-02 7.39936E-03 5.39333E-03 4.29206E-04 7.21652E-03 1.33361E-05
 1.77655E-01 8.99588E-04 1.90418E-03 1.33067E-02 2.20127E-02 8.76068E-03 4.74908E-03 3.87237E-04 1.04418E-02 9.05222E-06
 1.78557E-01 6.44906E-04 2.20535E-03 1.35158E-02 2.39421E-02 9.74660E-03 3.82345E-03 4.23482E-04 1.42272E-02 5.49473E-06
 1.79459E-01 5.25989E-04 2.52177E-03 1.31142E-02 2.38964E-02 1.00832E-02 3.19740E-03 5.38895E-04 1.79419E-02 3.24092E-06
 1.80361E-01 4.47459E-04 2.65056E-03 1.19856E-02 2.17076E-02 9.67817E-03 3.24562E-03 7.35254E-04 2.09134E-02 2.42064E-06
 1.81263E-01 3.83197E-04 2.53848E-03 1.01698E-02 1.79827E-02 8.80127E-03 3.87840E-03 1.02767E-03 2.24266E-02 2.86754E-06
 1.82164E-01 3.38764E-04 2.32617E-03 7.90500E-03 1.36939E-02 7.93296E-03 4.57732E-03 1.46673E-03 2.18334E-02 4.20987E-06
 1.83066E-01 3.23320E-04 2.21737E-03 5.57134E-03 9.66624E-03 7.42842E-03 4.73900E-03 2.15049E-03 1.89422E-02 5.98710E-06
 1.83968E-01 3.40614E-04 2.29415E-03 3.54039E-03 6.35278E-03 7.29442E-03 4.11030E-03 3.18973E-03 1.44146E-02 7.94154E-06
 1.84870E-01 3.86471E-04 2.45101E-03 2.03181E-03 3.88557E-03 7.20681E-03 2.94407E-03 4.61031E-03 9.59285E-03 1.02376E-05
 1.85772E-01 4.44804E-04 2.49067E-03 1.07173E-03 2.20659E-03 6.74492E-03 1.74130E-03 6.24458E-03 5.71053E-03 1.31295E-05
 1.86673E-01 4.89341E-04 2.28091E-03 5.49451E-04 1.16608E-03 5.71644E-03 8.72005E-04 7.72192E-03 3.22687E-03 1.62130E-05
 1.87575E-01 4.97569E-04 1.84575E-03 3.08456E-04 5.83816E-04 4.33398E-03 4.27471E-04 8.62397E-03 1.87439E-03 1.82510E-05
 1.88477E-01 4.68087E-04 1.33689E-03 2.15237E-04 2.91610E-04 3.07958E-03 3.28858E-04 8.72258E-03 1.16673E-03 1.81620E-05
 1.89379E-01 4.24874E-04 9.25962E-04 1.85148E-04 1.59714E-04 2.38198E-03 4.83351E-04 8.12114E-03 7.73862E-04 1.62078E-05
 1.90281E-01 4.01831E-04 7.06977E-04 1.75640E-04 1.04853E-04 2.38902E-03 8.46148E-04 7.19378E-03 5.62744E-04 1.38868E-05
 1.91182E-01 4.18465E-04 6.69992E-04 1.69512E-04 8.31447E-05 2.95154E-03 1.40066E-03 6.37246E-03 4.88675E-04 1.25665E-05
 1.92084E-01 4.65842E-04 7.39777E-04 1.62354E-04 7.57509E-05 3.72367E-03 2.11618E-03 5.91821E-03 5.22092E-04 1.23911E-05
 1.92986E-01 5.13223E-04 8.33293E-04 1.55282E-04 7.54399E-05 4.29860E-03 2.90993E-03 5.81345E-03 6.39265E-04 1.24876E-05
 1.93888E-01 5.29044E-04 8.98433E-04 1.50118E-04 7.84317E-05 4.39284E-03 3.63780E-03 5.85310E-03 8.34445E-04 1.19668E-05
 1.94790E-01 5.00158E-04 9.22440E-04 1.46923E-04 8.13768E-05 4.02447E-03 4.16687E-03 5.87041E-03 1.11835E-03 1.06808E-05
 1.95691E-01 4.36540E-04 9.18611E-04 1.44542E-04 8.13161E-05 3.51947E-03 4.52053E-03 5.91100E-03 1.48707E-03 9.25058E-06
 1.96593E-01 3.60708E-04 9.07339E-04 1.42174E-04 7.67609E-05 3.27636E-03 4.94512E-03 6.23194E-03 1.87897E-03 8.52439E-06
 1.97495E-01 2.92693E-04 9.00667E-04 1.39512E-04 6.85820E-05 3.45379E-03 5.76904E-03 7.17500E-03 2.17558E-03 8.96039E-06
 1.98397E-01 2.42480E-04 8.91864E-04 1.35832E-04 5.95965E-05 3.84212E-03 7.16286E-03 9.01804E-03 2.27506E-03 1.04048E-05
 1.99299E-01 2.11915E-04 8.55663E-04 1.30088E-04 5.25510E-05 4.03334E-03 9.01644E-03 1.18394E-02 2.17256E-03 1.23115E-05
 2.00200E-01 1.99110E-04 7.65763E-04 1.22158E-04 4.79826E-05 3.74593E-03 1.09781E-02 1.54084E-02 1.95622E-03 1.40300E-05
 2.01102E-01 1.99930E-04 6.19388E-04 1.13240E-04 4.41299E-05 3.02712E-03 1.25748E-02 1.91552E-02 1.72733E-03 1.49401E-05
 2.02004E-01 2.07646E-04 4.45025E-04 1.04567E-04 3.89821E-05 2.18043E-03 1.33924E-02 2.22765E-02 1.53631E-03 1.46076E-05
 2.02906E-01 2.14004E-04 2.83915E-04 9.59356E-05 3.21198E-05 1.52124E-03 1.32701E-02 2.39907E-02 1.38466E-03 1.30487E-05
 2.03808E-01 2.12071E-04 1.63802E-04 8.57515E-05 2.45975E-05 1.18035E-03 1.23391E-02 2.38540E-02 1.25939E-03 1.07927E-05
 2.04709E-01 1.98489E-04 8.87118E-05 7.25329E-05 1.76550E-05 1.08002E-03 1.08550E-02 2.19247E-02 1.14845E-03 8.52022E-06
 2.05611E-01 1.73431E-04 4.74294E-05 5.64907E-05 1.19217E-05 1.04899E-03 9.02864E-03 1.86396E-02 1.03173E-03 6.59122E-06
 2.06513E-01 1.39445E-04 2.63093E-05 3.98465E-05 7.51735E-06 9.61891E-04 7.03709E-03 1.45691E-02 8.80932E-04 4.95590E-06
 2.07415E-01 1.01117E-04 1.56672E-05 2.55632E-05 4.38176E-06 8.05888E-04 5.11406E-03 1.03090E-02 6.83976E-04 3.46627E-06
 2.08317E-01 6.47157E-05 1.00685E-05 1.55435E-05 2.37517E-06 6.50219E-04 3.52361E-03 6.46256E-03 4.64870E-04 2.14376E-06
 2.09218E-01 3.60205E-05 6.75966E-06 9.81309E-06 1.25438E-06 5.65214E-04 2.41649E-03 3.51158E-03 2.69394E-04 1.12995E-06
 2.10120E-01 1.75115E-05 4.46758E-06 7.09997E-06 7.16636E-07 5.59774E-04 1.73944E-03 1.62428E-03 1.30958E-04 4.96887E-07
 2.11022E-01 7.78296E-06 2.73358E-06 5.93485E-06 4.95933E-07 5.78775E-04 1.30641E-03 6.31196E-04 5.28866E-05 1.80281E-07
 2.11924E-01 3.52044E-06 1.47653E-06 5.33894E-06 4.18314E-07 5.53055E-04 9.54129E-04 2.04229E-04 1.76391E-05 5.36713E-08
 2.12826E-01 1.79660E-06 6.82116E-07 4.90421E-06 3.94878E-07 4.55211E-04 6.29645E-04 5.46960E-05 4.84103E-06 1.30754E-08
 2.13727E-01 9.94492E-07 2.64314E-07 4.55838E-06 3.88813E-07 3.13351E-04 3.59292E-04 1.20781E-05 1.09070E-06 2.60324E-09
 2.14629E-01 5.23934E-07 8.49199E-08 4.32197E-06 3.87476E-07 1.78312E-04 1.73732E-04 2.19347E-06 2.01411E-07 4.23279E-10
 2.15531E-01 2.39740E-07 2.24705E-08 4.19208E-06 3.87227E-07 8.34646E-05 7.05939E-05 3.27038E-07 3.04507E-08 5.61892E-11
 2.16433E-01 9.16185E-08 4.87808E-09 4.13520E-06 3.87188E-07 3.20557E-05 2.40171E-05 3.99829E-08 3.76623E-09 6.08864E-12
 2.17335E-01 2.88333E-08 8.66872E-10 4.11529E-06 3.87183E-07 1.00865E-05 6.82549E-06 4.00480E-09 3.80860E-10 5.38507E-13
 2.18236E-01 7.43570E-09 1.25943E-10 4.10968E-06 3.87182E-07 2.59757E-06 1.61673E-06 3.28427E-10 3.14771E-11 3.88730E-14
 2.19138E-01 1.56838E-09 1.49482E-11 4.10841E-06 3.87182E-07 5.47100E-07 3.18378E-07 2.20412E-11 2.12549E-12 2.29024E-15
 2.20040E-01 2.70352E-10 1.44881E-12 4.10818E-06 3.87182E-07 9.41900E-08 5.19861E-08 1.21005E-12 1.17235E-13 1.10125E-16
 2.20942E-01 3.80687E-11 1.14638E-13 4.10815E-06 3.87182E-07 1.32500E-08 7.01948E-09 5.43268E-14 5.28094E-15 4.32175E-18
 2.21844E-01 4.37775E-12 7.40417E-15 4.10814E-06 3.87182E-07 1.52000E-09 7.81826E-10 1.99414E-15 1.94250E-16 1.38421E-19
 2.22745E-01 4.11057E-13 3.90313E-16 4.10814E-06 3.87182E-07 1.40000E-10 7.16711E-11 5.98330E-17 5.83394E-18 3.61836E-21
 2.23647E-01 3.15113E-14 1.67924E-17 4.10814E-06 3.87182E-07 1.00000E-11 5.39757E-12 1.46700E-18 1.43046E-19 7.71950E-23
 2.24549E-01 1.97197E-15 5.89603E-19 4.10814E-06 3.87182E-07 0.00000E+00 3.33440E-13 2.90000E-20 2.86331E-21 1.34411E-24
 2.25451E-01 1.00734E-16 1.68944E-20 4.10814E-06 3.87182E-07 0.00000E+00 1.68800E-14 0.00000E+00 4.67866E-23 1.91007E-26
 2.26353E-01 4.20024E-18 3.95050E-22 4.10814E-06 3.87182E-07 0.00000E+00 7.00000E-16 0.00000E+00 6.24039E-25 2.21529E-28
 2.27255E-01 1.42947E-19 7.53850E-24 4.10814E-06 3.87182E-07 0.00000E+00 2.00000E-17 0.00000E+00 6.79397E-27 2.09691E-30
 2.28156E-01 3.97069E-21 1.17391E-25 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 6.03730E-29 1.61994E-32
 2.29058E-01 9.00200E-23 1.49177E-27 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 4.37882E-31 1.02138E-34
 2.29960E-01 1.66567E-24 1.54697E-29 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.59214E-33 5.25586E-37
 2.30862E-01 2.51540E-26 1.30909E-31 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 1.25238E-35 2.20734E-39
 2.31764E-01 3.10022E-28 9.04002E-34 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 4.93836E-38 7.56598E-42
 2.32665E-01 3.11847E-30 5.09420E-36 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 1.58926E-40 2.11656E-44
 2.33567E-01 2.56008E-32 2.34256E-38 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 4.17411E-43 4.83241E-47
 2.34469E-01 1.71523E-34 8.79044E-41 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 8.94719E-46 9.00467E-50
 2.35371E-01 9.37891E-37 2.69176E-43 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 1.56516E-48 1.36944E-52
 2.36273E-01 4.18542E-39 6.72616E-46 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.23448E-51 1.69975E-55
 2.37174E-01 1.52434E-41 1.37152E-48 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.60338E-54 1.72186E-58
 2.38076E-01 4.53085E-44 2.28214E-51 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.47536E-57 1.42357E-61
 2.38978E-01 1.09909E-46 3.09875E-54 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 1.92077E-60 9.60578E-65
 2.39880E-01 2.17592E-49 3.43348E-57 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 1.21632E-63 5.28999E-68
 2.40782E-01 3.51564E-52 3.10445E-60 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 6.28574E-67 2.37764E-71
 2.41683E-01 4.63575E-55 2.29055E-63 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.65091E-70 8.72182E-75
 2.42585E-01 4.98873E-58 1.37910E-66 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 9.12355E-74 2.61118E-78
 2.43487E-01 4.38139E-61 6.77575E-70 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.56249E-77 6.38024E-82
 2.44389E-01 3.14041E-64 2.71657E-73 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 5.87345E-81 1.27235E-85
 2.45291E-01 1.83701E-67 8.88766E-77 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 4.32855E-82 2.07083E-89
 2.46192E-01 8.76980E-71 2.37277E-80 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.15515E-78 2.75074E-93
 2.47094E-01 3.41700E-74 5.16926E-84 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 8.77850E-75 2.98212E-97
 2.47996E-01 6.95975E-75 9.18974E-88 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.91786E-71 2.63857E-101
 2.48898E-01 1.88479E-71 1.33316E-91 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 7.91429E-68 1.90537E-105
 2.49800E-01 4.17169E-68 1.57820E-95 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 1.75171E-64 1.12295E-109
 2.50701E-01 7.53466E-65 1.52456E-99 4.10814E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 3.16383E-61 5.40140E-114
 2.51603E-01 1.11050E-61 1.20179E-103 4.10811E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 4.66303E-58 2.12041E-118
 2.52505E-01 1.33560E-58 7.73067E-108 4.10798E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 5.60822E-55 6.79362E-123
 2.53407E-01 1.31080E-55 4.05796E-112 4.10751E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 5.50408E-52 1.77643E-127
 2.54309E-01 1.04978E-52 1.73821E-116 4.10608E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 4.40806E-49 3.79108E-132
 2.55210E-01 6.86060E-50 6.07571E-121 4.10253E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.88079E-46 6.60301E-137
 2.56112E-01 3.65872E-47 1.73299E-125 4.09538E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 1.53631E-43 9.38614E-142
 2.57014E-01 1.59221E-44 4.03364E-130 4.08357E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 6.68573E-41 1.08892E-146
 2.57916E-01 5.65420E-42 7.66126E-135 4.06729E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.37422E-38 1.03124E-151
 2.58818E-01 1.63850E-39 1.18743E-139 4.04762E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 6.88013E-36 2.96604E-150
 2.59719E-01 3.87457E-37 1.50181E-144 4.02398E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 1.62695E-33 3.45841E-145
 2.60621E-01 7.47659E-35 1.54998E-149 3.98980E-06 3.87182E-07 0.00000E+00 0.00000E+00 0.00000E+00 3.13946E-31 3.29141E-140
 2.61523E-01 1.17730E-32 4.75438E-152 3.92872E-06 3.87180E-07 0.00000E+00 0.00000E+00 0.00000E+00 4.94354E-29 2.55680E-135
 2.62425E-01 1.51276E-30 4.88492E-147 3.81472E-06 3.87174E-07 0.00000E+00 0.00000E+00 0.00000E+00 6.35220E-27 1.62115E-130
 2.63327E-01 1.58620E-28 4.10696E-142 3.61969E-06 3.87157E-07 0.00000E+00 0.00000E+00 0.00000E+00 6.66060E-25 8.39004E-126
 2.64228E-01 1.35722E-26 2.81765E-137 3.32862E-06 3.87106E-07 0.00000E+00 0.00000E+00 0.00000E+00 5.69911E-23 3.54424E-121
 2.65130E-01 9.47638E-25 1.57745E-132 2.95626E-06 3.86954E-07 0.00000E+00 0.00000E+00 0.00000E+00 3.97929E-21 1.22209E-116
 2.66032E-01 5.39931E-23 7.20662E-128 2.55257E-06 3.86514E-07 0.00000E+00 0.00000E+00 0.00000E+00 2.26730E-19 3.43957E-112
 2.66934E-01 2.51037E-21 2.68665E-123 2.18590E-06 3.85359E-07 0.00000E+00 0.00000E+00 2.00000E-21 1.05419E-17 7.90194E-108
 2.67836E-01 9.52445E-20 8.17323E-119 1.90974E-06 3.82695E-07 0.00000E+00 0.00000E+00 6.50000E-20 3.99985E-16 1.48181E-103
 2.68737E-01 2.94881E-18 2.02899E-114 1.73859E-06 3.77296E-07 0.00000E+00 0.00000E+00 2.00800E-18 1.23847E-14 2.26822E-99
 2.69639E-01 7.45003E-17 4.11029E-110 1.65150E-06 3.67482E-07 0.00000E+00 0.00000E+00 5.07530E-17 3.12932E-13 2.83408E-95
 2.70541E-01 1.53594E-15 6.79466E-106 1.61480E-06 3.51100E-07 0.00000E+00 0.00000E+00 1.04674E-15 6.45289E-12 2.89056E-91
 2.71443E-01 2.58406E-14 9.16574E-102 1.60125E-06 3.25678E-07 0.00000E+00 0.00000E+00 1.76205E-14 1.08598E-10 2.40654E-87
 2.72345E-01 3.54765E-13 1.00895E-97 1.59516E-06 2.89279E-07 0.00000E+00 0.00000E+00 2.42140E-13 1.49172E-09 1.63551E-83
 2.73246E-01 3.97464E-12 9.06316E-94 1.58732E-06 2.42202E-07 0.00000E+00 0.00000E+00 2.71696E-12 1.67266E-08 9.07330E-80
 2.74148E-01 3.63398E-11 6.64342E-90 1.56744E-06 1.88389E-07 0.00000E+00 0.00000E+00 2.49019E-11 1.53138E-07 4.10897E-76
 2.75050E-01 2.71151E-10 3.97382E-86 1.51853E-06 1.34792E-07 0.00000E+00 0.00000E+00 1.86540E-10 1.14514E-06 1.51901E-72
 2.75952E-01 1.65124E-09 1.93968E-82 1.41817E-06 8.85685E-08 0.00000E+00 0.00000E+00 1.14318E-09 6.99817E-06 4.58408E-69
 2.76854E-01 8.20763E-09 7.72599E-79 1.25019E-06 5.41189E-08 0.00000E+00 0.00000E+00 5.73998E-09 3.49818E-05 1.12931E-65
 2.77756E-01 3.33044E-08 2.51121E-75 1.02104E-06 3.19973E-08 0.00000E+00 0.00000E+00 2.36687E-08 1.43233E-04 2.27118E-62
 2.78657E-01 1.10348E-07 8.88086E-72 7.65209E-07 1.98388E-08 0.00000E+00 0.00000E+00 8.77474E-08 4.81440E-04 3.72881E-59
 2.79559E-01 2.98664E-07 1.92217E-68 5.29684E-07 1.41761E-08 0.00000E+00 0.00000E+00 2.47043E-07 1.33289E-03 4.99774E-56
 2.80461E-01 6.60725E-07 2.54622E-65 3.49573E-07 1.19656E-08 0.00000E+00 0.00000E+00 5.32024E-07 3.05452E-03 5.46853E-53
 2.81363E-01 1.19595E-06 3.66979E-62 2.35471E-07 1.12495E-08 0.00000E+00 0.00000E+00 1.05266E-06 5.83449E-03 4.88502E-50
 2.82265E-01 1.77394E-06 4.31610E-59 1.78814E-07 1.10584E-08 0.00000E+00 0.00000E+00 1.77218E-06 9.37277E-03 3.56262E-47
 2.83166E-01 2.16149E-06 5.52311E-56 1.64634E-07 1.10166E-08 0.00000E+00 0.00000E+00 2.79493E-06 1.27930E-02 2.12122E-44
 2.84068E-01 2.17148E-06 4.32556E-53 1.80092E-07 1.10091E-08 0.00000E+00 0.00000E+00 3.48807E-06 1.49777E-02 1.03116E-41
 2.84970E-01 1.80836E-06 2.07332E-50 2.15487E-07 1.10080E-08 0.00000E+00 0.00000E+00 3.44257E-06 1.51314E-02 4.09256E-39
 2.85872E-01 1.25769E-06 1.08127E-47 2.61381E-07 1.10079E-08 0.00000E+00 0.00000E+00 3.17770E-06 1.31943E-02 1.32619E-36
 2.86774E-01 7.37368E-07 4.60155E-45 3.07326E-07 1.10079E-08 0.00000E+00 0.00000E+00 2.49064E-06 9.87497E-03 3.50888E-34
 2.87675E-01 3.68198E-07 1.59801E-42 3.44397E-07 1.10079E-08 0.00000E+00 0.00000E+00 1.64079E-06 6.28466E-03 7.58043E-32
 2.88577E-01 1.57994E-07 4.52854E-40 3.68638E-07 1.10079E-08 0.00000E+00 0.00000E+00 9.00526E-07 3.36645E-03 1.33719E-29
 2.89479E-01 5.85375E-08 1.04723E-37 3.81478E-07 1.10079E-08 0.00000E+00 0.00000E+00 4.08963E-07 1.50401E-03 1.92611E-27
 2.90381E-01 1.87039E-08 1.97619E-35 3.86981E-07 1.10079E-08 0.00000E+00 0.00000E+00 1.52939E-07 5.56465E-04 2.26552E-25
 2.91283E-01 5.11941E-09 3.04313E-33 3.88891E-07 1.10079E-08 0.00000E+00 0.00000E+00 4.69436E-08 1.69636E-04 2.17608E-23
 2.92184E-01 1.18863E-09 3.82399E-31 3.89427E-07 1.10079E-08 0.00000E+00 0.00000E+00 1.18014E-08 4.24610E-05 1.70692E-21
 2.93086E-01 2.31748E-10 3.92119E-29 3.89549E-07 1.10079E-08 0.00000E+00 0.00000E+00 2.42660E-09 8.70701E-06 1.09346E-19
 2.93988E-01 3.76153E-11 3.28112E-27 3.89572E-07 1.10079E-08 0.00000E+00 0.00000E+00 4.08014E-10 1.46082E-06 5.72091E-18
 2.94890E-01 5.04920E-12 2.24043E-25 3.89575E-07 1.10079E-08 0.00000E+00 0.00000E+00 5.93165E-11 2.03519E-07 2.44466E-16
 2.95792E-01 5.57909E-13 1.24837E-23 3.89575E-07 1.10079E-08 0.00000E+00 0.00000E+00 4.01753E-11 5.56375E-08 8.53267E-15
 2.96693E-01 5.05876E-14 5.67624E-22 3.89575E-07 1.10079E-08 0.00000E+00 0.00000E+00 2.80934E-10 2.77238E-07 2.43271E-13
 2.97595E-01 3.80868E-15 2.10612E-20 3.89575E-07 1.10079E-08 0.00000E+00 0.00000E+00 1.90060E-09 1.86952E-06 5.66580E-12
 2.98497E-01 1.22259E-15 6.37689E-19 3.89575E-07 1.10079E-08 0.00000E+00 0.00000E+00 1.05782E-08 1.04253E-05 1.07802E-10
 2.99399E-01 1.53348E-14 1.57558E-17 3.89575E-07 1.10079E-08 0.00000E+00 0.00000E+00 4.84446E-08 4.78116E-05 1.67578E-09
 3.00301E-01 1.93098E-13 3.17670E-16 3.89576E-07 1.10079E-08 0.00000E+00 0.00000E+00 1.83154E-07 1.80703E-04 2.12847E-08
 3.01202E-01 1.99050E-12 5.22658E-15 3.89580E-07 1.10079E-08 0.00000E+00 0.00000E+00 5.74839E-07 5.64423E-04 2.20911E-07
 3.02104E-01 1.67884E-11 7.01724E-14 3.89613E-07 1.10079E-08 0.00000E+00 0.00000E+00 1.51396E-06 1.46238E-03 1.87374E-06
 3.03006E-01 1.15881E-10 7.68819E-13 3.89827E-07 1.10079E-08 0.00000E+00 0.00000E+00 3.42446E-06 3.15837E-03 1.29896E-05
 3.03908E-01 6.54762E-10 6.87374E-12 3.90956E-07 1.10079E-08 0.00000E+00 0.00000E+00 6.99341E-06 5.72364E-03 7.36086E-05
 3.04810E-01 3.02926E-09 5.01510E-11 3.95763E-07 1.10079E-08 0.00000E+00 0.00000E+00 1.40713E-05 8.78075E-03 3.41020E-04
 3.05711E-01 1.14790E-08 2.98601E-10 4.12246E-07 1.10079E-08 0.00000E+00 7.00000E-17 3.03016E-05 1.15400E-02 1.29191E-03
 3.06613E-01 3.56397E-08 1.45092E-09 4.57527E-07 1.10079E-08 0.00000E+00 2.35000E-15 6.90297E-05 1.31956E-02 4.00297E-03
 3.07515E-01 9.07033E-08 5.75387E-09 5.56441E-07 1.10079E-08 0.00000E+00 6.12100E-14 1.51587E-04 1.33724E-02 1.01476E-02
 3.08417E-01 1.89345E-07 1.86244E-08 7.26026E-07 1.10079E-08 0.00000E+00 1.30437E-12 2.95337E-04 1.22273E-02 2.10545E-02
 3.09319E-01 3.24596E-07 4.92128E-08 9.48454E-07 1.10079E-08 0.00000E+00 2.27432E-11 4.88983E-04 1.01964E-02 3.57715E-02
 3.10220E-01 4.58213E-07 1.06185E-07 1.15825E-06 1.10079E-08 0.00000E+00 3.24592E-10 6.77173E-04 7.73433E-03 4.98003E-02
 3.11122E-01 5.36472E-07 1.87166E-07 1.27148E-06 1.10079E-08 0.00000E+00 3.79368E-09 7.85605E-04 5.25332E-03 5.68609E-02
 3.12024E-01 5.31452E-07 2.69705E-07 1.24253E-06 1.10079E-08 0.00000E+00 3.63283E-08 7.81499E-04 3.12488E-03 5.33097E-02
 3.12926E-01 4.69246E-07 3.18107E-07 1.09511E-06 1.10079E-08 0.00000E+00 2.85199E-07 7.11505E-04 1.59344E-03 4.11097E-02
 3.13828E-01 4.09842E-07 3.07711E-07 8.98132E-07 1.10079E-08 0.00000E+00 1.83678E-06 6.65952E-04 6.84932E-04 2.61408E-02
 3.14729E-01 3.94609E-07 2.44894E-07 7.16655E-07 1.10079E-08 0.00000E+00 9.71129E-06 6.94694E-04 2.45289E-04 1.37645E-02
 3.15631E-01 4.16293E-07 1.61158E-07 5.83067E-07 1.10079E-08 0.00000E+00 4.21831E-05 7.58339E-04 7.26250E-05 6.05115E-03
 3.16533E-01 4.45198E-07 8.83735E-08 4.99535E-07 1.10079E-08 1.00000E-11 1.50654E-04 7.66914E-04 1.76909E-05 2.26200E-03
 3.17435E-01 4.86064E-07 4.09563E-08 4.53796E-07 1.10079E-08 4.00000E-11 4.42724E-04 6.67288E-04 3.53459E-06 7.49208E-04
 3.18337E-01 6.02850E-07 1.69610E-08 4.31785E-07 1.10079E-08 1.90000E-10 1.07131E-03 4.91739E-04 5.78131E-07 2.37619E-04
 3.19238E-01 8.70925E-07 8.96585E-09 4.23199E-07 1.10079E-08 6.90000E-10 2.13599E-03 3.29982E-04 7.73199E-08 7.87522E-05
 3.20140E-01 1.28396E-06 1.25029E-08 4.22742E-07 1.10079E-08 2.10000E-09 3.51085E-03 2.69922E-04 8.44913E-09 2.75525E-05
 3.21042E-01 1.70278E-06 2.82870E-08 4.29711E-07 1.10082E-08 5.22000E-09 4.75888E-03 3.62328E-04 7.54023E-10 9.37138E-06
 3.21944E-01 1.92141E-06 6.00777E-08 4.46596E-07 1.10095E-08 1.05900E-08 5.32041E-03 6.08967E-04 5.49391E-11 2.83750E-06
 3.22846E-01 1.81912E-06 1.11622E-07 4.76408E-07 1.10154E-08 1.77000E-08 4.90583E-03 9.50388E-04 3.26754E-12 7.27346E-07
 3.23747E-01 1.45848E-06 1.86982E-07 5.19361E-07 1.10360E-08 2.48100E-08 3.72991E-03 1.26588E-03 1.58618E-13 1.54391E-07
 3.24649E-01 1.02766E-06 2.94507E-07 5.71032E-07 1.10941E-08 3.04100E-08 2.33729E-03 1.41878E-03 6.28409E-15 2.68984E-08
 3.25551E-01 6.91631E-07 4.44746E-07 6.23641E-07 1.12282E-08 3.53700E-08 1.20642E-03 1.33602E-03 2.03176E-16 3.83304E-09
 3.26453E-01 5.04429E-07 6.32532E-07 6.69483E-07 1.14814E-08 4.28800E-08 5.12569E-04 1.05688E-03 5.36078E-18 4.46144E-10
 3.27355E-01 4.39214E-07 8.15708E-07 7.03954E-07 1.18750E-08 5.46300E-08 1.79115E-04 7.02608E-04 1.16046E-19 4.23916E-11
 3.28257E-01 4.58973E-07 9.21846E-07 7.26487E-07 1.23832E-08 6.70300E-08 5.14387E-05 3.94382E-04 2.97692E-20 3.28744E-12
 3.29158E-01 5.42723E-07 8.92921E-07 7.39460E-07 1.29347E-08 7.25600E-08 1.21304E-05 1.91781E-04 1.01115E-18 2.08051E-13
 3.30060E-01 6.69887E-07 7.31730E-07 7.46222E-07 1.34456E-08 6.60900E-08 2.34731E-06 9.23431E-05 3.00766E-17 1.07448E-14
 3.30962E-01 8.12010E-07 5.03240E-07 7.49552E-07 1.38651E-08 4.96800E-08 3.72717E-07 7.12742E-05 7.30181E-16 4.52826E-16
 3.31864E-01 9.43500E-07 2.88799E-07 7.51166E-07 1.42166E-08 3.06000E-08 4.90039E-08 1.16934E-04 1.44694E-14 1.55730E-17
 3.32766E-01 1.04643E-06 1.37664E-07 7.51970E-07 1.46328E-08 1.54000E-08 6.06800E-09 2.35645E-04 2.34070E-13 4.37034E-19
 3.33667E-01 1.10557E-06 5.42976E-08 7.52460E-07 1.53896E-08 6.33000E-09 1.76113E-09 4.27127E-04 3.09187E-12 1.00083E-20
 3.34569E-01 1.11233E-06 1.76643E-08 7.53003E-07 1.69167E-08 2.12000E-09 1.89635E-09 6.56656E-04 3.33599E-11 1.87030E-22
 3.35471E-01 1.07349E-06 4.73042E-09 7.54093E-07 1.97375E-08 5.80000E-10 2.61972E-09 8.60772E-04 2.94170E-10 2.85209E-24
 3.36373E-01 1.00455E-06 1.06089E-09 7.57009E-07 2.43178E-08 1.30000E-10 3.42492E-09 9.91869E-04 2.12185E-09 3.54909E-26
 3.37275E-01 9.13754E-07 3.19474E-10 7.65763E-07 3.09927E-08 2.00000E-11 3.92322E-09 1.05173E-03 1.25359E-08 3.60391E-28
 3.38176E-01 8.02405E-07 6.78084E-10 7.91352E-07 4.04343E-08 0.00000E+00 3.78711E-09 1.07078E-03 6.07889E-08 2.98630E-30
 3.39078E-01 6.86365E-07 2.64903E-09 8.57907E-07 5.50777E-08 0.00000E+00 3.02407E-09 1.07087E-03 2.42720E-07 2.01928E-32
 3.39980E-01 6.13196E-07 8.96244E-09 1.00646E-06 8.10086E-08 0.00000E+00 1.98185E-09 1.07646E-03 8.01866E-07 1.11419E-34
 3.40882E-01 6.49194E-07 2.54703E-08 1.28878E-06 1.28199E-07 0.00000E+00 1.06604E-09 1.15267E-03 2.20753E-06 5.01681E-37
 3.41784E-01 8.37960E-07 6.12125E-08 1.74624E-06 2.05944E-07 0.00000E+00 5.03843E-10 1.38692E-03 5.11539E-06 1.84331E-39
 3.42685E-01 1.15884E-06 1.25279E-07 2.37873E-06 3.12473E-07 0.00000E+00 4.66327E-10 1.80700E-03 1.01090E-05 5.52676E-42
 3.43587E-01 1.51506E-06 2.19284E-07 3.12176E-06 4.25170E-07 0.00000E+00 1.93269E-09 2.32601E-03 1.72991E-05 1.76160E-44
 3.44489E-01 1.76337E-06 3.28335E-07 3.85699E-06 5.04792E-07 0.00000E+00 9.66234E-09 2.78067E-03 2.60245E-05 1.42304E-42
 3.45391E-01 1.77989E-06 4.18873E-07 4.46323E-06 5.19064E-07 0.00000E+00 4.03516E-08 3.02687E-03 3.48430E-05 4.03758E-40
 3.46293E-01 1.53426E-06 4.52203E-07 4.87584E-06 4.67860E-07 0.00000E+00 1.37773E-07 3.01281E-03 4.19368E-05 9.35195E-38
 3.47194E-01 1.11681E-06 4.09872E-07 5.10613E-06 3.83951E-07 4.00000E-11 3.84625E-07 2.78992E-03 4.61185E-05 1.76845E-35
 3.48096E-01 6.83597E-07 3.09609E-07 5.21162E-06 3.09276E-07 5.50000E-10 8.80317E-07 2.47310E-03 4.81745E-05 2.73049E-33
 3.48998E-01 3.65323E-07 1.93720E-07 5.25268E-06 2.78902E-07 6.71000E-09 1.66265E-06 2.17702E-03 5.11037E-05 3.44273E-31
 3.49900E-01 2.45676E-07 9.99431E-08 5.26972E-06 3.56317E-07 6.73100E-08 2.63070E-06 1.95720E-03 5.77182E-05 3.54532E-29
 3.50802E-01 5.71800E-07 4.24486E-08 5.28357E-06 7.83170E-07 5.52530E-07 3.60127E-06 1.79268E-03 6.68626E-05 2.98254E-27
 3.51703E-01 2.81871E-06 1.53098E-08 5.30561E-06 2.35918E-06 3.70958E-06 4.51627E-06 1.63060E-03 7.26047E-05 2.05024E-25
 3.52605E-01 1.38955E-05 7.41103E-09 5.35134E-06 7.16266E-06 2.03716E-05 5.55406E-06 1.45236E-03 6.87641E-05 1.15197E-23
 3.53507E-01 5.87033E-05 1.54116E-08 5.46189E-06 1.94675E-05 9.15154E-05 6.90570E-06 1.28595E-03 5.46879E-05 5.29252E-22
 3.54409E-01 2.05332E-04 5.30921E-08 5.73771E-06 4.61633E-05 3.36340E-04 8.43407E-06 1.15470E-03 3.59149E-05 1.98912E-20
 3.55311E-01 5.88683E-04 1.58735E-07 6.37624E-06 9.54949E-05 1.01144E-03 9.71048E-06 1.04537E-03 1.93416E-05 6.11890E-19
 3.56212E-01 1.37916E-03 3.90206E-07 7.76304E-06 1.73232E-04 2.48934E-03 1.04476E-05 9.50941E-04 8.51778E-06 1.54165E-17
 3.57114E-01 2.63810E-03 7.89303E-07 1.10787E-05 2.77323E-04 5.01681E-03 1.07574E-05 9.29296E-04 3.06749E-06 3.18375E-16
 3.58016E-01 4.12048E-03 1.32963E-06 2.11696E-05 3.95553E-04 8.28780E-03 1.09373E-05 1.09009E-03 9.26927E-07 5.39449E-15
 3.58918E-01 5.26198E-03 1.93884E-06 5.55944E-05 5.12462E-04 1.12498E-02 1.11826E-05 1.51296E-03 3.63840E-07 7.50793E-14
 3.59820E-01 5.52275E-03 2.71980E-06 1.63631E-04 6.28248E-04 1.26133E-02 1.15907E-05 2.17556E-03 6.99272E-07 8.59536E-13
 3.60721E-01 4.85743E-03 4.36942E-06 4.47939E-04 7.87267E-04 1.18175E-02 1.22841E-05 2.95872E-03 2.40955E-06 8.10868E-12
 3.61623E-01 3.82630E-03 8.50588E-06 1.05646E-03 1.11248E-03 9.48753E-03 1.35000E-05 3.74276E-03 7.28556E-06 6.31767E-11
 3.62525E-01 3.19301E-03 1.73509E-05 2.10574E-03 1.83261E-03 6.87392E-03 1.58004E-05 4.53160E-03 1.84423E-05 4.07717E-10
 3.63427E-01 3.37682E-03 3.24881E-05 3.56680E-03 3.25339E-03 4.92804E-03 2.01541E-05 5.47577E-03 3.94296E-05 2.18797E-09
 3.64329E-01 4.20160E-03 5.38395E-05 5.25301E-03 5.59882E-03 3.90818E-03 2.72819E-05 6.72445E-03 7.24469E-05 9.81391E-09
 3.65230E-01 5.07091E-03 8.18123E-05 7.00258E-03 8.73408E-03 3.57570E-03 3.69837E-05 8.22732E-03 1.17388E-04 3.70398E-08
 3.66132E-01 5.41640E-03 1.25225E-04 8.88483E-03 1.19867E-02 3.51207E-03 5.05579E-05 9.70420E-03 1.73743E-04 1.18614E-07
 3.67034E-01 5.13463E-03 2.14680E-04 1.11266E-02 1.43454E-02 3.34492E-03 7.98444E-05 1.08356E-02 2.44866E-04 3.25379E-07
 3.67936E-01 4.73942E-03 4.18270E-04 1.37698E-02 1.50003E-02 2.91816E-03 1.62393E-04 1.14734E-02 3.41307E-04 7.72140E-07
 3.68838E-01 5.15367E-03 8.50721E-04 1.64637E-02 1.38017E-02 2.33196E-03 3.73010E-04 1.16576E-02 4.78626E-04 1.59925E-06
 3.69739E-01 7.28154E-03 1.65270E-03 1.86555E-02 1.12865E-02 1.78234E-03 8.11167E-04 1.14746E-02 6.65092E-04 2.91314E-06
 3.70641E-01 1.15247E-02 2.91656E-03 1.99439E-02 8.34807E-03 1.36314E-03 1.54716E-03 1.09607E-02 8.82468E-04 4.70470E-06
 3.71543E-01 1.73693E-02 4.58589E-03 2.02290E-02 5.86893E-03 1.03703E-03 2.54496E-03 1.01314E-02 1.08456E-03 6.82011E-06
 3.72445E-01 2.32858E-02 6.41432E-03 1.96361E-02 4.53199E-03 7.46310E-04 3.62332E-03 9.04497E-03 1.25080E-03 9.05934E-06
 3.73347E-01 2.72250E-02 8.02439E-03 1.84317E-02 4.80837E-03 4.89849E-04 4.50555E-03 7.81861E-03 1.49994E-03 1.13379E-05
 3.74248E-01 2.76342E-02 9.00685E-03 1.69991E-02 6.96121E-03 2.99553E-04 4.93449E-03 6.60823E-03 2.19446E-03 1.36966E-05
 3.75150E-01 2.43845E-02 9.02671E-03 1.57527E-02 1.09211E-02 1.88233E-04 4.78394E-03 5.57846E-03 3.88946E-03 1.60680E-05
 3.76052E-01 1.89045E-02 7.98342E-03 1.49512E-02 1.60944E-02 1.37242E-04 4.12476E-03 4.86078E-03 6.97805E-03 1.80523E-05
 3.76954E-01 1.33834E-02 6.16425E-03 1.45512E-02 2.13455E-02 1.18436E-04 3.22885E-03 4.50809E-03 1.11160E-02 1.90161E-05
 3.77856E-01 9.61615E-03 4.18971E-03 1.42336E-02 2.52973E-02 1.14297E-04 2.48779E-03 4.47270E-03 1.49450E-02 1.84618E-05
 3.78758E-01 8.23612E-03 2.73054E-03 1.35832E-02 2.68325E-02 1.19777E-04 2.25140E-03 4.62009E-03 1.66957E-02 1.63674E-05
 3.79659E-01 8.65787E-03 2.26242E-03 1.23024E-02 2.55283E-02 1.34043E-04 2.65418E-03 4.77597E-03 1.54337E-02 1.33840E-05
 3.80561E-01 9.59834E-03 3.03096E-03 1.03560E-02 2.18026E-02 1.55857E-04 3.52433E-03 4.81646E-03 1.18009E-02 1.08891E-05
 3.81463E-01 9.85229E-03 5.07662E-03 7.98974E-03 1.66985E-02 1.87725E-04 4.44807E-03 4.77257E-03 7.47391E-03 1.06784E-05
 3.82365E-01 8.90487E-03 8.15659E-03 5.61990E-03 1.14373E-02 2.43721E-04 4.98925E-03 4.82986E-03 3.93757E-03 1.40314E-05
 3.83267E-01 7.02234E-03 1.17009E-02 3.64357E-03 6.97886E-03 3.50039E-04 4.94282E-03 5.14459E-03 1.74773E-03 2.04302E-05
 3.84168E-01 4.85340E-03 1.50139E-02 2.27021E-03 3.78050E-03 5.27276E-04 4.43010E-03 5.61610E-03 6.79736E-04 2.70437E-05
 3.85070E-01 2.95669E-03 1.75962E-02 1.47651E-03 1.81513E-03 7.59296E-04 3.75073E-03 5.87693E-03 2.58283E-04 3.00953E-05
 3.85972E-01 1.59021E-03 1.92431E-02 1.09513E-03 7.74214E-04 9.82485E-04 3.13194E-03 5.56604E-03 1.15922E-04 2.75403E-05
 3.86874E-01 7.53303E-04 1.98749E-02 9.42173E-04 2.97143E-04 1.12615E-03 2.61989E-03 4.63434E-03 6.61570E-05 2.06058E-05
 3.87776E-01 3.14144E-04 1.94108E-02 8.90270E-04 1.07095E-04 1.17286E-03 2.18879E-03 3.38265E-03 4.10460E-05 1.25873E-05
 3.88677E-01 1.16852E-04 1.78712E-02 8.73923E-04 4.02331E-05 1.16521E-03 1.89100E-03 2.21749E-03 2.36358E-05 6.27534E-06
 3.89579E-01 4.06962E-05 1.55295E-02 8.64754E-04 1.80601E-05 1.15401E-03 1.85718E-03 1.38181E-03 1.17613E-05 2.55305E-06
 3.90481E-01 1.48101E-05 1.29047E-02 8.48055E-04 9.80673E-06 1.18723E-03 2.15918E-03 8.82263E-04 4.92345E-06 8.47597E-07
 3.91383E-01 6.40736E-06 1.06060E-02 8.10477E-04 5.68648E-06 1.41365E-03 2.68554E-03 6.02954E-04 1.71495E-06 2.29628E-07
 3.92285E-01 3.30159E-06 9.12893E-03 7.38877E-04 3.17085E-06 2.21513E-03 3.15479E-03 4.49343E-04 4.94661E-07 5.07653E-08
 3.93186E-01 1.82000E-06 8.64627E-03 6.27533E-04 1.62976E-06 4.14723E-03 3.29421E-03 4.05910E-04 1.17954E-07 9.15830E-09
 3.94088E-01 1.03068E-06 8.86766E-03 4.86110E-04 7.73431E-07 7.51618E-03 3.06723E-03 5.03038E-04 2.34686E-08 1.34885E-09
 3.94990E-01 6.49723E-07 9.12371E-03 3.38412E-04 3.62843E-07 1.17614E-02 2.73182E-03 7.48432E-04 5.87365E-09 1.67907E-10
 3.95892E-01 5.00021E-07 8.72659E-03 2.10296E-04 2.05938E-07 1.52914E-02 2.63404E-03 1.07676E-03 1.61645E-08 6.22312E-11
 3.96794E-01 4.54133E-07 7.39448E-03 1.16834E-04 1.70453E-07 1.63303E-02 2.91455E-03 1.35491E-03 9.67075E-08 2.98012E-10
 3.97695E-01 4.42227E-07 5.40522E-03 5.86824E-05 1.75751E-07 1.42752E-02 3.40823E-03 1.45199E-03 4.90284E-07 1.56067E-09
 3.98597E-01 4.34558E-07 3.35586E-03 2.73112E-05 1.77546E-07 1.02040E-02 3.81783E-03 1.33248E-03 2.04229E-06 6.74854E-09
 3.99499E-01 4.22195E-07 1.75131E-03 1.22546E-05 1.57873E-07 5.96630E-03 3.95854E-03 1.10083E-03 6.98306E-06 2.40180E-08
 4.00401E-01 4.12125E-07 7.62499E-04 5.50742E-06 1.19114E-07 2.86051E-03 3.81920E-03 9.60349E-04 1.95898E-05 7.04106E-08
 4.01303E-01 4.22946E-07 2.75707E-04 2.48144E-06 7.51000E-08 1.13332E-03 3.44270E-03 1.10944E-03 4.50685E-05 1.70152E-07
 4.02204E-01 4.66635E-07 8.30791E-05 1.07054E-06 3.92370E-08 3.79270E-04 2.84946E-03 1.62327E-03 8.49918E-05 3.39148E-07
 4.03106E-01 5.26676E-07 2.17677E-05 4.16559E-07 1.68973E-08 1.13196E-04 2.09987E-03 2.37517E-03 1.31325E-04 5.57773E-07
 4.04008E-01 5.59101E-07 6.10158E-06 1.39774E-07 5.97702E-09 3.32956E-05 1.35081E-03 3.07012E-03 1.66191E-04 7.57018E-07
 4.04910E-01 5.24245E-07 2.74079E-06 3.94005E-08 1.73259E-09 1.05523E-05 7.98188E-04 3.41779E-03 1.72179E-04 8.47790E-07
 4.05812E-01 4.20900E-07 1.80383E-06 9.20361E-09 4.10957E-10 3.50851E-06 5.95161E-04 3.32636E-03 1.45987E-04 7.83167E-07
 4.06713E-01 2.87883E-07 1.17103E-06 1.76909E-09 7.96820E-11 1.10071E-06 8.45664E-04 2.93041E-03 1.01266E-04 5.96434E-07
 4.07615E-01 1.73017E-07 6.45601E-07 2.78800E-10 1.26216E-11 3.00560E-07 1.61439E-03 2.43255E-03 5.74521E-05 3.74195E-07
 4.08517E-01 1.01812E-07 2.93338E-07 3.59525E-11 1.63261E-12 6.96600E-08 2.84523E-03 1.94457E-03 2.66517E-05 1.93240E-07
 4.09419E-01 7.06475E-08 1.09254E-07 3.78951E-12 1.72405E-13 1.92200E-08 4.23195E-03 1.47718E-03 1.01070E-05 8.20671E-08
 4.10321E-01 6.05839E-08 3.33165E-08 3.26268E-13 1.48610E-14 3.68300E-08 5.24084E-03 1.02736E-03 3.13267E-06 2.86364E-08
 4.11222E-01 5.43663E-08 8.31423E-09 2.29371E-14 1.04552E-15 1.64590E-07 5.39375E-03 6.29635E-04 7.93450E-07 8.20272E-09
 4.12124E-01 4.52309E-08 1.69738E-09 1.31636E-15 6.00297E-17 6.56500E-07 4.61211E-03 3.30682E-04 1.64201E-07 1.92720E-09
 4.13026E-01 3.63670E-08 2.83425E-10 6.16623E-17 2.81279E-18 2.21898E-06 3.28297E-03 1.46213E-04 2.77602E-08 3.71102E-10
 4.13928E-01 3.57169E-08 3.89563E-11 2.35741E-18 1.07555E-19 6.33637E-06 1.97786E-03 5.38358E-05 3.83368E-09 5.85276E-11
 4.14830E-01 5.15584E-08 6.88547E-12 7.35519E-20 3.35616E-21 1.52929E-05 1.11697E-03 1.63966E-05 4.32424E-10 7.55563E-12
 4.15731E-01 9.07424E-08 2.05504E-11 1.87276E-21 8.54601E-23 3.16022E-05 8.67828E-04 4.12154E-06 3.98355E-11 7.97990E-13
 4.16633E-01 1.56582E-07 1.29180E-10 3.89127E-23 1.77580E-24 5.90683E-05 1.25544E-03 8.84068E-07 2.99685E-12 6.89207E-14
 4.17535E-01 2.42704E-07 6.75589E-10 6.59802E-25 3.01112E-26 1.15218E-04 2.26255E-03 2.50816E-07 1.84106E-13 4.86593E-15
 4.18437E-01 3.26346E-07 2.88601E-09 9.12947E-27 4.16647E-28 2.74799E-04 3.78977E-03 2.71810E-07 9.23540E-15 2.80744E-16
 4.19339E-01 3.72961E-07 1.00727E-08 1.03082E-28 4.70449E-30 7.56321E-04 5.54292E-03 4.82470E-07 3.78277E-16 1.32333E-17
 4.20240E-01 3.57298E-07 2.87451E-08 9.49791E-31 4.33471E-32 1.99747E-03 7.01339E-03 7.47682E-07 1.26506E-17 5.09505E-19
 4.21142E-01 2.84543E-07 6.71627E-08 7.14129E-33 3.25920E-34 4.54354E-03 7.67160E-03 9.51702E-07 3.45422E-19 1.60204E-20
 4.22044E-01 1.88222E-07 1.28762E-07 4.38156E-35 1.99970E-36 8.57705E-03 7.32770E-03 9.93641E-07 7.70031E-21 4.11318E-22
 4.22946E-01 1.05919E-07 2.03274E-07 2.19373E-37 1.00120E-38 1.33046E-02 6.39554E-03 8.55444E-07 1.40145E-22 8.62207E-24
 4.23848E-01 5.77595E-08 2.65734E-07 8.96277E-40 4.09052E-41 1.69372E-02 5.75309E-03 6.15031E-07 2.08232E-24 1.47547E-25
 4.24749E-01 4.43882E-08 2.90083E-07 2.98816E-42 1.36377E-43 1.77325E-02 6.19840E-03 3.80218E-07 2.52586E-26 2.06112E-27
 4.25651E-01 5.73055E-08 2.67494E-07 8.12956E-45 3.71026E-46 1.53367E-02 7.88439E-03 2.14667E-07 2.50125E-28 2.35017E-29
 4.26553E-01 8.68866E-08 2.11261E-07 1.80482E-47 8.23704E-49 1.10365E-02 1.01501E-02 1.21077E-07 2.02201E-30 2.18723E-31
 4.27455E-01 1.22971E-07 1.44791E-07 3.26966E-50 1.49225E-51 6.67620E-03 1.18637E-02 7.22525E-08 1.33438E-32 1.66139E-33
 4.28357E-01 1.52780E-07 8.67937E-08 4.83364E-53 2.20603E-54 3.43979E-03 1.20690E-02 4.38102E-08 7.18862E-35 1.02995E-35
 4.29259E-01 1.62965E-07 4.54619E-08 5.83108E-56 2.66125E-57 1.53064E-03 1.05354E-02 2.46713E-08 3.16135E-37 5.21096E-38
 4.30160E-01 1.47003E-07 2.06150E-08 5.74018E-59 2.61977E-60 5.94492E-04 7.84452E-03 1.20665E-08 1.13489E-39 2.15160E-40
 4.31062E-01 1.10843E-07 7.98330E-09 4.61111E-62 2.10447E-63 2.02122E-04 4.96178E-03 4.95431E-09 3.32576E-42 7.25010E-43
 4.31964E-01 6.92780E-08 2.60484E-09 3.02264E-65 1.37951E-66 5.97985E-05 2.65581E-03 1.68208E-09 7.95566E-45 1.99368E-45
 4.32866E-01 3.56882E-08 7.08340E-10 1.61686E-68 7.37920E-70 1.52137E-05 1.19821E-03 4.69105E-10 1.55348E-47 4.47396E-48
 4.33768E-01 1.50972E-08 1.59292E-10 7.05762E-72 3.22104E-73 3.28588E-06 4.53821E-04 1.07130E-10 2.47615E-50 8.19312E-51
 4.34669E-01 5.23235E-09 2.94720E-11 2.51390E-75 1.14732E-76 5.96320E-07 1.43710E-04 2.00031E-11 3.22170E-53 1.22440E-53
 4.35571E-01 1.48352E-09 4.47188E-12 7.30701E-79 3.33486E-80 9.03900E-08 3.79002E-05 3.05124E-12 3.42159E-56 1.49318E-56
 4.36473E-01 3.43786E-10 5.55365E-13 1.73314E-82 7.90992E-84 1.14300E-08 8.29417E-06 3.80065E-13 2.96622E-59 1.48598E-59
 4.37375E-01 6.50786E-11 5.63844E-14 3.35453E-86 1.53098E-87 1.21000E-09 1.50135E-06 3.86483E-14 2.09897E-62 1.20677E-62
 4.38277E-01 1.00598E-11 4.67653E-15 5.29823E-90 2.41807E-91 1.10000E-10 2.24162E-07 3.20798E-15 1.21236E-65 7.99732E-66
 4.39178E-01 1.26952E-12 3.16731E-16 6.82863E-94 3.11653E-95 1.00000E-11 2.75430E-08 2.17333E-16 5.71587E-69 4.32486E-69
 4.40080E-01 1.30777E-13 1.75127E-17 7.18188E-98 3.27775E-99 0.00000E+00 2.77982E-09 1.20170E-17 2.19964E-72 1.90856E-72
 4.40982E-01 1.09958E-14 7.90403E-19 6.16375E-102 2.81308E-103 0.00000E+00 2.30103E-10 5.42000E-19 6.90932E-76 6.87300E-76
 4.41884E-01 7.54566E-16 2.91164E-20 4.31673E-106 1.97012E-107 0.00000E+00 1.56035E-11 2.00000E-20 1.77147E-79 2.01972E-79
 4.42786E-01 4.22599E-17 8.75386E-22 2.46699E-110 1.12591E-111 0.00000E+00 8.65990E-13 1.00000E-21 3.70717E-83 4.84331E-83
 4.43687E-01 1.93157E-18 2.14794E-23 1.15049E-114 5.25073E-116 0.00000E+00 3.93100E-14 0.00000E+00 6.33227E-87 9.47756E-87
 4.44589E-01 7.20500E-20 4.30126E-25 4.37824E-119 1.99819E-120 0.00000E+00 1.46000E-15 0.00000E+00 8.82838E-91 1.51340E-90
 4.45491E-01 2.19328E-21 7.02937E-27 1.35963E-123 6.20522E-125 0.00000E+00 4.00000E-17 0.00000E+00 1.00463E-94 1.97204E-94
 4.46393E-01 5.44867E-23 9.37524E-29 3.44542E-128 1.57246E-129 0.00000E+00 0.00000E+00 0.00000E+00 9.33104E-99 2.09692E-98
 4.47295E-01 1.10463E-24 1.02045E-30 7.12470E-133 3.25165E-134 0.00000E+00 0.00000E+00 0.00000E+00 7.07379E-103 1.81950E-102
 4.48196E-01 1.82757E-26 9.06445E-33 1.20225E-137 5.48696E-139 0.00000E+00 0.00000E+00 0.00000E+00 4.37692E-107 1.28832E-106
 4.49098E-01 2.46752E-28 6.57103E-35 1.65548E-142 7.55547E-144 0.00000E+00 0.00000E+00 0.00000E+00 2.21044E-111 7.44385E-111
 4.50000E-01 2.71877E-30 3.88746E-37 1.86019E-147 8.48974E-149 0.00000E+00 0.00000E+00 0.00000E+00 9.11124E-116 3.50975E-115
"""
    ll = _np.genfromtxt(io.StringIO(thedata),names=True)
    fgs=['CHali','CHaro','CH2','CH3','NH','NH2','NH3','OH','SH']
    return ll['Energy'].copy()*1e3, dict((k,ll[k].copy()) for k in fgs)
