from neo.Utils.NeoTestCase import NeoTestCase
import os
import binascii
from neo.IO.Helper import Helper
from neo.Core.Block import Block
from neocore.Cryptography.Crypto import Crypto
from neocore.Cryptography.MerkleTree import MerkleTree
from neocore.Fixed8 import Fixed8
from neo.Core.TX.StateTransaction import StateTransaction
from neo.Core.State.StateDescriptor import StateDescriptor, StateType


class BlocksTestCase(NeoTestCase):
    # raw block ( block # 343892 )
    rawblock = b'00000000b7def681f0080262aa293071c53b41fc3146b196067243700b68acd059734fd19543108bf9ddc738cbee2ed1160f153aa0d057f062de0aa3cbb64ba88735c23d43667e59543f050095df82b02e324c5ff3812db982f3b0089a21a278988efeec6a027b2501fd450140113ac66657c2f544e8ad13905fcb2ebaadfef9502cbefb07960fbe56df098814c223dcdd3d0efa0b43a9459e654d948516dcbd8b370f50fbecfb8b411d48051a408500ce85591e516525db24065411f6a88f43de90fa9c167c2e6f5af43bc84e65e5a4bb174bc83a19b6965ff10f476b1b151ae15439a985f33916abc6822b0bb140f4aae522ffaea229987a10d01beec826c3b9a189fe02aa82680581b78f3df0ea4d3f93ca8ea35ffc90f15f7db9017f92fafd9380d9ba3237973cf4313cf626fc40e30e50e3588bd047b39f478b59323868cd50c7ab54355d8245bf0f1988d37528f9bbfc68110cf917debbdbf1f4bdd02cdcccdc3269fdf18a6c727ee54b6934d840e43918dd1ec6123550ec37a513e72b34b2c2a3baa510dec3037cbef2fa9f6ed1e7ccd1f3f6e19d4ce2c0919af55249a970c2685217f75a5589cf9e54dff8449af155210209e7fd41dfb5c2f8dc72eb30358ac100ea8c72da18847befe06eade68cebfcb9210327da12b5c40200e9f65569476bbff2218da4f32548ff43b6387ec1416a231ee821034ff5ceeac41acf22cd5ed2da17a6df4dd8358fcb2bfb1a43208ad0feaab2746b21026ce35b29147ad09e4afe4ec4a7319095f08198fa8babbe3c56e970b143528d2221038dddc06ce687677a53d54f096d2591ba2302068cf123c1f2d75c2dddc542557921039dafd8571a641058ccc832c5e2111ea39b09c0bde36050914384f7a48bce9bf92102d02b1873a0863cd042cc717da31cea0d7cf9db32b74d4c72c01b0011503e2e2257ae01000095df82b000000000'

    rawblock_hex = b'\x00\x00\x00\x00\xb7\xde\xf6\x81\xf0\x08\x02b\xaa)0q\xc5;A\xfc1F\xb1\x96\x06rCp\x0bh\xac\xd0YsO\xd1\x95C\x10\x8b\xf9\xdd\xc78\xcb\xee.\xd1\x16\x0f\x15:\xa0\xd0W\xf0b\xde\n\xa3\xcb\xb6K\xa8\x875\xc2=Cf~YT?\x05\x00\x95\xdf\x82\xb0.2L_\xf3\x81-\xb9\x82\xf3\xb0\x08\x9a!\xa2x\x98\x8e\xfe\xecj\x02{%\x01\xfdE\x01@\x11:\xc6fW\xc2\xf5D\xe8\xad\x13\x90_\xcb.\xba\xad\xfe\xf9P,\xbe\xfb\x07\x96\x0f\xbeV\xdf\t\x88\x14\xc2#\xdc\xdd=\x0e\xfa\x0bC\xa9E\x9eeM\x94\x85\x16\xdc\xbd\x8b7\x0fP\xfb\xec\xfb\x8bA\x1dH\x05\x1a@\x85\x00\xce\x85Y\x1eQe%\xdb$\x06T\x11\xf6\xa8\x8fC\xde\x90\xfa\x9c\x16|.oZ\xf4;\xc8Ne\xe5\xa4\xbb\x17K\xc8:\x19\xb6\x96_\xf1\x0fGk\x1b\x15\x1a\xe1T9\xa9\x85\xf39\x16\xab\xc6\x82+\x0b\xb1@\xf4\xaa\xe5"\xff\xae\xa2)\x98z\x10\xd0\x1b\xee\xc8&\xc3\xb9\xa1\x89\xfe\x02\xaa\x82h\x05\x81\xb7\x8f=\xf0\xeaM?\x93\xca\x8e\xa3_\xfc\x90\xf1_}\xb9\x01\x7f\x92\xfa\xfd\x93\x80\xd9\xba27\x97<\xf41<\xf6&\xfc@\xe3\x0eP\xe3X\x8b\xd0G\xb3\x9fG\x8bY28h\xcdP\xc7\xabT5]\x82E\xbf\x0f\x19\x88\xd3u(\xf9\xbb\xfch\x11\x0c\xf9\x17\xde\xbb\xdb\xf1\xf4\xbd\xd0,\xdc\xcc\xdc2i\xfd\xf1\x8alr~\xe5Ki4\xd8@\xe49\x18\xdd\x1e\xc6\x125P\xec7\xa5\x13\xe7+4\xb2\xc2\xa3\xba\xa5\x10\xde\xc3\x03|\xbe\xf2\xfa\x9fn\xd1\xe7\xcc\xd1\xf3\xf6\xe1\x9dL\xe2\xc0\x91\x9a\xf5RI\xa9p\xc2hR\x17\xf7ZU\x89\xcf\x9eT\xdf\xf8D\x9a\xf1U!\x02\t\xe7\xfdA\xdf\xb5\xc2\xf8\xdcr\xeb05\x8a\xc1\x00\xea\x8cr\xda\x18\x84{\xef\xe0n\xad\xe6\x8c\xeb\xfc\xb9!\x03\'\xda\x12\xb5\xc4\x02\x00\xe9\xf6UiGk\xbf\xf2!\x8d\xa4\xf3%H\xffC\xb68~\xc1Aj#\x1e\xe8!\x03O\xf5\xce\xea\xc4\x1a\xcf"\xcd^\xd2\xda\x17\xa6\xdfM\xd85\x8f\xcb+\xfb\x1aC \x8a\xd0\xfe\xaa\xb2tk!\x02l\xe3[)\x14z\xd0\x9eJ\xfeN\xc4\xa71\x90\x95\xf0\x81\x98\xfa\x8b\xab\xbe<V\xe9p\xb1CR\x8d"!\x03\x8d\xdd\xc0l\xe6\x87gzS\xd5O\tm%\x91\xba#\x02\x06\x8c\xf1#\xc1\xf2\xd7\\-\xdd\xc5BUy!\x03\x9d\xaf\xd8W\x1ad\x10X\xcc\xc82\xc5\xe2\x11\x1e\xa3\x9b\t\xc0\xbd\xe3`P\x91C\x84\xf7\xa4\x8b\xce\x9b\xf9!\x02\xd0+\x18s\xa0\x86<\xd0B\xccq}\xa3\x1c\xea\r|\xf9\xdb2\xb7MLr\xc0\x1b\x00\x11P>."W\xae\x01\x00\x00\x95\xdf\x82\xb0\x00\x00\x00\x00'
    rb_hash = b'922ba0c0d06afbeec4c50b0541a29153feaa46c5d7304e7bf7f40870d9f3aeb0'
    rb_prev = b'd14f7359d0ac680b7043720696b14631fc413bc5713029aa620208f081f6deb7'
    rb_merlke = b'3dc23587a84bb6cba30ade62f057d0a03a150f16d12eeecb38c7ddf98b104395'
    rb_ts = 1501455939
    rb_h = 343892
    rb_nonce = int.from_bytes(binascii.unhexlify(b'5f4c322eb082df95'), 'big')
    rconsenusdata = 6866918707944415125

    rblock_tx_id = b'3dc23587a84bb6cba30ade62f057d0a03a150f16d12eeecb38c7ddf98b104395'
    rblock_tx_nonce = 2961366933
    rblock_inputs = []
    rblock_outputs = []

    # raw block 2 ( block #1)
    b2raw = b'00000000ef1f8f66a16fba100ed760f4ac6aa5a0d0bb8f4a0e92705b106761ef181718b3d0765298ceb5f57de7d2b0dab00ed25be4134706ada2d90adb8b7e3aba323a8e1abd125901000000d11f7a289214bdaff3812db982f3b0089a21a278988efeec6a027b2501fd450140884037dd265cb5f5a54802f53c2c8593b31d5b8a9c0bad4c7e366b153d878989d168080ac36b930036a9eb966b48c70bb41792e698fa021116f27c09643563b840e83ab14404d964a91dbac45f5460e88ad57196b1779478e3475334af8c1b49cd9f0213257895c60b5b92a4800eb32d785cbb39ae1f022528943909fd37deba63403677848bf98cc9dbd8fbfd7f2e4f34471866ea82ca6bffbf0f778b6931483700c17829b4bd066eb04983d3aac0bd46b9c8d03a73a8e714d3119de93cd9522e314054d16853b22014190063f77d9edf6fbccefcf71fffd1234f688823b4e429ae5fa639d0a664c842fbdfcb4d6e21f39d81c23563b92cffa09696d93c95bc4893a6401a43071d00d3e854f7f1f321afa7d5301d36f2195dc1e2643463f34ae637d2b02ae0eb11d4256c507a4f8304cea6396a7fce640f50acb301c2f6336d27717e84f155210209e7fd41dfb5c2f8dc72eb30358ac100ea8c72da18847befe06eade68cebfcb9210327da12b5c40200e9f65569476bbff2218da4f32548ff43b6387ec1416a231ee821034ff5ceeac41acf22cd5ed2da17a6df4dd8358fcb2bfb1a43208ad0feaab2746b21026ce35b29147ad09e4afe4ec4a7319095f08198fa8babbe3c56e970b143528d2221038dddc06ce687677a53d54f096d2591ba2302068cf123c1f2d75c2dddc542557921039dafd8571a641058ccc832c5e2111ea39b09c0bde36050914384f7a48bce9bf92102d02b1873a0863cd042cc717da31cea0d7cf9db32b74d4c72c01b0011503e2e2257ae010000d11f7a2800000000'
    b2hash = '0012f8566567a9d7ddf25acb5cf98286c9703297de675d01ba73fbfe6bcb841c'
    b2prev_hash = b'b3181718ef6167105b70920e4a8fbbd0a0a56aacf460d70e10ba6fa1668f1fef'
    b2height = 1
    b2merkle = '8e3a32ba3a7e8bdb0ad9a2ad064713e45bd20eb0dab0d2e77df5b5ce985276d0'
    b2nonce = int.from_bytes(binascii.unhexlify('afbd1492287a1fd1'), 'big')
    b2nextconsensus = 'AdyQbbn6ENjqWDa5JNYMwN3ikNcA4JeZdk'
    b2timestamp = 1494400282

    b2invocation = b'40884037dd265cb5f5a54802f53c2c8593b31d5b8a9c0bad4c7e366b153d878989d168080ac36b930036a9eb966b48c70bb41792e698fa021116f27c09643563b840e83ab14404d964a91dbac45f5460e88ad57196b1779478e3475334af8c1b49cd9f0213257895c60b5b92a4800eb32d785cbb39ae1f022528943909fd37deba63403677848bf98cc9dbd8fbfd7f2e4f34471866ea82ca6bffbf0f778b6931483700c17829b4bd066eb04983d3aac0bd46b9c8d03a73a8e714d3119de93cd9522e314054d16853b22014190063f77d9edf6fbccefcf71fffd1234f688823b4e429ae5fa639d0a664c842fbdfcb4d6e21f39d81c23563b92cffa09696d93c95bc4893a6401a43071d00d3e854f7f1f321afa7d5301d36f2195dc1e2643463f34ae637d2b02ae0eb11d4256c507a4f8304cea6396a7fce640f50acb301c2f6336d27717e84'
    b2verification = b'55210209e7fd41dfb5c2f8dc72eb30358ac100ea8c72da18847befe06eade68cebfcb9210327da12b5c40200e9f65569476bbff2218da4f32548ff43b6387ec1416a231ee821034ff5ceeac41acf22cd5ed2da17a6df4dd8358fcb2bfb1a43208ad0feaab2746b21026ce35b29147ad09e4afe4ec4a7319095f08198fa8babbe3c56e970b143528d2221038dddc06ce687677a53d54f096d2591ba2302068cf123c1f2d75c2dddc542557921039dafd8571a641058ccc832c5e2111ea39b09c0bde36050914384f7a48bce9bf92102d02b1873a0863cd042cc717da31cea0d7cf9db32b74d4c72c01b0011503e2e2257ae'

    b2tx_len = 1

    b2tx_id = b'8e3a32ba3a7e8bdb0ad9a2ad064713e45bd20eb0dab0d2e77df5b5ce985276d0'
    b2tx_nonce = 679092177
    b2tx_vin = []
    b2tx_vout = []

    @staticmethod
    def BlockIndexOne():
        block = Helper.AsSerializableWithType(BlocksTestCase.b2raw, 'neo.Core.Block.Block')
        return block

    def test_block_deserialize(self):

        block = Helper.AsSerializableWithType(self.rawblock_hex, 'neo.Core.Block.Block')

        self.assertEqual(self.rb_prev, block.PrevHash.ToBytes())
        self.assertEqual(self.rb_merlke, block.MerkleRoot.ToBytes())
        self.assertEqual(self.rb_ts, block.Timestamp)
        self.assertEqual(self.rb_h, block.Index)
        self.assertEqual(self.rb_nonce, block.ConsensusData)
        self.assertEqual(self.rconsenusdata, block.ConsensusData)
        #        self.assertEqual(self.rb_hash, block.HashToString())
        tx = block.Transactions[0]

        self.assertEqual(tx.Nonce, self.rblock_tx_nonce)

        self.assertEqual(len(tx.inputs), 0)
        self.assertEqual(len(tx.outputs), 0)
        self.assertEqual(len(tx.Attributes), 0)

        self.assertEqual(type(tx.scripts), list)

        rawdata = block.RawData()
        compair_data = self.rawblock[:len(rawdata)]
        self.assertEqual(rawdata, compair_data)
        out = block.Hash.ToBytes()
        self.assertEqual(out, self.rb_hash)

        root = MerkleTree.ComputeRoot([tx.Hash for tx in block.Transactions])
        self.assertEqual(root, block.MerkleRoot)

    def test_block_two(self):

        hexdata = binascii.unhexlify(self.b2raw)

        block = Helper.AsSerializableWithType(hexdata, 'neo.Core.Block.Block')
        self.assertEqual(block.Index, self.b2height)
        self.assertEqual(block.ConsensusData, self.b2nonce)
        self.assertEqual(block.Timestamp, self.b2timestamp)
        self.assertEqual(block.PrevHash.ToBytes(), self.b2prev_hash)

        self.assertEqual(block.Hash.ToString(), self.b2hash)

        next_consensus_address = Crypto.ToAddress(block.NextConsensus)

        self.assertEqual(next_consensus_address, self.b2nextconsensus)

        witness = block.Script
        ins = binascii.hexlify(witness.InvocationScript)
        vns = binascii.hexlify(witness.VerificationScript)

        self.assertEqual(ins, self.b2invocation)
        self.assertEqual(vns, self.b2verification)
        self.assertEqual(len(block.Transactions), self.b2tx_len)

        tx = block.Transactions[0]

        self.assertEqual(tx.inputs, self.b2tx_vin)
        self.assertEqual(tx.outputs, self.b2tx_vout)

        self.assertEqual(tx.Nonce, self.b2tx_nonce)

        txhash = tx.Hash.ToBytes()
        self.assertEqual(txhash, self.b2tx_id)

        root = MerkleTree.ComputeRoot([tx.Hash for tx in block.Transactions])
        self.assertEqual(root, block.MerkleRoot)

    sf_hash = b'6bfcdff84a3341d7a9299b358f098504e14c99c6774ebaa5dfd631371155533c'
    sf_raw = b'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'
    sf_merk = b'07e5d0324398aa9ddd60362178989f2e415793393cdd4a118b35c05ad5996c9b'
    sf_txlen = 2
    sf_tx1_id = b'f2195b0382fca1d8648fe0f9bdbf441ac4f651f614bfbf5d92ba842a9698e9f1'
    sf_tx2_id = b'4feb0081f9425cab84269127bef0a871a84d4408f09923d17ebb257cd231b362'
    sf2_vin_len = 1
    sf2_vo_len = 1

    def test_block_seventyfour(self):

        hexdata = binascii.unhexlify(self.sf_raw)

        block = Helper.AsSerializableWithType(hexdata, 'neo.Core.Block.Block')

        self.assertEqual(block.MerkleRoot.ToBytes(), self.sf_merk)
        self.assertEqual(block.Hash.ToBytes(), self.sf_hash)

    pb_raw = b'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'
    pb_hash = b'077c7fc9a85d777aeb42e1076bd98451f16e59354bfb6fed998ccabd93f6ccb9'

    def test_block_publish_tx(self):

        hexdata = binascii.unhexlify(self.pb_raw)

        block = Helper.AsSerializableWithType(hexdata, 'neo.Core.Block.Block')

        self.assertEqual(block.Hash.ToBytes(), self.pb_hash)

    t992 = b'000000001db78fc59944ef5a8c865efc09b244896e8f32253fb21fb47e13128706cae385b9c95339d1fa2ea86bcdb3e79d05a5b42c35789dff5c54027d24fa549a66684c3f9b1359b00b000066666542ed4bbc15f3812db982f3b0089a21a278988efeec6a027b2501fd4501400bfee22cb2e369debb4fe5c52bcc33c17b25aa9402aa915e3db82fd41e711966d15e255434b4a99622a831ba0fc6cc99f288a2605818cd4bc0b807a05ffe1c9740e16607602ca56dec01c17d3ad1ffa9b425bc38bd7a4bca7348fed11b867bdf16cb9d571bab51f56d7e8c8114f6a01d4e0c107ea99dd5e23a4ae488cbd26d17c340d3771b97e559fc116680ff7cf4e9aa36826a40d8cd3c59a578d45078fbcac9f1b98f125acb7de9235ff1087c1efde2ecfc2b82e95ec98769adb39ee88da9745740caced10c3b24ad134da0561396abfa4d94c3847226214c9ff451b593b0d3959e2ea6abe3afd9bbce1f7d3cd9ab61b80dd5afd81dbc987a488160f9dafaacd8b2408038ce11fa724bba34987ff001fba349735273a3b7a580828c3b762b837b958100fc65acae1598b23b32303d10ec99ee2421cbc5838a7e261d8e3accbef7ff5ef155210209e7fd41dfb5c2f8dc72eb30358ac100ea8c72da18847befe06eade68cebfcb9210327da12b5c40200e9f65569476bbff2218da4f32548ff43b6387ec1416a231ee821034ff5ceeac41acf22cd5ed2da17a6df4dd8358fcb2bfb1a43208ad0feaab2746b21026ce35b29147ad09e4afe4ec4a7319095f08198fa8babbe3c56e970b143528d2221038dddc06ce687677a53d54f096d2591ba2302068cf123c1f2d75c2dddc542557921039dafd8571a641058ccc832c5e2111ea39b09c0bde36050914384f7a48bce9bf92102d02b1873a0863cd042cc717da31cea0d7cf9db32b74d4c72c01b0011503e2e2257ae0100006666654200000000'
    t992h = b'bcfc016b33f6abc1f1109878f6d69a8aaf42b9ac09c5793507afbbdccdb8f68f'
    t992m = b'4c68669a54fa247d02545cff9d78352cb4a5059de7b3cd6ba82efad13953c9b9'

    def test_2992(self):

        hexdata = binascii.unhexlify(self.t992)

        block = Helper.AsSerializableWithType(hexdata, 'neo.Core.Block.Block')
        self.assertEqual(block.MerkleRoot.ToBytes(), self.t992m)
        self.assertEqual(block.Hash.ToBytes(), self.t992h)

        json = block.ToJson()
        self.assertEqual(json['index'], 2992)
        self.assertEqual(json['hash'], '0x%s' % self.t992h.decode('utf-8'))
        self.assertEqual(json['merkleroot'], '0x%s' % self.t992m.decode('utf-8'))
        self.assertEqual(len(json['tx']), 1)
        self.assertEqual(json['tx'][0]['txid'], '0x4c68669a54fa247d02545cff9d78352cb4a5059de7b3cd6ba82efad13953c9b9')

    big_tx_hash = b'93fb1184b95fd785022dea163ffc60ef6fa548ca240b49dd8339700cee581991'

    def test_1050514(self):

        path = '%s/fixtures/1050514.txt' % os.getcwd()

        with open(path, 'rb') as f:
            #
            blockraw = f.read().strip()

            hex = binascii.unhexlify(blockraw)

            block = Helper.AsSerializableWithType(hex, 'neo.Core.Block.Block')

            self.assertEqual(self.big_tx_hash, block.Hash.ToBytes())

        json = block.ToJson()

        self.assertEqual(json['index'], 1050514)
        self.assertEqual(json['hash'], '0x%s' % self.big_tx_hash.decode('utf-8'))
        self.assertEqual(json['nonce'], 'a62f207d4f00af81')
        self.assertEqual(json['nextconsensus'], 'APyEx5f4Zm4oCHwFWiSTaph1fPBxZacYVR')
        self.assertEqual(len(json['tx']), 65)

        for tx in json['tx']:
            if tx['txid'] == '1dc3543a5b54fcfce3fefba6c772f3a59740b2b1784690b3c66e2f7052f002bb':
                bigclaim = tx
                self.assertEqual(len(bigclaim['claims']), 1756)
                vout = bigclaim['vout'][0]
                self.assertEqual(vout['ScriptHash'], 'AFnRHRQceaUgFQxPTttAQzAZvsjGSNtHCH')
                self.assertEqual(vout['Value'], 5786800133972)

    b_1321456_merlke = '7020347797526cea93a0cd47303e3274afe6f79dd0d184a0e55cc673fdd1d382'

    b_132156_txids = [
        'a4b0def576397d4b66b5911888867dcee0fe67833f35c9c6d9ad12af34b25190',
        '1cd3060b7f75addd9bbd078c7eb0483a8b6c05b5a932f3d487674ff17404a81f',
        '9a299703b27187a6a9157cbb90bf372421aab0e0786e50eb4c292ecd0171ef06',
        '61e4af6da434c708865bec387e60cb09476871a92ea2a3716d78cf34eda16f6e',
        '167c5a3905d6f3495f41fef9094e97fe42ce2ba608cffd4a74324d8242de5589',
        'bc21700872ae2ed27983ad64e485e069b6e9d341aca3bfcb18ee953b77aaebe3',
        'bdf3f4357cf5bca685607f405abb1ec2202b699da0a1db16911302205b402fe3',
        'cccadeee3df2c4d70a4b8563b374bb3c4eb60c850999f81fec5ee56149814249',
        '3c6c8c77f05f9e0da53a072e694eec498a3894f683cbf59f70625780c31f55c3',
        '8d95ac8a60bc004baa779a9e9da2f465c13108505c70daad95a9c977228dbf46',
        '170cfd5e0ce4162937f2e4c4331de7bb3a1cb342fb3e7167ed16fbbd8020b791',
        '50e404d8850545fe773e82af93a1b84643cc6403d44f882f7f7b2eb23e42fc60',
        '14ff62cb005ba8697e8300877a1c6c1ca8f89c8460f747e9a79bede69f8fb65b',
        '3ac733706d44c8409569014f8b1cdcb01e001bc1b1c56551197591559fe20af0',
        '9af1fcaab6fec80922e25dbea34c534c743dcf8d10f76af1892526c2879d3a70',
    ]

    def test_1321456(self):

        path = '%s/fixtures/1321456.txt' % os.getcwd()

        with open(path, 'rb') as f:
            blockraw = f.read().strip()

            hex = binascii.unhexlify(blockraw)

            block = Helper.AsSerializableWithType(hex, 'neo.Core.Block.Block')

            for tx in block.Transactions:
                hash = tx.Hash.ToString()

                self.assertTrue(hash in self.b_132156_txids)

            self.assertEqual(block.MerkleRoot.ToString(), self.b_1321456_merlke)

    # def test_testnet797966(self):
    #
    #     path = '%s/fixtures/797966.txt' % os.getcwd()
    #
    #     with open(path, 'rb') as f:
    #         blockraw = f.read().strip()
    #
    #         hex = binascii.unhexlify(blockraw)
    #
    #         block = Helper.AsSerializableWithType(hex, 'neo.Core.Block.Block')
    #
    #         trimmed = block.Trim()
    #
    #         trimmed_unhex = binascii.unhexlify(trimmed)
    #
    #         blockfrom_trimmed = Block.FromTrimmedData(trimmed_unhex)
    #
    #         self.assertEqual(blockfrom_trimmed.Version, block.Version)
    #         self.assertEqual(blockfrom_trimmed.Index, block.Index)
    #         self.assertEqual(blockfrom_trimmed.PrevHash, block.PrevHash)
    #         self.assertEqual(blockfrom_trimmed.MerkleRoot, block.MerkleRoot)
    #         self.assertEqual(blockfrom_trimmed.Timestamp, block.Timestamp)
    #         self.assertEqual(blockfrom_trimmed.Index, block.Index)
    #         self.assertEqual(blockfrom_trimmed.ConsensusData, block.ConsensusData)
    #         self.assertEqual(blockfrom_trimmed.NextConsensus, block.NextConsensus)
    #         self.assertEqual(len(block.Transactions), len(blockfrom_trimmed.Transactions))

    block_bytes = b'\x00\x00\x00\x00\xed\x00.\xa6\xc6(\xa2\x80\x1a\x95E\x8f\xb4\xaa\xcc$\x98\x031\xd3\xca2!-\x1a\xaa/m!\xb8V\xa4Jj\xd4\xad6\xc1a\xb25-5\xe5\xd7\xec\xe2\x13T\xbb\xf6\x02\xa4\xaes\x98((\xd7i\n\x81\xfa\xf6\xf0neZ\\\xe6\x0f\x00\x88\xb7\xcf\xab\xb3d\x88M\xf3\x81-\xb9\x82\xf3\xb0\x08\x9a!\xa2x\x98\x8e\xfe\xecj\x02{%\x01\xfdE\x01@\xa0Z\xf8\xda\x93*\xd0\x04\xdcq\x06\xd5\x86\x07G\xd0 \\;\xb3cg\xcc\xf3=}B\x07\xf4\'\xf1k\xe4\x80|\xb9\x07\x9e\xa8\x8b\x9f\xf0\xd7 \xbb\x1a\x82\xa1\xa5\x9a\xb0\x9e\xf4\x1a\xb7\x89S\xd4\xeai\x8f\x8b\xcb_@x\x8a%\x07\x93\x14\x04\xbc\r\x9d\xdb\xc2A\xc0\x8d\x07\x13F\xd1An\xf8$\x02\xa0\xe4\n1?I\x9a\xfc.\xf0e\x8c\xda\x9dv\xc3\xac\xd7\xdav\x91:V\xba\xb2pK\xceF\xc4Z\xc6boj\xdc\x1e\xc3\xd4\x06@\xd9\x8cH\xa7\xa5*\xba\xb9\xac~\xf8\x05\xecONc\xb0\xa5\x1aH0\xfd\x11K>pL\x84\xd9\xaeK,\x87\x92x5d\xc1\xa7m\x1f!)\x9fl\xa5\x12>\x07md\x83\x17#\x1f"\xb9Uk\x82\xaf\xfb\xaa\xbb@n\x8b\x1a\x96\x85O\xf2\x84-\xeb\xa3d\xe4\xad\x9e":\xbc\xab5\xe6$P\xef\x91\xe3\xc3\xaf\x11\xc2f\xde\xf6\xec\xc55k\xe7\xdf\x9a?Z2v\xedF4\x01p)5\r\xd2t+\x10Y\xe2\x8a\x05I:\x13I@\xc3H\xb3/\x06\x08\xcb\xcf\xd1\xa5\xd8\xa6,3m1\x909\x1b\x17\xda^\xea\xee\xcc\x1e\x02\x90h\xb0\x04\x1a\x94\x92~o\xf2^\x0e\xca\x821\xb90\t\x98\xd2\xad\xc8\xdc\xb5<\xa4\x99\xc2c\xe9\xb9\x91\x7fS\xa0=\t\xf1U!\x02\t\xe7\xfdA\xdf\xb5\xc2\xf8\xdcr\xeb05\x8a\xc1\x00\xea\x8cr\xda\x18\x84{\xef\xe0n\xad\xe6\x8c\xeb\xfc\xb9!\x03\'\xda\x12\xb5\xc4\x02\x00\xe9\xf6UiGk\xbf\xf2!\x8d\xa4\xf3%H\xffC\xb68~\xc1Aj#\x1e\xe8!\x03O\xf5\xce\xea\xc4\x1a\xcf"\xcd^\xd2\xda\x17\xa6\xdfM\xd85\x8f\xcb+\xfb\x1aC \x8a\xd0\xfe\xaa\xb2tk!\x02l\xe3[)\x14z\xd0\x9eJ\xfeN\xc4\xa71\x90\x95\xf0\x81\x98\xfa\x8b\xab\xbe<V\xe9p\xb1CR\x8d"!\x03\x8d\xdd\xc0l\xe6\x87gzS\xd5O\tm%\x91\xba#\x02\x06\x8c\xf1#\xc1\xf2\xd7\\-\xdd\xc5BUy!\x03\x9d\xaf\xd8W\x1ad\x10X\xcc\xc82\xc5\xe2\x11\x1e\xa3\x9b\t\xc0\xbd\xe3`P\x91C\x84\xf7\xa4\x8b\xce\x9b\xf9!\x02\xd0+\x18s\xa0\x86<\xd0B\xccq}\xa3\x1c\xea\r|\xf9\xdb2\xb7MLr\xc0\x1b\x00\x11P>."W\xae\x02\x00\x00\x88\xb7\xcf\xab\x00\x00\x00\x00\x90\x00\x01H!\x03\xc0\x89\xd7\x12+\x84\nI5#N\x82\xe2j\xe5\xef\xd0\xc2\xac\xb6\'#\x9d\xc9\xf2\x071\x137\xb6\xf2\xc1\nRegistered\x01\x01\x00\x01\xcbA\x84\xf0\xa9nrel\x1f\xbd\xd4\xf7\\\xcaVu\x19\xe9\t\xfdC\xce\xfc\xec\x13\xd6\xc6\xab\xcb\x92\xa1\x00\x00\x01\xe7-(iy\xeel\xb1\xb7\xe6]\xfd\xdf\xb2\xe3\x84\x10\x0b\x8d\x14\x8ewX\xdeB\xe4\x16\x8bqy,`\x00\xb8\xfb\x05\x01\t\x00\x00q\xf9\xcf\x7f\x0e\xc7N\xc0\xb0\xf2\x8a\x92\xb1.\x10\x81WL\n\xf0\x01A@\x87\x80\xd7\xb3\xc0\xaa\xdcS\x98\x15=\xf5\xe2\xf1\xcf\x15\x9d\xb2\x1b\x8b\x0f4\xd3\x99M\x86T3\xf7\x9f\xaf\xacAh7\x83\xc4\x8a\xefQ\x0bgf\x0e1W\xb7\x01\xb9\xcaM\xd9\x94j8]W\x8f\xba}\xd2oHI#!\x03\xc0\x89\xd7\x12+\x84\nI5#N\x82\xe2j\xe5\xef\xd0\xc2\xac\xb6\'#\x9d\xc9\xf2\x071\x137\xb6\xf2\xc1\xac'

    def test_testnet10412011(self):

        block = Helper.AsSerializableWithType(self.block_bytes, 'neo.Core.Block.Block')
        self.assertEqual(len(block.Transactions), 2)
        state_tx = block.Transactions[1]
        self.assertIsInstance(state_tx, StateTransaction)
        self.assertEqual(len(state_tx.Descriptors), 1)
        descriptor = state_tx.Descriptors[0]
        self.assertIsInstance(descriptor, StateDescriptor)
        self.assertEqual(descriptor.Type, StateType.Validator)
        self.assertEqual(descriptor.SystemFee, Fixed8.FromDecimal(1000))
        self.assertEqual(descriptor.Key, binascii.unhexlify(b'03c089d7122b840a4935234e82e26ae5efd0c2acb627239dc9f207311337b6f2c1'))

    def test_block_size(self):
        # MainNet block 2504147
        raw_block = b'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'
        block = Helper.AsSerializableWithType(binascii.unhexlify(raw_block), 'neo.Core.Block.Block')
        self.assertEqual(2876, block.Size())

    def test_block_size_with_claim_transactions(self):
        # MainNet block 2504280
        raw_block = b'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'
        block = Helper.AsSerializableWithType(binascii.unhexlify(raw_block), 'neo.Core.Block.Block')
        self.assertEqual(5115, block.Size())
