from .StateBase import StateBase
from neo.Core.IO.BinaryReader import BinaryReader
from neo.IO.MemoryStream import StreamManager
from neo.Core.Size import GetVarSize


class StorageItem(StateBase):

    def __init__(self, value=None):
        """
        Create an instance.

        Args:
            value (bytearray): value to store.
        """
        if value is None:
            self.Value = bytearray(0)
        else:
            self.Value = value
        self.IsConstant = False

    def Clone(self):
        """
        Clone self.

        Returns:
            StorageItem:
        """
        return StorageItem(value=self.Value)

    def FromReplica(self, replica):
        """
        Get StorageItem object from a replica.
        Args:
            replica (obj): must have `Value` member.

        Returns:
            StorageItem:
        """
        self.Value = replica.Value

    def Size(self):
        """
        Get the total size in bytes of the object.

        Returns:
            int: size.
        """
        return super(StorageItem, self).Size() + GetVarSize(self.Value)

    def Deserialize(self, reader):
        """
        Deserialize full object.

        Args:
            reader (neo.Core.IO.BinaryReader):
        """
        super(StorageItem, self).Deserialize(reader)
        self.Value = reader.ReadVarBytes()

    @staticmethod
    def DeserializeFromDB(buffer):
        """
        Deserialize full object.

        Args:
            buffer (bytes, bytearray, BytesIO): (Optional) data to create the stream from.

        Returns:
            StorageItem:
        """
        m = StreamManager.GetStream(buffer)
        reader = BinaryReader(m)
        v = StorageItem()
        v.Deserialize(reader)
        StreamManager.ReleaseStream(m)
        return v

    def Serialize(self, writer):
        """
        Serialize full object.

        Args:
            writer (neo.IO.BinaryWriter):
        """
        super(StorageItem, self).Serialize(writer)
        writer.WriteVarBytes(self.Value)
