# SPDX-License-Identifier: BSD-3-Clause
# Copyright (c) 2026 Antoine COLLET
"""
Test the NestedGridPlotter base class.

Note that the tests are very similar to the examples found in the tutorial and
follow the same order. The

@author: Antoine COLLET
"""

import re
import struct
from contextlib import nullcontext as does_not_raise
from itertools import product

import matplotlib as mpl
import nested_grid_plotter as ngp
import numpy as np
import pytest
from matplotlib.axes import Axes
from matplotlib.figure import SubFigure
from matplotlib.lines import Line2D
from nested_grid_plotter.base_plotter import (
    _make_kwargs_retrocompatible,
    is_lgd_overlapping_axis,
)
from packaging.version import Version


@pytest.mark.parametrize(
    "gridspec_kw, width_ratios, height_ratios, expected_exception",
    (
        ({}, None, None, does_not_raise()),
        (
            dict(width_ratios=[1, 1, 1], height_ratios=[1, 1, 1]),
            None,
            None,
            does_not_raise(),
        ),
        (
            dict(),
            [1, 1, 1],
            [1, 1, 1],
            does_not_raise(),
        ),
        (
            dict(width_ratios=[1, 1, 1], height_ratios=[1, 1, 1]),
            [1, 1, 1],
            None,
            pytest.raises(
                ValueError,
                match=(
                    "'width_ratios' must not be defined both as parameter and as "
                    "key in 'gridspec_kw'"
                ),
            ),
        ),
        (
            dict(width_ratios=[1, 1, 1], height_ratios=[1, 1, 1]),
            None,
            [1, 1, 1],
            pytest.raises(
                ValueError,
                match=(
                    "'height_ratios' must not be defined both as parameter and as "
                    "key in 'gridspec_kw'"
                ),
            ),
        ),
    ),
)
def test_SubplotsMosaicBuilder(
    gridspec_kw, width_ratios, height_ratios, expected_exception
) -> None:
    with expected_exception:
        ngp.SubplotsMosaicBuilder(
            mosaic=[["A panel", "A panel", "edge"], ["C panel", ".", "edge"]],
            gridspec_kw=gridspec_kw,
            width_ratios=width_ratios,
            height_ratios=height_ratios,
        )


@pytest.mark.filterwarnings("error")
def test_make_kwargs_retrocompatible() -> None:
    with pytest.raises(
        UserWarning,
        match=re.escape(
            'Parameter "per_subplot_kw" is supported from matplotlib 3.7 '
            "while you use version 3.6 and it is consequently "
            "ignored."
        ),
    ):
        _make_kwargs_retrocompatible(
            "3.6",
            ngp.SubplotsMosaicBuilder(
                mosaic=[["A panel", "A panel", "edge"], ["C panel", ".", "edge"]],
                per_subplot_kw={},
            ),
        )

    _make_kwargs_retrocompatible(
        "3.6",
        ngp.SubplotsMosaicBuilder(
            mosaic=[["A panel", "A panel", "edge"], ["C panel", ".", "edge"]],
        ),
    )


@pytest.fixture
def tmp_folder(tmp_path_factory):
    """Create a temporary directory"""
    return tmp_path_factory.mktemp("tmp_folder")


def test_plotter_without_input_args():
    plotter = ngp.Plotter()
    assert list(plotter.sf_dict.keys()) == []
    assert list(plotter.grouped_ax_dict.keys()) == ["fig"]
    assert list(plotter.grouped_ax_dict["fig"].keys()) == ["ax1-1"]
    assert list(plotter.ax_dict.keys()) == ["ax1-1"]

    x = np.linspace(0, 2, 100)  # Sample data.
    ax = plotter.ax_dict["ax1-1"]
    ax.plot(x, x, label="linear")
    ax.plot(x, x**2, label="quadratic")
    ax.plot(x, x**3, label="cubic")
    ax.set_xlabel("x label")  # Add an x-label to the axes.
    ax.set_ylabel("y label")  # Add a y-label to the axes.
    ax.set_title("Simple Plot")  # Add a title to the axes.
    ax.legend()
    # Add a legend.
    # plt.show()


def test_multiple_subfigs_no_mosaic():
    plotter = ngp.Plotter(
        ngp.Figure(constrained_layout=True),
        builder=ngp.SubfigsBuilder(nrows=2, ncols=3),
    )
    assert list(plotter.sf_dict.keys()) == [
        "subfig_1",
        "subfig_2",
        "subfig_3",
        "subfig_4",
        "subfig_5",
        "subfig_6",
    ]
    assert list(plotter.grouped_ax_dict.keys()) == [
        "subfig_1",
        "subfig_2",
        "subfig_3",
        "subfig_4",
        "subfig_5",
        "subfig_6",
    ]
    for i, j in product([1, 2], [1, 2, 3]):
        assert list(plotter.grouped_ax_dict[f"subfig_{i * j}"].keys()) == [
            f"subfig_{i * j}_ax1-1"
        ]
    assert sorted(list(plotter.ax_dict.keys())) == [
        "subfig_1_ax1-1",
        "subfig_2_ax1-1",
        "subfig_3_ax1-1",
        "subfig_4_ax1-1",
        "subfig_5_ax1-1",
        "subfig_6_ax1-1",
    ]
    plotter.identify_axes()  # Helper to add the name of the axis on the plot
    # plt.show()


def test_unique_subfig_with_mosaic():
    plotter = ngp.Plotter(
        ngp.Figure(constrained_layout=True),
        builder=ngp.SubfigsBuilder(
            sub_builders={
                "my_subfigure_name": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax1-1", "ax1-2", "ax1-3"], ["ax2-1", "ax2-2", "ax2-3"]]
                ),
            }
        ),
    )
    assert list(plotter.sf_dict.keys()) == ["my_subfigure_name"]
    assert list(plotter.grouped_ax_dict.keys()) == ["my_subfigure_name"]
    assert list(plotter.grouped_ax_dict["my_subfigure_name"].keys()) == [
        "ax1-1",
        "ax1-2",
        "ax1-3",
        "ax2-1",
        "ax2-2",
        "ax2-3",
    ]
    assert sorted(list(plotter.ax_dict.keys())) == [
        "ax1-1",
        "ax1-2",
        "ax1-3",
        "ax2-1",
        "ax2-2",
        "ax2-3",
    ]
    plotter.identify_axes()  # Helper to add the name of the axis on the plot
    # plt.show()

    # Get the attribute
    plotter.axes_names


def test_multiple_subfigs_1_row_with_mosaic():
    plotter = ngp.Plotter(
        ngp.Figure(constrained_layout=True, figsize=(10, 10)),
        builder=ngp.SubfigsBuilder(
            nrows=1,
            ncols=2,
            sub_builders={
                "the_left_sub_figure": ngp.SubplotsMosaicBuilder(
                    mosaic=[["lt1", "lt1"], ["lb1", "rb1"]],
                    gridspec_kw=dict(height_ratios=[2, 1], width_ratios=[2, 1]),
                    sharey=False,
                ),
                "the_right_sub_figure": ngp.SubplotsMosaicBuilder(
                    mosaic=[["l2", "rt2"], ["l2", "rb2"]],
                    gridspec_kw=dict(height_ratios=[2, 1], width_ratios=[2, 1]),
                    sharey=False,
                ),
            },
        ),
    )
    assert list(plotter.sf_dict.keys()) == [
        "the_left_sub_figure",
        "the_right_sub_figure",
    ]
    assert list(plotter.grouped_ax_dict.keys()) == [
        "the_left_sub_figure",
        "the_right_sub_figure",
    ]
    assert list(plotter.grouped_ax_dict["the_left_sub_figure"].keys()) == [
        "lt1",
        "lb1",
        "rb1",
    ]
    assert list(plotter.grouped_ax_dict["the_right_sub_figure"].keys()) == [
        "l2",
        "rt2",
        "rb2",
    ]
    assert sorted(list(plotter.ax_dict.keys())) == [
        "l2",
        "lb1",
        "lt1",
        "rb1",
        "rb2",
        "rt2",
    ]
    plotter.identify_axes()  # Helper to add the name of the axis on the plot
    # plt.show()


def test_multiple_subfigs_2_rows_with_mosaic():
    plotter = ngp.Plotter(
        ngp.Figure(constrained_layout=True, figsize=(10, 10)),
        builder=ngp.SubfigsBuilder(
            nrows=2,
            ncols=2,
            sub_builders={
                "the_top_left_sub_figure": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax1"], ["ax2"]]
                ),
                "the_top_right_sub_figure": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax3"], ["ax4"]]
                ),
                "the_bottom_left_sub_figure": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax5"], ["ax6"]]
                ),
                "the_bottom_right_sub_figure": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax7"], ["ax8"]]
                ),
            },
        ),
    )
    assert list(plotter.sf_dict.keys()) == [
        "the_top_left_sub_figure",
        "the_top_right_sub_figure",
        "the_bottom_left_sub_figure",
        "the_bottom_right_sub_figure",
    ]
    assert list(plotter.grouped_ax_dict.keys()) == [
        "the_top_left_sub_figure",
        "the_top_right_sub_figure",
        "the_bottom_left_sub_figure",
        "the_bottom_right_sub_figure",
    ]
    assert list(plotter.grouped_ax_dict["the_top_left_sub_figure"].keys()) == [
        "ax1",
        "ax2",
    ]
    assert list(plotter.grouped_ax_dict["the_top_right_sub_figure"].keys()) == [
        "ax3",
        "ax4",
    ]
    assert sorted(list(plotter.ax_dict.keys())) == [
        "ax1",
        "ax2",
        "ax3",
        "ax4",
        "ax5",
        "ax6",
        "ax7",
        "ax8",
    ]
    plotter.identify_axes()  # Helper to add the name of the axis on the plot
    # plt.show()


def test_less_keys_in_subplots_mosaic_params_than_subfigs():
    with pytest.raises(
        Exception,
        match=(
            "Error while creating subfigures for fig, 1 builders have been provided, "
            "but there are 1 rows and 2 cols, i.e., 2 builders expected!"
        ),
    ):
        ngp.Plotter(
            ngp.Figure(constrained_layout=True, figsize=(10, 10)),
            builder=ngp.SubfigsBuilder(
                nrows=1,
                ncols=2,
                sub_builders={
                    "the_left_sub_figure": ngp.SubplotsMosaicBuilder(
                        mosaic=[["t-left", "t-left"], ["b-left", "b-right"]],
                    ),
                },
            ),
        )


def test_error_more_keys_in_subplots_mosaic_params_than_subfigs():
    with pytest.raises(Exception):
        return ngp.Plotter(
            ngp.Figure(constrained_layout=True, figsize=(10, 10)),
            builder=ngp.SubfigsBuilder(
                nrows=1,
                ncols=2,
                sub_builders={
                    "the_left_sub_figure": ngp.SubplotsMosaicBuilder(
                        mosaic=[["tl1", "tl1"], ["bl1", "br1"]],
                    ),
                    "the_center_sub_figure": ngp.SubplotsMosaicBuilder(
                        mosaic=[["tl2", "tl2"], ["bl2", "br2"]],
                    ),
                    "the_right_sub_figure": ngp.SubplotsMosaicBuilder(
                        mosaic=[["tl3", "tl3"], ["bl3", "br3"]],
                    ),
                },
            ),
        )


def test_error_same_axis_names_used_in_multiple_subfigures():
    # Test multiple duplicatedaxis names
    with pytest.raises(Exception):
        return ngp.Plotter(
            builder=ngp.SubfigsBuilder(
                nrows=1,
                ncols=2,
                sub_builders={
                    "the_left_sub_figure": ngp.SubplotsMosaicBuilder(
                        mosaic=[["ax11", "ax11"], ["ax12", "ax13"]],
                    ),
                    "the_right_sub_figure": ngp.SubplotsMosaicBuilder(
                        mosaic=[["ax12", "ax11"], ["ax12", "ax13"]],
                    ),
                },
            )
        )

    # Test single duplicatedaxis name
    with pytest.raises(Exception):
        return ngp.Plotter(
            ngp.Figure(constrained_layout=True, figsize=(10, 10)),
            builder=ngp.SubfigsBuilder(
                nrows=1,
                ncols=2,
                sub_builders={
                    "the_left_sub_figure": ngp.SubplotsMosaicBuilder(
                        mosaic=[["ax11", "ax11"]],
                    ),
                    "the_right_sub_figure": ngp.SubplotsMosaicBuilder(
                        mosaic=[["ax12", "ax11"]],
                    ),
                },
            ),
        )


def test_savefig(tmp_path_factory) -> None:
    tmp_folder = tmp_path_factory.mktemp("data")
    plotter = ngp.Plotter()
    ax = plotter.ax_dict["ax1-1"]
    x = np.linspace(0, 2, 100)  # Sample data.
    ax.plot(x, x, label="linear")
    ax.legend()

    plotter.savefig(tmp_folder.joinpath("test_fig"))


def test_plotter_get_bbox_extra_artists_no_kwargs() -> None:
    assert ngp.Plotter()._get_bbox_extra_artists() is not None


def test_savefif_with_legend(tmp_path_factory) -> None:
    tmp_folder = tmp_path_factory.mktemp("data")

    plotter = gen_complex_example_fig()
    x = np.linspace(0, 2, 100)  # Sample data.
    for ax_name, ax in plotter.ax_dict.items():
        ax.plot(x, x, label=f"linear {ax_name}")  # Plot some data on the axes.
        ax.plot(x, x**2, label=f"quadratic {ax_name}")  # Plot more data on the axes...
        ax.plot(x, x**3, label=f"cubic {ax_name}")  # ... and some more.

    # Add some lines and spans and add it to the legend
    handle = plotter.ax_dict["lt1"].axvline(
        x=1.0, color="k", linewidth=3, linestyle="--"
    )
    plotter.add_extra_legend_item("lt1", handle, "My left extra legend item")

    handle = plotter.ax_dict["lb1"].axvspan(xmin=0.25, xmax=0.75, color="yellow")
    plotter.add_extra_legend_item("lt1", handle, "My left span")

    handle = plotter.ax_dict["l2"].axhline(
        y=4.0, xmin=0.2, xmax=0.8, color="r", linewidth=3, linestyle="--"
    )
    plotter.add_extra_legend_item("l2", handle, "My right extra legend item")

    handle = plotter.ax_dict["l2"].axhspan(ymin=1.0, ymax=7.0, color="blue", alpha=0.3)
    plotter.add_extra_legend_item("l2", handle, "My right span")

    # Add the legend to subfigures (we place it to the bottom)
    plotter.add_fig_legend(
        name="the_left_sub_figure",
        fontsize=10,
        ncol=2,
        bbox_x_shift=-0.25,
        bbox_y_shift=-0.06,
    )
    plotter.add_fig_legend(
        name="the_right_sub_figure",
        fontsize=10,
        ncol=2,
        bbox_x_shift=+0.25,
        bbox_y_shift=-0.06,
    )

    # Add a title
    plotter.fig.suptitle("Main figure suptitle")
    plotter.sf_dict["the_left_sub_figure"].supxlabel("Left figure supxlabel")
    plotter.sf_dict["the_left_sub_figure"].supylabel("Left sub figure supylabel")

    # Add the "full" fig legend just for the example (we place it to the top)
    _ = plotter.add_fig_legend(fontsize=10, loc="top", ncol=4, bbox_y_shift=+0.06)

    # Now we save the figure.
    dpi = 72
    savepath = tmp_folder.joinpath("with_plt_savefig.png")
    # save the figure with matplotlib built-in method
    plotter.fig.savefig(savepath, dpi=dpi)

    # This size of the figure should be unchanged => the figure legends are ignored.
    with open(savepath, "rb") as f:
        data = f.read()
    w, h = struct.unpack(">LL", data[16:24])
    expected_fig_size = plotter.fig.get_size_inches()
    np.testing.assert_equal(expected_fig_size * dpi, np.array([int(w), int(h)]))

    # Now we use our implementation
    savepath2 = tmp_folder.joinpath("with_plotter_savefig.png")
    # save the figure
    plotter.savefig(savepath2, dpi=dpi)

    # Assert the size of the save figure with native python
    with open(savepath2, "rb") as f2:
        data2 = f2.read()
    w2, h2 = struct.unpack(">LL", data2[16:24])
    # The default behavior is to use tight_layout = True. So the added figure are taken
    # into account and the figure size is increased.
    assert int(w2) > int(w)
    assert int(h2) > int(h)


def test_close() -> None:
    plotter = ngp.Plotter()
    plotter.close()


def gen_complex_example_fig() -> ngp.Plotter:
    return ngp.Plotter(
        ngp.Figure(
            constrained_layout=True,  # Always use this to prevent overlappings
            figsize=(15, 6),
        ),
        builder=ngp.SubfigsBuilder(
            nrows=1,
            ncols=2,
            sub_builders={
                "the_left_sub_figure": ngp.SubplotsMosaicBuilder(
                    mosaic=[["lt1", "lt1"], ["lb1", "rb1"]],
                    gridspec_kw=dict(height_ratios=[2, 1], width_ratios=[2, 1]),
                    sharey=False,
                ),
                "the_right_sub_figure": ngp.SubfigsBuilder(
                    nrows=1,
                    ncols=2,
                    width_ratios=[2, 1],
                    sub_builders={
                        "the_right_left_sub_figure": ngp.SubplotsMosaicBuilder(
                            mosaic=[["l2"]],
                        ),
                        "the_right_right_sub_figure": ngp.SubplotsMosaicBuilder(
                            mosaic=[["rt2"], ["rb2"]],
                            gridspec_kw=dict(height_ratios=[2, 1]),
                            sharey=False,
                        ),
                    },
                ),
            },
        ),
    )


def test_get_axis() -> None:
    plotter = gen_complex_example_fig()
    # Get an axis
    assert isinstance(plotter.get_axis("rb2"), Axes)

    with pytest.raises(ValueError, match='The axis "test" does not exists!'):
        plotter.get_axis("test")


def test_get_axes() -> None:
    plotter = gen_complex_example_fig()
    # Get an axis
    for ax in plotter.get_axes(["lt1", "l2", "rb2"]):
        assert isinstance(ax, Axes)

    for ax in plotter.get_axes(["lt1"]):
        assert isinstance(ax, Axes)

    for ax in plotter.get_axes(list(plotter.ax_dict.keys())):
        assert isinstance(ax, Axes)

    with pytest.raises(ValueError, match='The axis "test" does not exists!'):
        plotter.get_axes(["test"])


def test_get_subfigure() -> None:
    plotter = gen_complex_example_fig()
    # Get a subfigure
    assert isinstance(plotter.get_subfigure("the_right_sub_figure"), SubFigure)

    with pytest.raises(ValueError, match='The subfigure "test" does not exists!'):
        plotter.get_subfigure("test")


def test_add_grid_to_all_axes_of_a_subfigure():
    # Create a plot
    plotter = gen_complex_example_fig()
    # Add the grids
    plotter.add_grid_and_tick_prams_to_all_axes(
        subfigure_name="the_left_sub_figure", colors="red", grid_color="red"
    )

    for subfig_name, ax_dict in plotter.grouped_ax_dict.items():
        if subfig_name == "the_left_sub_figure":
            for ax in ax_dict.values():
                assert ax.xaxis.get_major_ticks()[0].gridline.get_visible()
                assert ax.yaxis.get_major_ticks()[0].gridline.get_visible()
        else:
            for ax in ax_dict.values():
                assert not ax.xaxis.get_major_ticks()[0].gridline.get_visible()
                assert not ax.yaxis.get_major_ticks()[0].gridline.get_visible()


def test_add_grid_to_all_axes_of_the_plot():
    # Create a plot
    plotter = gen_complex_example_fig()
    # Add the grids to all subfigures' plots
    plotter.add_grid_and_tick_prams_to_all_axes(
        colors="blue", grid_color="blue", length=5
    )
    for name, ax in plotter.ax_dict.items():
        if name in ["lt1", "lb1"]:
            assert ax.xaxis.get_major_ticks()[0].gridline.get_visible()
            assert ax.yaxis.get_major_ticks()[0].gridline.get_visible()


# Generate a figure and add some data to it
def generate_legend_test_figure() -> ngp.Plotter:
    _plotter = gen_complex_example_fig()
    x = np.linspace(0, 2, 100)  # Sample data.
    for ax_name, ax in _plotter.ax_dict.items():
        ax.plot(x, x, label=f"linear {ax_name}")  # Plot some data on the axes.
        ax.plot(x, x**2, label=f"quadratic {ax_name}")  # Plot more data on the axes...
        ax.plot(x, x**3, label=f"cubic {ax_name}")  # ... and some more.
    return _plotter


def generate_legend_test_figure_common_items() -> ngp.Plotter:
    _plotter = gen_complex_example_fig()
    x = np.linspace(0, 2, 100)  # Sample data.
    for ax_name, ax in _plotter.ax_dict.items():
        ax.plot(
            x, x, label="the common linear legend item"
        )  # Plot some data on the axes.
        ax.plot(
            x, x**2, label="the common quadratic legend item"
        )  # Plot more data on the axes...
        ax.plot(x, x**3, label="the common cubic legend item")  # ... and some more.
    return _plotter


@pytest.mark.parametrize("is_outside_frame", (False, True))
def test_axis_and_fig_add_legend(is_outside_frame: bool):
    plotter = generate_legend_test_figure()

    # Test that there are no legend on any axes / figure
    # https://github.com/matplotlib/matplotlib/blob/v3.5.1/lib/matplotlib/figure.py#L942-L1075
    assert plotter.fig.legends == []  # stored as a list for Figure
    for subfig in plotter.sf_dict.values():
        assert subfig.legends == []  # stored as a list at the SubFigure level
    for ax_name, ax in plotter.ax_dict.items():
        assert ax.legend_ is None

    # Add some legends and test that it has been correctly added
    for ax_name in plotter.ax_dict.keys():
        if is_outside_frame:
            plotter.add_axis_legend_outside_frame(ax_name, fontsize=10)
        else:
            plotter.add_axis_legend(ax_name, fontsize=10)
    for ax_name, ax in plotter.ax_dict.items():
        # Handles
        if Version(mpl.__version__) >= Version("3.7"):
            assert len(ax.legend_.legend_handles) == 3  # ty: ignore[unresolved-attribute, possibly-missing-attribute]
        else:
            assert len(ax.legend_.legendHandles) == 3  # ty: ignore[unresolved-attribute, possibly-missing-attribute]
        # Labels
        assert [t._text for t in ax.legend_.texts] == [  # ty: ignore[unresolved-attribute, possibly-missing-attribute]
            f"linear {ax_name}",
            f"quadratic {ax_name}",
            f"cubic {ax_name}",
        ]

    # Test fig legend
    plotter.add_fig_legend(fontsize=10, ncol=2)
    assert len(plotter.fig.legends) == 1
    all_labels = [t._text for ax in plotter.axes for t in ax.legend_.texts]  # ty: ignore[unresolved-attribute, possibly-missing-attribute]
    assert sorted([t._text for t in plotter.fig.legends[0].texts]) == sorted(all_labels)  # ty: ignore[unresolved-attribute, possibly-missing-attribute]


def test_add_fig_legend_with_duplicated_labels_among_axes():
    # Test fig legend with common labels
    plotter = generate_legend_test_figure_common_items()
    plotter.add_fig_legend(fontsize=10)
    assert sorted([t._text for t in plotter.fig.legends[0].texts]) == sorted(  # ty: ignore[unresolved-attribute, possibly-missing-attribute]
        [
            "the common linear legend item",
            "the common quadratic legend item",
            "the common cubic legend item",
        ]
    )


def test_add_fig_legend_with_empty_figure():
    """Empty figure i.e., no data."""
    plotter = ngp.Plotter()
    plotter.add_fig_legend(fontsize=10)
    assert len(plotter.fig.legends) == 0


def test_add_fig_legend_multiple_times():
    plotter = generate_legend_test_figure_common_items()
    # Test the fact that only one legend could be added:
    for i in range(3):
        plotter.fig.legend()

    assert len(plotter.fig.legends) == 3

    for i in range(3):
        plotter.add_fig_legend(fontsize=10)
    assert len(plotter.fig.legends) == 1


def test_add_additional_legend_item():
    plotter = generate_legend_test_figure_common_items()
    handle = Line2D([0, 0], [0, 1], color="k", linewidth=3, linestyle="--")
    plotter.add_extra_legend_item("lt1", handle, "My extra legend item")
    plotter.add_axis_legend("lt1")
    plotter.add_fig_legend(fontsize=10, ncol=2)

    ax_labels = [t._text for t in plotter.ax_dict["lt1"].legend_.texts]  # ty: ignore[unresolved-attribute, possibly-missing-attribute]
    fig_labels = [t._text for t in plotter.fig.legends[0].texts]  # ty: ignore[unresolved-attribute, possibly-missing-attribute]

    assert ax_labels[-1] == "My extra legend item"
    assert fig_labels[-1] == "My extra legend item"


@pytest.mark.parametrize(
    "loc,expected_exception",
    [
        ("left", does_not_raise()),
        ("right", does_not_raise()),
        ("bottom", does_not_raise()),
        ("top", does_not_raise()),
        ("smth_that_is_not_valid", pytest.raises(ValueError)),
    ],
)
def test_custom_fig_legend_location(loc, expected_exception):
    plotter = gen_complex_example_fig()
    x = np.linspace(0, 2, 100)  # Sample data.
    for ax_name, ax in plotter.ax_dict.items():
        ax.plot(
            x, x, label="the common linear legend item"
        )  # Plot some data on the axes.
        ax.plot(
            x, x**2, label="the common quadratic legend item"
        )  # Plot more data on the axes...
        ax.plot(x, x**3, label="the common cubic legend item")  # ... and some more.

    with expected_exception:
        plotter.add_fig_legend(
            fontsize=10, loc=loc, bbox_x_shift=0.35, bbox_y_shift=0.0
        )

    # TODO: find a way to assert the position of the legend ???


def test_clear_all_axes():
    plotter = gen_complex_example_fig()
    x = np.linspace(0, 2, 100)  # Sample data.
    for ax_name, ax in plotter.ax_dict.items():
        ax.plot(
            x, x, label="the common linear legend item"
        )  # Plot some data on the axes.
        ax.plot(
            x, x**2, label="the common quadratic legend item"
        )  # Plot more data on the axes...
        ax.plot(x, x**3, label="the common cubic legend item")  # ... and some more.

    handle = Line2D([0, 0], [0, 1], color="k", linewidth=3, linestyle="--")
    plotter.add_extra_legend_item("lt1", handle, "My left extra legend item")

    handle = Line2D([0, 0], [0, 1], color="r", linewidth=3, linestyle="--")
    plotter.add_extra_legend_item("l2", handle, "My right extra legend item")

    plotter.add_fig_legend(fontsize=10, loc="top", bbox_x_shift=0.35, bbox_y_shift=0.0)
    plotter.add_fig_legend(
        name="the_left_sub_figure", fontsize=10, bbox_x_shift=0.35, bbox_y_shift=0.0
    )
    plotter.add_fig_legend(
        name="the_right_sub_figure", fontsize=10, bbox_x_shift=0.35, bbox_y_shift=0.0
    )

    for ax_name in plotter.ax_dict.keys():
        assert len(plotter._get_axis_legend_items(ax_name)[0]) != 0

    assert len(plotter.fig.legends) != 0
    for key, subfig in plotter.sf_dict.items():
        if key in ["the_left_sub_figure", "the_right_sub_figure"]:
            assert len(subfig.legends) != 0
        else:
            assert len(subfig.legends) == 0

    assert len(plotter._additional_handles) != 0
    assert len(plotter._additional_labels) != 0

    # We clear all
    plotter.clear_all_axes()

    for ax_name in plotter.ax_dict.keys():
        assert len(plotter._get_axis_legend_items(ax_name)[0]) == 0

    assert len(plotter.fig.legends) == 0
    for subfig in plotter.sf_dict.values():
        assert len(subfig.legends) == 0

    assert len(plotter._additional_handles) == 0
    assert len(plotter._additional_labels) == 0


def test_add_axis_legend_outside_frame_no_handles():
    assert ngp.Plotter().add_axis_legend_outside_frame("ax1-1") is None


def gen_complex_example_fig2(is_constrained_layout: bool) -> ngp.Plotter:
    return ngp.Plotter(
        ngp.Figure(
            constrained_layout=is_constrained_layout,
            figsize=(20, 10),
        ),
        builder=ngp.SubfigsBuilder(
            nrows=2,
            ncols=4,
            hspace=0.0,
            width_ratios=[1, 1, 1, 2],
            sub_builders={
                "sf1-1": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax1-1"]],
                ),
                "sf1-2": ngp.SubplotsMosaicBuilder(mosaic=[["ax1-2"]]),
                "sf1-3": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax1-3"]],
                ),
                "sf1-4": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax1-4"]],
                ),
                "sf2-1": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax2-1"]],
                ),
                "sf2-2": ngp.SubplotsMosaicBuilder(mosaic=[["ax2-2"]]),
                "sf2-3": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax2-3"]],
                ),
                "sf2-4": ngp.SubplotsMosaicBuilder(
                    mosaic=[["ax2-4"]],
                ),
            },
        ),
    )


@pytest.mark.parametrize(
    "is_add_bbox_x_y_shifts, is_constrained_layout",
    (((False, True), (True, True), (False, False))),
)
def test_add_axis_legend_outside_frame(
    is_add_bbox_x_y_shifts: bool, is_constrained_layout: bool
) -> None:
    plotter = gen_complex_example_fig2(is_constrained_layout)
    plotter.identify_axes()
    x = np.arange(100)
    for ax_name, ax in plotter.ax_dict.items():
        for i in range(4):
            ax.plot(x, np.cumsum(np.random.random(x.size)), label=f"c {i}")
        ax.set_title(ax_name)

    legends = []

    # Case 1: for lt and lb we add subfigure legend (we use corresponding subfigures)
    for ax_name, sf_name in zip(["ax1-1", "ax2-1"], ["sf1-1", "sf2-1"]):
        plotter.ax_dict[ax_name].set_facecolor("none")
        plotter.ax_dict[ax_name].set_xlabel(
            "x_label\ntwo levels", bbox={"color": "red"}
        )
        lgd = plotter.add_fig_legend(sf_name, ncols=2)

        # here legend overlaps
        if lgd is not None:  # required for ty type check
            assert is_lgd_overlapping_axis(plotter.get_axis(ax_name), lgd)

    # Case 2 for rt and rb we add axis legend
    for ax_name in ["ax1-2", "ax2-2"]:
        plotter.ax_dict[ax_name].set_facecolor("none")
        plotter.ax_dict[ax_name].set_xlabel(
            "x_label\ntwo levels", bbox={"color": "green"}
        )
        legends.append(plotter.add_axis_legend_outside_frame(ax_name, ncols=2))

        # here legend does not overlap
        if is_constrained_layout:
            assert not is_lgd_overlapping_axis(plotter.get_axis(ax_name), legends[-1])

    for ax_name in ["ax1-3", "ax2-3"]:
        plotter.ax_dict[ax_name].set_facecolor("none")
        plotter.ax_dict[ax_name].set_xlabel(
            "x_label\nthree\nlevels", bbox={"color": "green"}
        )
        if is_add_bbox_x_y_shifts:
            legends.append(
                plotter.add_axis_legend_outside_frame(
                    ax_name, ncols=2, loc="top", bbox_x_shift=0.0, bbox_y_shift=0.0
                )
            )
        else:
            legends.append(
                plotter.add_axis_legend_outside_frame(ax_name, ncols=2, loc="top")
            )

        # here legend does not overlap
        if not is_add_bbox_x_y_shifts and is_constrained_layout:
            assert not is_lgd_overlapping_axis(plotter.get_axis(ax_name), legends[-1])

    for ax_name, loc in zip(["ax1-4", "ax2-4"], ["right", "left"]):
        plotter.ax_dict[ax_name].set_facecolor("none")
        plotter.ax_dict[ax_name].set_xlabel(
            "x_label \ntwo levels", bbox={"color": "green"}
        )
        plotter.ax_dict[ax_name].set_ylabel("y_label", bbox={"color": "green"})
        if is_add_bbox_x_y_shifts:
            legends.append(
                plotter.add_axis_legend_outside_frame(
                    ax_name, ncols=1, loc=loc, bbox_x_shift=0.0, bbox_y_shift=0.0
                )
            )
        else:
            legends.append(
                plotter.add_axis_legend_outside_frame(ax_name, ncols=1, loc=loc)
            )

        # here legend does not overlap
        if not is_add_bbox_x_y_shifts and is_constrained_layout:
            assert not is_lgd_overlapping_axis(plotter.get_axis(ax_name), legends[-1])

    # for sf_name, subfig in plotter.grouped_sf_dict.items():
    #     plotter.add_fig_legend(sf_name, ncols=2, bbox_y_shift=-0.1)
    plotter.fig
