#!/usr/bin/env python

import os
import base64
import hashlib
import urllib.parse
import threading
import webbrowser
from http.server import HTTPServer, BaseHTTPRequestHandler

import click
import requests
import logging
import time

from nextcode.config import get_profile_config, get_default_profile, create_profile
from nextcode.exceptions import InvalidProfile
from nextcodecli.utils import abort

log = logging.getLogger(__name__)


def legacy_login():
    try:
        config = get_profile_config()
    except InvalidProfile as ex:
        click.secho(str(ex), fg="red")
        abort("Please create a profile with: nextcode profile add [name]")
    profile_name = get_default_profile()

    root_url = config["root_url"]
    login_server = f"{root_url.rstrip('/')}/api-key-service"

    click.secho("Launching login webpage ==> Please authenticate and then press continue.", fg="yellow")
    click.launch(login_server)
    click.pause()

    # Note: readline must be imported for click.prompt() to accept long strings. Don't ask me why.
    import readline

    api_key = click.prompt("API Key", type=str, hide_input=True)
    try:
        create_profile(profile_name, api_key=api_key, root_url=root_url)
        click.secho("Profile {} has been updated with api key".format(profile_name), bold=True, fg="green")
    except InvalidProfile as ex:
        abort(ex)


# === PKCE Utilities ===
def generate_pkce_pair():
    verifier = base64.urlsafe_b64encode(os.urandom(64)).decode("utf-8").rstrip("=")
    challenge = base64.urlsafe_b64encode(
        hashlib.sha256(verifier.encode("utf-8")).digest()
    ).decode("utf-8").rstrip("=")
    return verifier, challenge

# === Simple HTTP Server to Capture Redirect ===
class CallbackHandler(BaseHTTPRequestHandler):
    def do_GET(self):
        parsed_path = urllib.parse.urlparse(self.path)
        if parsed_path.path != "/callback":
            self.send_response(404)
            self.end_headers()
            return

        params = urllib.parse.parse_qs(parsed_path.query)
        self.server.auth_code = params.get("code", [None])[0]
        self.server.state = params.get("state", [None])[0]
        self.send_response(200)
        self.send_header("Content-type", "text/html")
        self.end_headers()
        self.wfile.write(b"<html><body><h2>You may now return to your terminal.</h2></body></html>")

    def log_message(self, format, *args):
        if getattr(self.server, "verbose_logging", False):
            super().log_message(format, *args)
        # Otherwise suppress HTTP server logs


def start_server(port, verbose=False):
    httpd = HTTPServer(("", port), CallbackHandler)
    httpd.verbose_logging = verbose
    threading.Thread(target=httpd.serve_forever, daemon=True).start()
    return httpd


@click.command()
@click.argument('tier', type=click.Choice(['dev', 'stg', 'prd']), required=False)
@click.option(
    '-t',
    '--token',
    'is_token',
    is_flag=True,
    help="Print refresh token instead of writing into current profile",
)
@click.option('-v', '--verbose', is_flag=True, help="Enable verbose HTTP server logging for troubleshooting")
@click.option('-l', '--legacy', is_flag=True, help="Use legacy login method")
@click.option(
    "--client-id",
    default="gdb-api-key-service",
    show_default=True,
    help="OIDC client id",
)
@click.option(
    "--scope",
    default="openid offline_access",
    show_default=True,
    help="Space-delimited OAuth scope",
)
def cli(tier, is_token, verbose, legacy, client_id, scope):
    """
    Authenticate against keycloak.

    TIER: Optional environment tier (dev|stg|prd). Defaults to profile detected tier.

    VERBOSE: Enable verbose HTTP server logging for troubleshooting.

    TOKEN: Print refresh token instead of writing into current profile.

    LEGACY: Use legacy login method.
    """
    if legacy:
        legacy_login()
        return

    try:
        config = get_profile_config()
    except InvalidProfile as ex:
        click.secho(str(ex), fg="red")
        abort("Please create a profile with: nextcode profile add [name]")
    profile_name = get_default_profile()

    if tier is None:
        root_url = config["root_url"]
        if ".dev." in root_url:
            tier = 'dev'
        elif ".stg." in root_url:
            tier = 'stg'
        else:
            tier = 'prd'

    configs = {
        "dev":  ("https://auth.dev.engops.genedx.net", "genedx-dev"),
        "stg":  ("https://auth.stg.engops.genedx.net", "genedx-stg"),
        "prd":  ("https://auth.engops.genedx.net", "genedx"),
    }
    keycloak_url, realm = configs[tier]
    redirect_uri = "http://localhost:8000/callback"
    port = 8000

    verifier, challenge = generate_pkce_pair()
    state = base64.urlsafe_b64encode(os.urandom(32)).decode("utf-8").rstrip("=")

    auth_url = (
        f"{keycloak_url}/realms/{realm}/protocol/openid-connect/auth"
        f"?response_type=code&client_id={client_id}"
        f"&redirect_uri={urllib.parse.quote(redirect_uri)}"
        f"&state={state}&scope={urllib.parse.quote(scope)}&code_challenge={challenge}&code_challenge_method=S256"
    )

    click.secho("Waiting for browser login (Ctrl+C to cancel)...", fg="green")
    server = start_server(port, verbose)

    click.secho("Opening browser...", fg="green")
    webbrowser.open(auth_url)

    while not getattr(server, "auth_code", None):
        time.sleep(0.1)

    code = server.auth_code
    server.shutdown()

    if not code:
        click.secho("Failed to retrieve code from callback", fg="red")
        exit(1)

    if getattr(server, "state", None) != state:
        click.secho("State mismatch; aborting.", fg="red")
        exit(1)

    click.secho("Exchanging code for tokens...", fg="green")
    token_url = f"{keycloak_url}/realms/{realm}/protocol/openid-connect/token"
    response = requests.post(
        token_url,
        data={
            "grant_type": "authorization_code",
            "client_id": client_id,
            "code": code,
            "redirect_uri": redirect_uri,
            "code_verifier": verifier,
        },
        headers={"Content-Type": "application/x-www-form-urlencoded"},
    )

    tokens = response.json()
    api_key = tokens['refresh_token']

    if is_token:
        click.echo(api_key)
        return

    try:
        create_profile(profile_name, api_key=api_key, root_url=config["root_url"])
        click.secho("Profile {} has been updated with api key".format(profile_name), bold=True, fg="green")
    except InvalidProfile as ex:
        abort(ex)

