import base64
import hashlib
import logging
import os
import platform
import sys
import webbrowser
from urllib.parse import parse_qs, urlparse

from keycloak import KeycloakOpenID

from nextlayer.sdk.errors import AuthenticationError

from . import utils

log = logging.getLogger(__name__)


def start_authorization_flow(k_client: KeycloakOpenID, redirect_uri: str | None = None):
    """
    Starts the Browser-based Authorization Code Flow with State, Nonce and PKCE.
    """
    if redirect_uri is None:
        redirect_uri = "https://login-callback.nextlayer.at/"

    if not sys.stdin.isatty():
        raise AuthenticationError(
            "Cannot start browser based authorization flow in non-interactive mode."
        )

    # Generate security parameters
    state = utils.generate_random_string(16)
    nonce = utils.generate_random_string(16)

    # PKCE Params
    # 1. Code Verifier (random String, 43-128 characters)
    code_verifier = utils.generate_random_string(64)
    # 2. Code Challenge (SHA256 Hash of Verifier, Base64 urlsafe)
    _hashed_verifier = hashlib.sha256(code_verifier.encode("utf-8")).digest()
    code_challenge = (
        base64.urlsafe_b64encode(_hashed_verifier).rstrip(b"=").decode("utf-8")
    )
    code_challenge_method = "S256"

    # generate Auth URL
    auth_url = k_client.auth_url(
        redirect_uri=redirect_uri,
        scope="openid",
        state=state,  # CSRF Protection
        nonce=nonce,  # OIDC Protection
        # code_challenge=code_challenge,
        # code_challenge_method=code_challenge_method,
    )
    # not yet supported by the python-keycloak library, so we need to add PKCE params manually
    auth_url += f"&code_challenge={code_challenge}&code_challenge_method={code_challenge_method}"

    sys.stderr.write("\n1. Please open the following URL in your browser to log in:")
    sys.stderr.write("-" * 70 + "\n")
    sys.stderr.write(auth_url + "\n")
    sys.stderr.write("-" * 70 + "\n")

    if platform.system() == "Windows" or os.environ.get("DISPLAY"):
        # open the URL in the default browser on supported platforms
        try:
            webbrowser.open_new_tab(auth_url)
            sys.stderr.write("The login page was opened in your default browser.\n")
        except Exception:
            pass

    sys.stderr.write(
        f"\n2. Please copy the complete redirect URL (starting with {redirect_uri}?) and paste it here:\n"
    )
    try:
        redirect_full_url = input("> ")
    except (EOFError, KeyboardInterrupt):
        sys.stderr.write("\nInput cancelled.\n")
        raise AuthenticationError(
            "No Callback-URL entered - Authorization flow aborted."
        )

    # Parse the entered URL and extract State/Code
    try:
        parsed_url = urlparse(redirect_full_url)
        query_params = parse_qs(parsed_url.query)

        if "code" not in query_params or "state" not in query_params:
            log.error("Code or State-Param missing in Redirect-URL.")
            raise AuthenticationError("missing code or state-param in redirect URL")

        # Verify the returned State against the initially sent
        returned_state = query_params["state"][0]
        if returned_state != state:
            log.warning(
                "SECURITY ERROR: The returned 'state' does not match the sent state."
            )
            log.debug(f"Sent: {state}, Received: {returned_state}")
            raise AuthenticationError("Attack attempt detected (CSRF)")

        auth_code = query_params["code"][0]
        log.debug(
            f"[OK] State verified and authorization code successfully received: {auth_code[:10]}..."
        )

    except Exception as e:
        log.error(f"ERROR parsing URL: {e}")
        raise AuthenticationError("Invalid Redirect URL")

    # Exchange code for token (with Code Verifier for PKCE)
    log.debug("Exchanging code for Access Token (with PKCE Verifier)...")
    try:
        token_response = k_client.token(
            grant_type="authorization_code",
            code=auth_code,
            redirect_uri=redirect_uri,
            code_verifier=code_verifier,  # key for PKCE
        )

        id_token = token_response.get("id_token") or ""
        sys.stderr.write("\n--- SUCCESS! Token received ---\n")

        # # Validate ID Token and verify Nonce
        id_token_nonce = k_client.decode_token(id_token).get("nonce")
        if id_token_nonce != nonce:
            raise AuthenticationError("Nonce mismatch in ID Token!")

        return token_response

    except Exception as e:
        log.error(f"ERROR during token exchange (Token endpoint call): {e}")
        raise
