import click
from tqdm import tqdm

# Own imports:
from niceplot.draw1dratio import draw1dratio
from niceplot.draw2dhist import draw2dhist
from niceplot.draw2dscatter import draw2dscatter
import niceplot.utils as utils
from niceplot.reader import Reader

@click.command()
@click.argument('config_file')
def niceplot(config_file: str) -> None:
    """Module to make nice looking root plots in the ATLAS Style.
    See https://gitlab.cern.ch/jwuerzin/nice-plot or https://pypi.org/project/niceplotpy/ for documentation.
    """
    # Print welcome message and version number:
    utils.printwelcome()
    
    # Read in config file and prep corresponding dictionary with pandas.DataFrames:
    reader = Reader(config_file)   
    
    dfdict = reader.prepdfdict()

    savestr = "\n"

    # Loop over all configurations and plotting configs; Make one plot for all configs & plot configs:
    # for plot in reader.plots:
    for plot in tqdm(reader.plots, desc="Generating Plots", unit="plots"):
        
        if plot.type == '1dratio':
            # Make one 1dratio plot with specific configuration:
            savestr += draw1dratio(
                dfdict=dfdict,
                x=plot.x,
                denominator=plot.denominator,
                numerator=plot.numerator,
                denomlab=utils.getaddinfo(reader.configurations, plot.denominator),
                numlab=utils.getaddinfo(reader.configurations, plot.numerator),
                nbins=plot.nbins,
                range=plot.range,
                ylab=plot.ylab,
                suffix=f"{plot.denominator}_over_{plot.numerator}",
                logy=plot.logy,
                output_dir=reader.output_dir,
                subdir=plot.subdir
            ) 
        elif plot.type == '2dhist':
            # Make one 2D (exclusion) Histogram for every dataframe configuration:
            for config in reader.configurations:
                # plot only for specified configs:
                if plot.configurations is not None and config.name not in plot.configurations: continue
                savestr += draw2dhist(
                    x=dfdict[config.name].get(plot.x),
                    y=dfdict[config.name].get(plot.y),
                    nbins=plot.nbins,
                    binrange=plot.range,
                    xlab=plot.xlab,
                    ylab=plot.ylab,
                    z=dfdict[config.name].get(plot.z),
                    zopt=plot.zopt,
                    suffix=config.name,
                    addinfo=config.addinfo,
                    output_dir=reader.output_dir,
                    subdir=plot.subdir,
                    addnumbers=plot.addnumbers,
                    doballs=plot.doballs
                )
        elif plot.type == '2dscatter':
            # Make one 2D scatter plot for every dataframe configuration:
            for config in reader.configurations:
                # plot only for specified configs:
                if plot.configurations is not None and config.name not in plot.configurations: continue
                savestr += draw2dscatter(
                    x=dfdict[config.name].get(plot.x),
                    y=dfdict[config.name].get(plot.y),
                    z=dfdict[config.name].get(plot.z),
                    range=plot.range,
                    xlab=plot.xlab,
                    ylab=plot.ylab,
                    suffix=config.name,
                    addinfo=config.addinfo,
                    output_dir=reader.output_dir,
                    subdir=plot.subdir
                )
        else:
            raise(ValueError(f"Plot type {plot.type} not recognised! Supported types are: 1dratio, 2dhist and 2dscatter"))
        
    print(savestr)
    print("Plots generated successfully. Have a great day!!")