import BwApi
import json
import sys
import os
import subprocess

curDir = os.path.dirname(os.path.realpath(__file__))
resourcePath = os.path.join(curDir, '../resources/')


class Callback(BwApi.CallbackBase):
    def Run(self, garment_id, callback_id, data_string):

        # run all command one after the other...
        try:
            if callback_id == 2:
                render.render_current_state_schematic()

            if callback_id == 3:
                render.render_current_state_normal()

            if callback_id == 4:
                render.render_current_state_ray_trace()

            if callback_id == 5:
                render.render_all_colorways_normal()

            if callback_id == 6:
                render.render_all_colorways_ray_trace()

            if callback_id == 7:
                render.render_multipack_normal()

            if callback_id == 8:
                render.render_multipack_ray_trace()

            if callback_id == 9:
                render.render_turntable_normal()

            if callback_id == 10:
                render.render_turntable_ray_trace()

            if callback_id == 11:
                render.export_browzwear_pbr_fbx()

            if callback_id == 12:
                render.export_multipack_browzwear_pbr_fbx()

            if callback_id == 13:
                render.export_autodesk_fbx()

            if callback_id == 14:
                render.export_browzwear_3d_object()

            if callback_id == 15:
                render.export_obj()

            if callback_id == 16:
                render.export_vray_scene()

        except Exception as e:
            BwApi.WndMessageBox("Error {0}".format(str(e)), BwApi.BW_API_MB_OK)

        return 1


def get_app_data_path():
    return os.path.join(os.path.expanduser('~'), 'render')


def get_window_view():
    if sys.platform == 'darwin':
        return "open -a Finder.app {}"
    elif sys.platform == 'win32':
        return "explorer.exe {}"


class Render:
    def __init__(self):
        self.garmentId = ''
        if sys.platform == 'win32':
            self.startupinfo = subprocess.STARTUPINFO()
            self.startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW

    def render_current_state_schematic(self):
        self.garmentId = BwApi.GarmentId()

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'PNG,png')
        if file_path == '':
            return

        render_params = self.load_json_into_params("RenderCurrentStateSchematic.json", file_path)
        BwApi.RenderImage(self.garmentId, json.dumps(render_params))

        if self.render_succeed():
            self.open_folder(os.path.dirname(file_path))

    def render_current_state_normal(self):
        self.garmentId = BwApi.GarmentId()

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'PNG,png')
        if file_path == '':
            return

        render_params = self.load_json_into_params("RenderCurrentStateNormal.json", file_path)
        BwApi.RenderImage(self.garmentId, json.dumps(render_params))

        if self.render_succeed():
            self.open_folder(os.path.dirname(file_path))

    def render_current_state_ray_trace(self):

        self.garmentId = BwApi.GarmentId()

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'PNG,png')
        if file_path == '':
            return

        render_params = self.load_json_into_params("RenderCurrentStateRayTrace.json", file_path)
        cmd = BwApi.RenderRayTraceCMD(self.garmentId, json.dumps(render_params))

        if self.render_succeed():
            self.execute_ray_trace(cmd)
            self.open_folder(os.path.dirname(file_path))

    def render_all_colorways_normal(self):
        self.garmentId = BwApi.GarmentId()

        folder_path = BwApi.WndFileDialog(0, 1, '', '', '', '')
        if folder_path == '':
            return

        colorways = BwApi.GarmentColorwayIds(self.garmentId)
        for colorway in colorways:
            BwApi.ColorwayCurrentSet(self.garmentId, colorway)
            BwApi.GarmentDress(self.garmentId)
            colorway_name = BwApi.ColorwayNameGet(self.garmentId, colorway)
            render_params = self.load_json_into_params("RenderCurrentStateNormal.json", os.path.join(folder_path, 'normal_{}.png'.format(colorway_name)))
            BwApi.RenderImage(self.garmentId, json.dumps(render_params))

        if self.render_succeed():
            self.open_folder(folder_path)

    def render_all_colorways_ray_trace(self):
        self.garmentId = BwApi.GarmentId()

        folder_path = BwApi.WndFileDialog(0, 1, '', '', '', '')
        if folder_path == '':
            return

        colorways = BwApi.GarmentColorwayIds(self.garmentId)
        for colorway in colorways:
            BwApi.ColorwayCurrentSet(self.garmentId, colorway)
            BwApi.GarmentDress(self.garmentId)
            colorway_name = BwApi.ColorwayNameGet(self.garmentId, colorway)
            image_name = 'raytrace_{}.png'.format(colorway_name)
            render_params = self.load_json_into_params("RenderCurrentStateRayTrace.json", os.path.join(folder_path, image_name))

            cmd = BwApi.RenderRayTraceCMD(self.garmentId, json.dumps(render_params))

            if self.render_succeed():
                self.execute_ray_trace(cmd)

        if self.render_succeed():
            BwApi.WndMessageBox("All the rendering tasks completed", BwApi.BW_API_MB_OK)
            self.open_folder(folder_path)

    def render_multipack_normal(self):
        self.garmentId = BwApi.GarmentId()

        if not self.number_of_colorways_check():
            return

        file_path = BwApi.WndFileDialog(0, 1, '', '', '', 'PNG,png')
        if file_path == '':
            return

        render_params = self.load_json_into_params("RenderMultipackNormal.json", file_path)

        self.resolve_colorways(render_params)

        BwApi.RenderImage(self.garmentId, json.dumps(render_params))

        if self.render_succeed():
            self.open_folder(os.path.dirname(file_path))

    def render_multipack_ray_trace(self):
        self.garmentId = BwApi.GarmentId()

        if not  self.number_of_colorways_check():
            return

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'PNG,png')
        if file_path == '':
            return

        render_params = self.load_json_into_params("RenderMultipackRayTrace.json", file_path)
        self.resolve_colorways(render_params)
        cmd = BwApi.RenderRayTraceCMD(self.garmentId, json.dumps(render_params))

        if self.render_succeed():
            self.execute_ray_trace(cmd)
            self.open_folder(os.path.dirname(file_path))

    def render_turntable_normal(self):
        self.garmentId = BwApi.GarmentId()

        folder_path = BwApi.WndFileDialog(0, 1, '', '', '', '')
        if folder_path == '':
            return

        render_params = self.load_turn_table_json_into_params("RenderTurntableNormal.json", folder_path)
        BwApi.RenderImage(self.garmentId, json.dumps(render_params))

        if self.render_succeed():
            self.open_folder(folder_path)

    def render_turntable_ray_trace(self):
        self.garmentId = BwApi.GarmentId()

        folder_path = BwApi.WndFileDialog(0, 1, '', '', '', '')
        if folder_path == '':
            return

        render_params = self.load_turn_table_json_into_params("RenderTurntableRayTrace.json", folder_path)
        cmd = BwApi.RenderRayTraceCMD(self.garmentId, json.dumps(render_params))

        if self.render_succeed():
            self.execute_ray_trace(cmd)

            BwApi.WndMessageBox("Rendering task is running in the background", BwApi.BW_API_MB_OK)
            self.open_folder(folder_path)

    def export_browzwear_pbr_fbx(self):
        self.garmentId = BwApi.GarmentId()

        if not self.number_of_colorways_check():
            return

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'FBX,fbx|ZIP,zip')
        if file_path == '':
            return

        export_params = self.load_json_into_export_params("ExportBrowzwearFBX.json", file_path)

        BwApi.RenderExport3DObject(self.garmentId, json.dumps(export_params))

        if self.render_succeed():
            self.open_folder(os.path.dirname(file_path))

    def export_multipack_browzwear_pbr_fbx(self):
        self.garmentId = BwApi.GarmentId()

        if not self.number_of_colorways_check():
            return

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'FBX,fbx|ZIP,zip')
        if file_path == '':
            return

        export_params = self.load_json_into_export_params("ExportBrowzwearFBXMultipack.json", file_path)
        self.resolve_colorways_export3d(export_params)

        BwApi.RenderExport3DObject(self.garmentId, json.dumps(export_params))

        if self.render_succeed():
            self.open_folder(os.path.dirname(file_path))

    def export_autodesk_fbx(self):
        self.garmentId = BwApi.GarmentId()

        if not self.number_of_colorways_check():
            return

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'FBX,fbx|ZIP,zip')
        if file_path == '':
            return

        export_params = self.load_json_into_export_params("ExportAutodeskFBX.json", file_path)

        BwApi.RenderExport3DObject(self.garmentId, json.dumps(export_params))

        if self.render_succeed():
            self.open_folder(os.path.dirname(file_path))

    def export_browzwear_3d_object(self):
        self.garmentId = BwApi.GarmentId()

        if not self.number_of_colorways_check():
            return

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'BWO,bwo')
        if file_path == '':
            return

        export_params = self.load_json_into_export_params("ExportBrowzwear3DObject.json", file_path)

        BwApi.RenderExport3DObject(self.garmentId, json.dumps(export_params))

        if self.render_succeed():
            self.open_folder(os.path.dirname(file_path))

    def export_obj(self):
        self.garmentId = BwApi.GarmentId()

        if not self.number_of_colorways_check():
            return

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'OBJ,obj|ZIP,zip')
        if file_path == '':
            return

        export_params = self.load_json_into_export_params("ExportOBJ.json", file_path)

        BwApi.RenderExport3DObject(self.garmentId, json.dumps(export_params))

        if self.render_succeed():
            self.open_folder(os.path.dirname(file_path))

    def export_vray_scene(self):
        self.garmentId = BwApi.GarmentId()

        if not self.number_of_colorways_check():
            return

        file_path = BwApi.WndFileDialog(0, 0, '', '', '', 'VRAYSCENE,vrscene')
        if file_path == '':
            return

        export_params = self.load_json_into_export_params("ExportVrayScene.json", file_path)

        BwApi.RenderExport3DObject(self.garmentId, json.dumps(export_params))

        if self.render_succeed():
            self.open_folder(os.path.dirname(file_path))

    @staticmethod
    def load_json_into_params(json_file_name, path):
        with open(os.path.join(resourcePath, json_file_name)) as data_file:
            render_params = json.load(data_file)
            render_params["captures"][0]["path"] = path
            return render_params

    @staticmethod
    def load_json_into_export_params(json_file_name, path):
        with open(os.path.join(resourcePath, json_file_name)) as data_file:
            export_params = json.load(data_file)
            export_params["path"] = path
            return export_params

    @staticmethod
    def load_turn_table_json_into_params(json_file_name, folder_name):
        with open(os.path.join(resourcePath, json_file_name)) as data_file:
            render_params = json.load(data_file)

            i = 1
            for capture in render_params["captures"]:
                capture["path"] = os.path.join(folder_name, 'image_{}.png'.format(i))
                i = i + 1

            return render_params

    def execute_ray_trace(self, cmd):
        global process
        if sys.platform == 'darwin':
            process = subprocess.Popen(cmd, shell=True)
        elif sys.platform == 'win32':
            process = subprocess.Popen(cmd, startupinfo=self.startupinfo)

        BwApi.WndMessageBox("Rendering task is running in the background", BwApi.BW_API_MB_OK)
        process.wait()

    @staticmethod
    def open_folder(path):
        last_error = BwApi.GetLastError()
        if last_error == BwApi.BW_API_ERROR_SUCCESS:
            if sys.platform == 'darwin':
                subprocess.call(get_window_view().format(path), shell=True)
            elif sys.platform == 'win32':
                subprocess.run(get_window_view().format(path))

    @staticmethod
    def render_succeed():
        last_error = BwApi.GetLastError()
        if last_error != BwApi.BW_API_ERROR_SUCCESS:

            if last_error == BwApi.BW_API_ERROR_NO_SNAPSHOT_AVAILABLE:
                BwApi.WndMessageBox("No snapshot is loaded", BwApi.BW_API_MB_OK)

            return False

        return True

    @staticmethod
    def number_of_colorways_check():
        garment_id = BwApi.GarmentId()
        if len(BwApi.GarmentColorwayIds(garment_id)) < 2:
            BwApi.WndMessageBox("Number of colorways must be more than 1", BwApi.BW_API_MB_OK)
            return False
        return True

    @staticmethod
    def resolve_colorways(render_params):
        garment_id = BwApi.GarmentId()
        colorways = BwApi.GarmentColorwayIds(garment_id)
        for i in range(len(render_params["multipack"])):
            render_params["multipack"][i]["colorway_id"] = colorways[i]

    @staticmethod
    def resolve_colorways_export3d(export_params):

        if export_params['layout']:
            garment_id = BwApi.GarmentId()
            colorways = BwApi.GarmentColorwayIds(garment_id)
            for i in range(len(export_params["layout"]["multipack"])):
                export_params["layout"]["multipack"][i]["colorway_id"] = colorways[i]


def BwApiPluginInit():
    BwApi.MenuFunctionAdd('Render current state (schematic)', callback, 2)
    BwApi.MenuFunctionAdd('Render current state (normal)', callback, 3)
    BwApi.MenuFunctionAdd('Render current state (ray-trace)', callback, 4)
    BwApi.MenuFunctionAdd('Render all colorways (normal)', callback, 5)
    BwApi.MenuFunctionAdd('Render all colorways (ray-trace)', callback, 6)
    BwApi.MenuFunctionAdd('Render multipack (normal)', callback, 7)
    BwApi.MenuFunctionAdd('Render multipack (ray-trace)', callback, 8)
    BwApi.MenuFunctionAdd('Render turntable (normal)', callback, 9)
    BwApi.MenuFunctionAdd('Render turntable (ray-trace)', callback, 10)

    BwApi.MenuFunctionAdd('-------------------------', callback, -1)

    BwApi.MenuFunctionAdd('Export Browzwear PBR FBX', callback, 11)
    BwApi.MenuFunctionAdd('Export multipack Browzwear PBR FBX', callback, 12)
    BwApi.MenuFunctionAdd('Export Autodesk FBX', callback, 13)
    BwApi.MenuFunctionAdd('Export Browzwear 3D Object', callback, 14)
    BwApi.MenuFunctionAdd('Export Obj', callback, 15)
    BwApi.MenuFunctionAdd('Export VRay scene ', callback, 16)

    BwApi.MenuFunctionReloadAdd()

    return int(0x000e0010)


callback = Callback()
render = Render()
