#!/usr/bin/python
#************************************************************************
# Copyright 2021 O7 Conseils inc (Philippe Gosselin)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#************************************************************************


#--------------------------------
#
#--------------------------------
# https://docs.python-requests.org/en/master/
import pprint
import logging
import datetime
import requests
import pytz



#*************************************************
#  Canadian Water Level Service
#
# https://tides.gc.ca/eng/info/WebServicesWLD
# Get all Stations: https://api-iwls.dfo-mpo.gc.ca/api/v1/stations
#
# {"id":"5cebf1e23d0f4a073c4bc0f6","code":"03248","officialName":"Vieux-Québec","operating":true,"latitude":46.811111,"longitude":-71.201944,"type":"PERMANENT","timeSeries":[{"id":"5cebf1e23d0f4a073c4bc0e1","code":"wlo","nameEn":"Water level official value","nameFr":"Niveau d'eau, valeur officielle","phenomenonId":"5ce598df487b84486892821c"},{"id":"5cebf1e23d0f4a073c4bc0ea","code":"wlp","nameEn":"Water level predictions","nameFr":"Prédictions de niveaux d'eau","phenomenonId":"5ce598df487b84486892821c"},{"id":"5cebf1e23d0f4a073c4bc0ec","code":"wt1","nameEn":"Water temperature sensor 1","nameFr":"Température de l'eau, capteur 1","phenomenonId":"5ce598df487b84486892821f"},{"id":"5cebf1e23d0f4a073c4bc0ef","code":"ap1","nameEn":"Atmospheric pressure sensor 1","nameFr":"Pression atmosphérique, capteur 1","phenomenonId":"5ce598df487b84486892821e"},{"id":"5d9dd7d533a9f593161c40d9","code":"wlp-hilo","nameEn":"High and Low Tide Predictions","nameFr":"Prédictions de hautes et basses marées","phenomenonId":"5ce598df487b84486892821c"},{"id":"5dc1c1b26828eee38e1e0922","code":"wlf-spine","nameEn":"Water level forecasts generated by SPINE","nameFr":"Prévisions de niveaux d'eau générées par SPINE","phenomenonId":"5ce598df487b84486892821c"}]}
#*************************************************
urlIwls = 'https://api-iwls.dfo-mpo.gc.ca'
tzEastern = pytz.timezone('US/Eastern')


#*************************************************
#
#*************************************************
def GetAllStation():

    logger = logging.getLogger('ca_wls.GetAllStation')

    url = urlIwls + '/api/v1/stations'
    logger.debug(f'Requesting URL -> {url}')

    r = requests.get(url)
    ret = r.json()

    logger.debug(f'Number of station received: {len(ret)}' )


    return ret


#*************************************************
#
#*************************************************
def GetClosestStation(lon, lat):

    logger = logging.getLogger('ca_wls.GetSerie')
    logger.debug(f'Parameters -> lon: {lon}  lat: {lat}')

    stations = GetAllStation()
    theStation = None
    theDist = None

    for station in stations :

        dist = (abs(lon - station["longitude"])**2) + (abs(lat - station["latitude"])**2)

        if station["operating"] == False: continue

        if theDist is None: theDist = dist
        if dist <= theDist:
            theStation = station
            theDist = dist



    logger.debug(f'GetClosestStation found station: {theStation["officialName"]}')
    #pprint.pprint(theStation)

    return theStation



#*************************************************
#
#*************************************************
def GetSerie(id, days = 5, serie = 'wlp'):

    logger = logging.getLogger('ca_wls.GetSerie')
    logger.debug(f'Parameters -> id: {id}; last: {days} days; serie: {serie}')

    url = urlIwls + f'/api/v1/stations/{id}/data'

    now = datetime.datetime.utcnow()
    dtFrom = dtTo = now.isoformat("T", "seconds") + 'Z'

    if days >= 0 :
        to = now + datetime.timedelta(days=days)
        dtTo = to.isoformat("T", "seconds") + 'Z'
    else:
        days = abs(days)
        past = now - datetime.timedelta(days=days)
        dtFrom = past.isoformat("T", "seconds") + 'Z',

    ret = []

    params = {
        "time-series-code" : serie,
        "from" : dtFrom,
        "to" : dtTo,
    }

    #pprint.pprint(params)
    r = requests.get(url,params=params)

    if r.status_code >= 300 :
        logger.warning(f'Wrong http status code: {r.status_code} Error: {r.json()}')
        r.raise_for_status()

    datas = r.json()
    #pprint.pprint(datas)

    for data in datas:
        utcTime = datetime.datetime.fromisoformat(data["eventDate"][:-1]).astimezone(tz=pytz.utc)
        eastTime = tzEastern.normalize(utcTime)

        ret.append({
            "dtu" : utcTime.isoformat(),
            "dtl" : eastTime.isoformat(),
            "value" : data["value"]
        })

    #pprint.pprint(ret)

    logger.debug(f'Number of data point received: {len(ret)}' )
    return ret

#*************************************************
#
#*************************************************
def GetTidesHiLo(id, days = 5):

    logger = logging.getLogger('ca_wls.GetTidesHiLo')
    logger.debug(f'Parameters -> id: {id}; last: {days} days')

    return GetSerie(id, days=days, serie='wlp-hilo')

#*************************************************
#
#*************************************************
def GetTidesSerie(id, days = 5):

    logger = logging.getLogger('ca_wls.GetTidesSerie')
    logger.debug(f'Parameters -> id: {id}; last: {days} days')

    return GetSerie(id, days=days, serie='wlp')


#*************************************************
#
#*************************************************
def GetLastTemperature(id) :

    logger = logging.getLogger('ca_wls.GetLastTemperature')
    logger.debug(f'Parameters -> id: {id}')

    datas = GetSerie(id, days=-1, serie='wt1')

    lastTemp = None
    lastTime = None

    if len(datas) > 0 :
        lastTemp = datas[-1]["value"]
        lastTime = datas[-1]["dtl"]

    logger.debug(f'Found: {lastTemp} C at {lastTime}' )
    return lastTemp, lastTime


#*************************************************
#
#*************************************************
if __name__ == "__main__":

    logging.basicConfig(level=logging.DEBUG)

    theStation = GetClosestStation(-71.1948, 46.8424)
    pprint.pprint(theStation)
    id = theStation['id']
    # hilo = GetTidesHiLo("5cebf1e23d0f4a073c4bc0f6", 7)

    wls = GetTidesSerie("5cebf1e23d0f4a073c4bc0f6", 7)
    pprint.pprint(wls)

    #GetLastTemperature("5cebf1e23d0f4a073c4bc0f6")