""" Module to Reports in PDF Format """
import logging
import datetime
import os

import pkg_resources
import fpdf

from o7pdf.colors import Colors

logger = logging.getLogger(__name__)
logging.getLogger('fontTools.subset').setLevel(logging.ERROR)

# https://py-pdf.github.io/fpdf2/fpdf/fpdf.html



#*************************************************
# https://pyfpdf.github.io/fpdf2/fpdf/
#*************************************************
class Template(fpdf.FPDF):
    """Class temaplate to generate PDF Report"""

    filename : str = "report.pdf"

    TEXT_FG = Colors.N800

    SECTION_TITLE_BG =  Colors.O7_COLOR_ALT
    SECTION_TITLE_FG =  Colors.N0

    SUB_TITLE_BG =  Colors.N40
    SUB_TITLE_FG =  Colors.O7_COLOR_MAIN


    #*************************************************
    #
    #*************************************************
    def __init__(self,
        filename : str = "report.pdf",
        username : str = None,
        updated : str = None,
        orientation : str = 'portrait',
        logo : str = None,
    ):


        super().__init__(orientation=orientation, unit='mm', format='letter')

        self.res_dir = pkg_resources.resource_filename(__name__, 'res')

        self.add_font(family='OpenSans', style='',  fname=os.path.join(self.res_dir, 'OpenSans-Regular.ttf'))
        self.add_font(family='OpenSans', style='B', fname=os.path.join(self.res_dir, 'OpenSans-Bold.ttf'))
        self.add_font(family='OpenSans', style='I', fname=os.path.join(self.res_dir, 'OpenSans-Italic.ttf'))
        self.add_font(family='OpenSans', style='BI', fname=os.path.join(self.res_dir,'OpenSans-BoldItalic.ttf'))
        self.set_font("OpenSans", size = 10)

        self.filename = filename
        self.title = "Template Report"
        self.username = username
        self.updated = datetime.datetime.fromisoformat(str(updated)).strftime('%Y-%m-%d') if updated else ''
        self.subtitle = None
        self.date : str = datetime.date.today().strftime('%Y-%m-%d')

        # Setting resurces
        self.logo = os.path.join(self.res_dir, logo) if logo else None



    #*************************************************
    #
    #*************************************************
    def save(self):
        """Save the PDF Report"""

        dir_name = os.path.dirname(self.filename)

        if len(dir_name) > 1 and not os.path.exists(dir_name):
            os.makedirs(dir_name)

        self.output(self.filename)

        print(f"PDF Report saved to: {self.filename}")

    #*************************************************
    #
    #*************************************************
    def cell(self, text_trim : bool = False, **kwargs): # pylint: disable=arguments-differ
        """Cell with support for text trim"""

        if text_trim:
            text = kwargs.get('text', '')
            cell_width = kwargs.get('w', 0)
            text_width = self.get_string_width(text)
            if text_width > cell_width:
                kwargs['text'] = text[:int(cell_width / text_width * len(text))]


        super().cell(**kwargs)


    #*************************************************
    #
    #*************************************************
    def header(self):
        """Format Header"""

        # Logo
        with self.local_context():

            title = self.title if self.subtitle is None else f'{self.title} - {self.subtitle}'
            self.set_font("OpenSans", size = 6)
            self.set_text_color(**self.TEXT_FG)
            self.set_draw_color(**self.SUB_TITLE_BG)
            self.set_line_width(0.1)
            self.cell(w = 0, text = f'**{self.date}**', new_x ="LMARGIN", new_y="NEXT", align = 'L', border = '', markdown=True)
            self.cell(w = 0, text = title, new_x ="LMARGIN", new_y="TOP", align = 'L', border = '', markdown=True)

            self.set_xy( (-15.0) - self.r_margin, self.t_margin)
            self.set_font("OpenSans", size = 6)
            self.cell(w = None, text = "powered by", new_x ="LEFT", new_y="NEXT", align = 'R', border = '', markdown=True)

            self.set_font("OpenSans", size = 10)
            self.set_text_color(**self.SUB_TITLE_FG)
            self.cell(w = None, h = 5, text = "**O7**", new_x ="RIGHT", new_y="TOP", align = 'L', border = '', markdown=True)
            self.set_x(self.get_x() - 2)
            self.set_font("OpenSans", size = 6)
            self.cell(w = None, h = 5, text = "Conseils", new_x ="RIGHT", new_y="TOP", align = 'L', border = '', markdown=True)


            #self.cell(w = 0, text = "O7 Conseils", new_x ="LMARGIN", new_y="TOP", align = 'L', border = '', markdown=True)

            self.set_xy( (-15.0) - self.r_margin, self.t_margin)
            #self.image(os.path.join(self.res_dir, 'caswil_logo_txt_500.png'), w=15, link='https://caswil.com')

            line_y = self.t_margin + 7
            self.line(x1=self.l_margin, y1=line_y, x2=self.epw + self.l_margin, y2=line_y)

        self.set_x(self.l_margin)
        self.set_y(self.t_margin + 8)



    #*************************************************
    #
    #*************************************************
    def footer(self):
        """Format Footer"""
        # Position at 1.5 cm from bottom

        with self.local_context():
            self.set_text_color(**self.TEXT_FG)
            self.set_font("OpenSans", size = 6)
            # Page number
            self.set_y(-15)

            if self.username :
                self.cell(
                    w = 0, h = 10,
                    text = f'Généré par: **{self.username}**',
                    border=0, align='L',
                    new_x ="LEFT", new_y="NEXT",
                    markdown=True
                )

            if self.updated :
                self.set_y(-12)
                self.cell(
                    w = 0, h = 10,
                    text = f'Données du: **{self.updated}**',
                    border=0, align='L',
                    new_x ="RIGHT", new_y="NEXT",
                    markdown=True
                )

            self.set_y(-15)
            self.cell(
                w = 0, h = 10,
                text = 'Page ' + str(self.page_no()) + ' / {nb}',
                border=0, align='R',
                new_x ="RIGHT", new_y="TOP"
            )


    # *************************************************
    #
    # *************************************************
    def report_head(self):
        """Top of first page"""

        page_start = self.get_y()

        with self.local_context():

            self.set_text_color(**self.TEXT_FG)

            # self.image(
            #     os.path.join(self.res_dir, 'rbc-wm-logo-fr.png'),
            #     x = self.l_margin, y = page_start + 3.5, h=9.5,
            #     alt_text='RBC Gestion de Patrimoine', link='https://www.rbcwealthmanagement.com/fr-ca/')

            self.set_xy(self.l_margin + self.epw * 0.2, page_start)
            self.set_font("OpenSans", size = 14)
            self.start_section(name = self.title, level=0)
            self.cell(
                w = self.epw * 0.7, h = 16,
                text = f'**{self.title}** ', fill=False,
                new_x ="LEFT", new_y="NEXT", align = 'L', border = 0, markdown=True)


        self.set_xy(self.l_margin, page_start + 18)

    #*************************************************
    #
    #*************************************************
    def section_title(self, title : str):
        """Format Section Title"""

        with self.local_context():
            self.set_font("OpenSans", size = 12)

            self.set_draw_color(**self.SECTION_TITLE_BG)
            self.set_fill_color(**self.SECTION_TITLE_BG)
            self.set_text_color(**self.SECTION_TITLE_FG)

            self.ln(3)
            self.set_x(self.l_margin)
            self.start_section(name = title, level=1)
            self.cell(
                w = self.epw, h=self.font_size * 1.75,
                text = title, fill=True,
                new_x ="LMARGIN", new_y="NEXT",
                align = 'C', border = '', markdown=True
            )
            self.ln(5)

    #*************************************************
    #
    #*************************************************
    def sub_title(self, title : str, link : str = None):
        """Format Section Title"""

        with self.local_context():
            self.set_font("OpenSans", size = 8)

            self.set_draw_color(**self.SUB_TITLE_BG)
            self.set_fill_color(**self.SUB_TITLE_BG)
            self.set_text_color(**self.SUB_TITLE_FG)

            self.set_x(self.l_margin)
            # https://py-pdf.github.io/fpdf2/fpdf/fpdf.html#fpdf.fpdf.FPDF.cell
            self.cell(
                w = self.epw, h=self.font_size * 1.5,
                text = f'  {title}', fill=True,
                new_x ="LMARGIN", new_y="NEXT",
                align = 'L', border = '', markdown=True,
                link = link
            )
            self.ln(1)


#*************************************************
#
#*************************************************
if __name__ == "__main__":


    the_pdf = Template( )
