"""
Pagination utilities for OdeCloud SDK.
"""

from typing import Any, AsyncIterator, Callable, Generic, Iterator, List, TypeVar

T = TypeVar("T")


class PaginatedResponse(Generic[T]):
    """A paginated response from the API."""

    def __init__(
        self,
        data: List[T],
        page: int,
        page_size: int,
        total: int,
        has_next_page: bool,
    ):
        self.data = data
        self.page = page
        self.page_size = page_size
        self.total = total
        self.has_next_page = has_next_page

    def __iter__(self) -> Iterator[T]:
        return iter(self.data)

    def __len__(self) -> int:
        return len(self.data)

    def __repr__(self) -> str:
        return (
            f"PaginatedResponse(page={self.page}, "
            f"page_size={self.page_size}, "
            f"total={self.total}, "
            f"items={len(self.data)})"
        )


class SyncPaginator(Generic[T]):
    """
    Auto-paginator for synchronous iteration through all results.

    Example:
        for entry in client.time_entries.list_all():
            print(entry.id)
    """

    def __init__(
        self,
        fetch_page: Callable[..., PaginatedResponse[T]],
        page_size: int = 100,
        **kwargs: Any,
    ):
        self._fetch_page = fetch_page
        self._page_size = page_size
        self._kwargs = kwargs

    def __iter__(self) -> Iterator[T]:
        page = 1
        while True:
            response = self._fetch_page(
                page=page,
                page_size=self._page_size,
                **self._kwargs,
            )
            for item in response.data:
                yield item
            if not response.has_next_page:
                break
            page += 1


class AsyncPaginator(Generic[T]):
    """
    Auto-paginator for asynchronous iteration through all results.

    Example:
        async for entry in client.time_entries.list_all():
            print(entry.id)
    """

    def __init__(
        self,
        fetch_page: Callable[..., Any],  # Returns Coroutine[PaginatedResponse[T]]
        page_size: int = 100,
        **kwargs: Any,
    ):
        self._fetch_page = fetch_page
        self._page_size = page_size
        self._kwargs = kwargs

    async def __aiter__(self) -> AsyncIterator[T]:
        page = 1
        while True:
            response = await self._fetch_page(
                page=page,
                page_size=self._page_size,
                **self._kwargs,
            )
            for item in response.data:
                yield item
            if not response.has_next_page:
                break
            page += 1
