"""
Custom exceptions for OdeCloud SDK.
"""

from typing import Any, Optional


class OdeCloudError(Exception):
    """Base exception for OdeCloud SDK errors."""

    def __init__(
        self,
        message: str,
        status_code: Optional[int] = None,
        response: Optional[Any] = None,
    ):
        super().__init__(message)
        self.message = message
        self.status_code = status_code
        self.response = response

    def __str__(self) -> str:
        if self.status_code:
            return f"[{self.status_code}] {self.message}"
        return self.message


class AuthenticationError(OdeCloudError):
    """Raised when authentication fails (401)."""

    pass


class ForbiddenError(OdeCloudError):
    """Raised when the API key lacks required scope (403)."""

    pass


class NotFoundError(OdeCloudError):
    """Raised when a resource is not found (404)."""

    pass


class ValidationError(OdeCloudError):
    """Raised when request validation fails (422)."""

    pass


class RateLimitError(OdeCloudError):
    """Raised when rate limit is exceeded (429)."""

    def __init__(
        self,
        message: str,
        retry_after: Optional[int] = None,
        **kwargs: Any,
    ):
        super().__init__(message, **kwargs)
        self.retry_after = retry_after


class ServerError(OdeCloudError):
    """Raised when the server returns a 5xx error."""

    pass
