"""
Organizations resource for OdeCloud API.
"""

from typing import TYPE_CHECKING, Any, Dict, Iterator, Optional

from odecloud.models.orgs import Org
from odecloud.pagination import AsyncPaginator, PaginatedResponse, SyncPaginator

if TYPE_CHECKING:
    from odecloud.client import AsyncOdeCloud, OdeCloud


class OrgsResource:
    """
    Synchronous organizations resource.

    Example:
        # List organizations
        orgs = client.orgs.list()
        for org in orgs:
            print(f"{org.name}")

        # Search for a specific org
        orgs = client.orgs.list(search="JVCKenwood")

        # Get org details
        org = client.orgs.get("org-123")
        print(f"{org.name}: {org.description}")
    """

    def __init__(self, client: "OdeCloud"):
        self._client = client

    def list(
        self,
        *,
        page: int = 1,
        page_size: int = 20,
        search: Optional[str] = None,
    ) -> PaginatedResponse[Org]:
        """
        List organizations you have access to.

        Args:
            page: Page number (default: 1)
            page_size: Items per page (default: 20, max: 100)
            search: Search by organization name

        Returns:
            PaginatedResponse containing Org objects
        """
        params: Dict[str, Any] = {"page": page, "pageSize": page_size}

        if search:
            params["search"] = search

        response = self._client._request("GET", "/orgs", params=params)

        return PaginatedResponse(
            data=[Org.model_validate(o) for o in response["data"]],
            page=response["page"],
            page_size=response["pageSize"],
            total=response["total"],
            has_next_page=response["hasNextPage"],
        )

    def list_all(
        self,
        *,
        page_size: int = 100,
        search: Optional[str] = None,
    ) -> Iterator[Org]:
        """
        Auto-paginate through all organizations.

        Args:
            page_size: Items per page (default: 100)
            search: Search by organization name

        Yields:
            Org objects
        """
        return SyncPaginator(
            self.list,
            page_size=page_size,
            search=search,
        )

    def get(self, org_id: str) -> Org:
        """
        Get details of an organization.

        Args:
            org_id: The organization ID

        Returns:
            Org object
        """
        response = self._client._request("GET", f"/orgs/{org_id}")
        return Org.model_validate(response)


class AsyncOrgsResource:
    """Asynchronous organizations resource."""

    def __init__(self, client: "AsyncOdeCloud"):
        self._client = client

    async def list(
        self,
        *,
        page: int = 1,
        page_size: int = 20,
        search: Optional[str] = None,
    ) -> PaginatedResponse[Org]:
        """List organizations you have access to."""
        params: Dict[str, Any] = {"page": page, "pageSize": page_size}

        if search:
            params["search"] = search

        response = await self._client._request("GET", "/orgs", params=params)

        return PaginatedResponse(
            data=[Org.model_validate(o) for o in response["data"]],
            page=response["page"],
            page_size=response["pageSize"],
            total=response["total"],
            has_next_page=response["hasNextPage"],
        )

    def list_all(
        self,
        *,
        page_size: int = 100,
        search: Optional[str] = None,
    ) -> AsyncPaginator[Org]:
        """Auto-paginate through all organizations."""
        return AsyncPaginator(
            self.list,
            page_size=page_size,
            search=search,
        )

    async def get(self, org_id: str) -> Org:
        """Get details of an organization."""
        response = await self._client._request("GET", f"/orgs/{org_id}")
        return Org.model_validate(response)
