"""
Profile resource for OdeCloud API.
"""

from typing import TYPE_CHECKING, Any, Dict, Optional

from odecloud.models.profile import Profile, ProfileUpdate, SocialLinks

if TYPE_CHECKING:
    from odecloud.client import AsyncOdeCloud, OdeCloud


class ProfileResource:
    """
    Synchronous profile resource.

    Example:
        # Get your profile
        profile = client.profile.get()
        print(f"Hello, {profile.full_name}!")
        print(f"Tagline: {profile.tagline}")

        # Update your profile
        updated = client.profile.update(
            tagline="Senior ERP Consultant | NetSuite Expert",
            location="San Francisco, CA",
            social_links={"linkedin": "https://linkedin.com/in/yourname"}
        )
    """

    def __init__(self, client: "OdeCloud"):
        self._client = client

    def get(self) -> Profile:
        """
        Get your profile.

        Returns:
            Profile object
        """
        response = self._client._request("GET", "/me")
        return Profile.model_validate(response)

    def update(
        self,
        *,
        first_name: Optional[str] = None,
        last_name: Optional[str] = None,
        tagline: Optional[str] = None,
        about_me: Optional[str] = None,
        location: Optional[str] = None,
        timezone: Optional[str] = None,
        social_links: Optional[Dict[str, str]] = None,
        is_public: Optional[bool] = None,
    ) -> Profile:
        """
        Update your profile.

        Args:
            first_name: Your first name
            last_name: Your last name
            tagline: Professional tagline (max 200 chars)
            about_me: About me section (max 5000 chars)
            location: Your location
            timezone: Your timezone
            social_links: Dict with linkedin, twitter, github, website
            is_public: Whether your profile is public

        Returns:
            Updated Profile object
        """
        body: Dict[str, Any] = {}

        if first_name is not None:
            body["firstName"] = first_name
        if last_name is not None:
            body["lastName"] = last_name
        if tagline is not None:
            body["tagline"] = tagline
        if about_me is not None:
            body["aboutMe"] = about_me
        if location is not None:
            body["location"] = location
        if timezone is not None:
            body["timezone"] = timezone
        if is_public is not None:
            body["isPublic"] = is_public
        if social_links is not None:
            body["socialLinks"] = social_links

        response = self._client._request("PATCH", "/me", json=body)
        return Profile.model_validate(response)


class AsyncProfileResource:
    """Asynchronous profile resource."""

    def __init__(self, client: "AsyncOdeCloud"):
        self._client = client

    async def get(self) -> Profile:
        """Get your profile."""
        response = await self._client._request("GET", "/me")
        return Profile.model_validate(response)

    async def update(
        self,
        *,
        first_name: Optional[str] = None,
        last_name: Optional[str] = None,
        tagline: Optional[str] = None,
        about_me: Optional[str] = None,
        location: Optional[str] = None,
        timezone: Optional[str] = None,
        social_links: Optional[Dict[str, str]] = None,
        is_public: Optional[bool] = None,
    ) -> Profile:
        """Update your profile."""
        body: Dict[str, Any] = {}

        if first_name is not None:
            body["firstName"] = first_name
        if last_name is not None:
            body["lastName"] = last_name
        if tagline is not None:
            body["tagline"] = tagline
        if about_me is not None:
            body["aboutMe"] = about_me
        if location is not None:
            body["location"] = location
        if timezone is not None:
            body["timezone"] = timezone
        if is_public is not None:
            body["isPublic"] = is_public
        if social_links is not None:
            body["socialLinks"] = social_links

        response = await self._client._request("PATCH", "/me", json=body)
        return Profile.model_validate(response)
