"""
Projects resource for OdeCloud API.
"""

from typing import TYPE_CHECKING, Any, Dict, Iterator, Optional

from odecloud.models.projects import Project, ProjectDetail, Task
from odecloud.pagination import AsyncPaginator, PaginatedResponse, SyncPaginator

if TYPE_CHECKING:
    from odecloud.client import AsyncOdeCloud, OdeCloud


class ProjectsResource:
    """
    Synchronous projects resource.

    Example:
        # List projects
        projects = client.projects.list(page=1)
        for project in projects:
            print(f"{project.title}: {project.total_time_hours}h tracked")

        # Get project details with members
        project = client.projects.get("project-123")
        for member in project.members:
            print(f"  - {member.full_name}")

        # List tasks for a project
        tasks = client.projects.list_tasks("project-123")
        for task in tasks:
            print(f"  {task.title}: {task.current_time_hours}h")
    """

    def __init__(self, client: "OdeCloud"):
        self._client = client

    def list(
        self,
        *,
        page: int = 1,
        page_size: int = 20,
        search: Optional[str] = None,
    ) -> PaginatedResponse[Project]:
        """
        List projects you have access to.

        Args:
            page: Page number (default: 1)
            page_size: Items per page (default: 20, max: 100)
            search: Search by project title

        Returns:
            PaginatedResponse containing Project objects
        """
        params: Dict[str, Any] = {"page": page, "pageSize": page_size}

        if search:
            params["search"] = search

        response = self._client._request("GET", "/projects", params=params)

        return PaginatedResponse(
            data=[Project.model_validate(p) for p in response["data"]],
            page=response["page"],
            page_size=response["pageSize"],
            total=response["total"],
            has_next_page=response["hasNextPage"],
        )

    def list_all(
        self,
        *,
        page_size: int = 100,
        search: Optional[str] = None,
    ) -> Iterator[Project]:
        """
        Auto-paginate through all projects.

        Args:
            page_size: Items per page (default: 100)
            search: Search by project title

        Yields:
            Project objects
        """
        return SyncPaginator(
            self.list,
            page_size=page_size,
            search=search,
        )

    def get(self, project_id: str) -> ProjectDetail:
        """
        Get detailed information about a project.

        Args:
            project_id: The project ID

        Returns:
            ProjectDetail object with members
        """
        response = self._client._request("GET", f"/projects/{project_id}")
        return ProjectDetail.model_validate(response)

    def list_tasks(
        self,
        project_id: str,
        *,
        page: int = 1,
        page_size: int = 20,
        status: Optional[str] = None,
    ) -> PaginatedResponse[Task]:
        """
        List tasks for a project.

        Args:
            project_id: The project ID
            page: Page number (default: 1)
            page_size: Items per page (default: 20, max: 100)
            status: Filter by task status

        Returns:
            PaginatedResponse containing Task objects
        """
        params: Dict[str, Any] = {"page": page, "pageSize": page_size}

        if status:
            params["status"] = status

        response = self._client._request(
            "GET", f"/projects/{project_id}/tasks", params=params
        )

        return PaginatedResponse(
            data=[Task.model_validate(t) for t in response["data"]],
            page=response["page"],
            page_size=response["pageSize"],
            total=response["total"],
            has_next_page=response["hasNextPage"],
        )

    def list_all_tasks(
        self,
        project_id: str,
        *,
        page_size: int = 100,
        status: Optional[str] = None,
    ) -> Iterator[Task]:
        """
        Auto-paginate through all tasks for a project.

        Args:
            project_id: The project ID
            page_size: Items per page (default: 100)
            status: Filter by task status

        Yields:
            Task objects
        """
        return SyncPaginator(
            lambda **kwargs: self.list_tasks(project_id, **kwargs),
            page_size=page_size,
            status=status,
        )


class AsyncProjectsResource:
    """Asynchronous projects resource."""

    def __init__(self, client: "AsyncOdeCloud"):
        self._client = client

    async def list(
        self,
        *,
        page: int = 1,
        page_size: int = 20,
        search: Optional[str] = None,
    ) -> PaginatedResponse[Project]:
        """List projects you have access to."""
        params: Dict[str, Any] = {"page": page, "pageSize": page_size}

        if search:
            params["search"] = search

        response = await self._client._request("GET", "/projects", params=params)

        return PaginatedResponse(
            data=[Project.model_validate(p) for p in response["data"]],
            page=response["page"],
            page_size=response["pageSize"],
            total=response["total"],
            has_next_page=response["hasNextPage"],
        )

    def list_all(
        self,
        *,
        page_size: int = 100,
        search: Optional[str] = None,
    ) -> AsyncPaginator[Project]:
        """Auto-paginate through all projects."""
        return AsyncPaginator(
            self.list,
            page_size=page_size,
            search=search,
        )

    async def get(self, project_id: str) -> ProjectDetail:
        """Get detailed information about a project."""
        response = await self._client._request("GET", f"/projects/{project_id}")
        return ProjectDetail.model_validate(response)

    async def list_tasks(
        self,
        project_id: str,
        *,
        page: int = 1,
        page_size: int = 20,
        status: Optional[str] = None,
    ) -> PaginatedResponse[Task]:
        """List tasks for a project."""
        params: Dict[str, Any] = {"page": page, "pageSize": page_size}

        if status:
            params["status"] = status

        response = await self._client._request(
            "GET", f"/projects/{project_id}/tasks", params=params
        )

        return PaginatedResponse(
            data=[Task.model_validate(t) for t in response["data"]],
            page=response["page"],
            page_size=response["pageSize"],
            total=response["total"],
            has_next_page=response["hasNextPage"],
        )

    def list_all_tasks(
        self,
        project_id: str,
        *,
        page_size: int = 100,
        status: Optional[str] = None,
    ) -> AsyncPaginator[Task]:
        """Auto-paginate through all tasks for a project."""

        async def fetch_page(**kwargs: Any) -> PaginatedResponse[Task]:
            return await self.list_tasks(project_id, **kwargs)

        return AsyncPaginator(
            fetch_page,
            page_size=page_size,
            status=status,
        )
