"""
OdeCloud API client.
"""

import time
from typing import Any, Dict, Optional

import httpx

from odecloud.exceptions import (
    AuthenticationError,
    ForbiddenError,
    NotFoundError,
    OdeCloudError,
    RateLimitError,
    ServerError,
    ValidationError,
)
from odecloud.resources.time_entries import AsyncTimeEntriesResource, TimeEntriesResource
from odecloud.resources.projects import AsyncProjectsResource, ProjectsResource
from odecloud.resources.profile import AsyncProfileResource, ProfileResource
from odecloud.resources.orgs import AsyncOrgsResource, OrgsResource


DEFAULT_BASE_URL = "https://server.odecloud.app/api/v1/public"
DEFAULT_TIMEOUT = 30.0
MAX_RETRIES = 3


class OdeCloud:
    """
    Synchronous OdeCloud API client.

    Example:
        from odecloud import OdeCloud

        client = OdeCloud(api_key="ode_live_...")

        # List time entries
        entries = client.time_entries.list()
        for entry in entries:
            print(f"{entry.task_title}: {entry.total_hours}h")

        # Get your profile
        profile = client.profile.get()
        print(f"Hello, {profile.full_name}!")
    """

    def __init__(
        self,
        api_key: str,
        *,
        base_url: str = DEFAULT_BASE_URL,
        timeout: float = DEFAULT_TIMEOUT,
        max_retries: int = MAX_RETRIES,
    ):
        """
        Initialize the OdeCloud client.

        Args:
            api_key: Your OdeCloud API key (starts with ode_live_ or ode_test_)
            base_url: API base URL (default: https://server.odecloud.app/api/v1/public)
            timeout: Request timeout in seconds (default: 30)
            max_retries: Maximum retries for rate-limited requests (default: 3)
        """
        if not api_key:
            raise ValueError("API key is required")

        self._api_key = api_key
        self._base_url = base_url.rstrip("/")
        self._timeout = timeout
        self._max_retries = max_retries

        self._client = httpx.Client(
            base_url=self._base_url,
            headers={
                "Authorization": f"Bearer {api_key}",
                "Content-Type": "application/json",
                "User-Agent": "odecloud-python/0.2.0",
            },
            timeout=timeout,
        )

        # Initialize resources
        self.time_entries = TimeEntriesResource(self)
        self.projects = ProjectsResource(self)
        self.profile = ProfileResource(self)
        self.orgs = OrgsResource(self)

    def _request(
        self,
        method: str,
        path: str,
        *,
        params: Optional[Dict[str, Any]] = None,
        json: Optional[Dict[str, Any]] = None,
    ) -> Any:
        """Make an HTTP request with retry logic."""
        retries = 0

        while True:
            try:
                response = self._client.request(
                    method,
                    path,
                    params=params,
                    json=json,
                )

                return self._handle_response(response)

            except RateLimitError as e:
                retries += 1
                if retries >= self._max_retries:
                    raise

                # Wait and retry
                wait_time = e.retry_after or (2**retries)
                time.sleep(wait_time)

    def _handle_response(self, response: httpx.Response) -> Any:
        """Handle API response and raise appropriate exceptions."""
        if response.status_code == 204:
            return None

        try:
            data = response.json()
        except Exception:
            data = None

        if response.is_success:
            return data

        # Handle errors
        message = "An error occurred"
        if isinstance(data, dict):
            message = data.get("detail", data.get("message", str(data)))
        elif data:
            message = str(data)

        error_kwargs = {
            "message": message,
            "status_code": response.status_code,
            "response": data,
        }

        if response.status_code == 401:
            raise AuthenticationError(**error_kwargs)
        elif response.status_code == 403:
            raise ForbiddenError(**error_kwargs)
        elif response.status_code == 404:
            raise NotFoundError(**error_kwargs)
        elif response.status_code == 422:
            raise ValidationError(**error_kwargs)
        elif response.status_code == 429:
            retry_after = response.headers.get("Retry-After")
            raise RateLimitError(
                **error_kwargs,
                retry_after=int(retry_after) if retry_after else None,
            )
        elif response.status_code >= 500:
            raise ServerError(**error_kwargs)
        else:
            raise OdeCloudError(**error_kwargs)

    def close(self) -> None:
        """Close the HTTP client."""
        self._client.close()

    def __enter__(self) -> "OdeCloud":
        return self

    def __exit__(self, *args: Any) -> None:
        self.close()


class AsyncOdeCloud:
    """
    Asynchronous OdeCloud API client.

    Example:
        import asyncio
        from odecloud import AsyncOdeCloud

        async def main():
            async with AsyncOdeCloud(api_key="ode_live_...") as client:
                # List time entries
                entries = await client.time_entries.list()
                for entry in entries:
                    print(f"{entry.task_title}: {entry.total_hours}h")

                # Get your profile
                profile = await client.profile.get()
                print(f"Hello, {profile.full_name}!")

        asyncio.run(main())
    """

    def __init__(
        self,
        api_key: str,
        *,
        base_url: str = DEFAULT_BASE_URL,
        timeout: float = DEFAULT_TIMEOUT,
        max_retries: int = MAX_RETRIES,
    ):
        """
        Initialize the async OdeCloud client.

        Args:
            api_key: Your OdeCloud API key (starts with ode_live_ or ode_test_)
            base_url: API base URL (default: https://server.odecloud.app/api/v1/public)
            timeout: Request timeout in seconds (default: 30)
            max_retries: Maximum retries for rate-limited requests (default: 3)
        """
        if not api_key:
            raise ValueError("API key is required")

        self._api_key = api_key
        self._base_url = base_url.rstrip("/")
        self._timeout = timeout
        self._max_retries = max_retries

        self._client = httpx.AsyncClient(
            base_url=self._base_url,
            headers={
                "Authorization": f"Bearer {api_key}",
                "Content-Type": "application/json",
                "User-Agent": "odecloud-python/0.2.0",
            },
            timeout=timeout,
        )

        # Initialize resources
        self.time_entries = AsyncTimeEntriesResource(self)
        self.projects = AsyncProjectsResource(self)
        self.profile = AsyncProfileResource(self)
        self.orgs = AsyncOrgsResource(self)

    async def _request(
        self,
        method: str,
        path: str,
        *,
        params: Optional[Dict[str, Any]] = None,
        json: Optional[Dict[str, Any]] = None,
    ) -> Any:
        """Make an HTTP request with retry logic."""
        import asyncio

        retries = 0

        while True:
            try:
                response = await self._client.request(
                    method,
                    path,
                    params=params,
                    json=json,
                )

                return self._handle_response(response)

            except RateLimitError as e:
                retries += 1
                if retries >= self._max_retries:
                    raise

                # Wait and retry
                wait_time = e.retry_after or (2**retries)
                await asyncio.sleep(wait_time)

    def _handle_response(self, response: httpx.Response) -> Any:
        """Handle API response and raise appropriate exceptions."""
        if response.status_code == 204:
            return None

        try:
            data = response.json()
        except Exception:
            data = None

        if response.is_success:
            return data

        # Handle errors
        message = "An error occurred"
        if isinstance(data, dict):
            message = data.get("detail", data.get("message", str(data)))
        elif data:
            message = str(data)

        error_kwargs = {
            "message": message,
            "status_code": response.status_code,
            "response": data,
        }

        if response.status_code == 401:
            raise AuthenticationError(**error_kwargs)
        elif response.status_code == 403:
            raise ForbiddenError(**error_kwargs)
        elif response.status_code == 404:
            raise NotFoundError(**error_kwargs)
        elif response.status_code == 422:
            raise ValidationError(**error_kwargs)
        elif response.status_code == 429:
            retry_after = response.headers.get("Retry-After")
            raise RateLimitError(
                **error_kwargs,
                retry_after=int(retry_after) if retry_after else None,
            )
        elif response.status_code >= 500:
            raise ServerError(**error_kwargs)
        else:
            raise OdeCloudError(**error_kwargs)

    async def close(self) -> None:
        """Close the HTTP client."""
        await self._client.aclose()

    async def __aenter__(self) -> "AsyncOdeCloud":
        return self

    async def __aexit__(self, *args: Any) -> None:
        await self.close()
