"""
Profile models for OdeCloud API.
"""

from datetime import datetime
from typing import List, Optional

from pydantic import BaseModel, Field


class SocialLinks(BaseModel):
    """Social media links."""

    linkedin: Optional[str] = None
    twitter: Optional[str] = None
    github: Optional[str] = None
    website: Optional[str] = None


class WorkExperience(BaseModel):
    """A work experience entry."""

    id: Optional[str] = Field(default=None, alias="_id")
    company: Optional[str] = None
    position: Optional[str] = None
    location: Optional[str] = None
    description: Optional[str] = None
    start_date: Optional[datetime] = Field(default=None, alias="startDate")
    end_date: Optional[datetime] = Field(default=None, alias="endDate")
    is_current: bool = Field(default=False, alias="isCurrent")

    class Config:
        populate_by_name = True


class Profile(BaseModel):
    """User profile."""

    id: str = Field(..., alias="_id")
    username: Optional[str] = None
    email: Optional[str] = None
    first_name: Optional[str] = Field(default=None, alias="firstName")
    last_name: Optional[str] = Field(default=None, alias="lastName")
    tagline: Optional[str] = None
    about_me: Optional[str] = Field(default=None, alias="aboutMe")
    avatar_url: Optional[str] = Field(default=None, alias="avatarUrl")
    location: Optional[str] = None
    timezone: Optional[str] = None
    social_links: Optional[SocialLinks] = Field(default=None, alias="socialLinks")
    work_experiences: List[WorkExperience] = Field(
        default_factory=list, alias="workExperiences"
    )
    skills: List[str] = Field(default_factory=list)
    is_public: bool = Field(default=False, alias="isPublic")
    created_at: Optional[datetime] = Field(default=None, alias="createdAt")

    class Config:
        populate_by_name = True

    @property
    def full_name(self) -> str:
        """Get the user's full name."""
        parts = [self.first_name, self.last_name]
        return " ".join(p for p in parts if p) or "Unknown"


class ProfileUpdate(BaseModel):
    """Request body for updating profile."""

    first_name: Optional[str] = Field(default=None, alias="firstName", max_length=100)
    last_name: Optional[str] = Field(default=None, alias="lastName", max_length=100)
    tagline: Optional[str] = Field(default=None, max_length=200)
    about_me: Optional[str] = Field(default=None, alias="aboutMe", max_length=5000)
    location: Optional[str] = Field(default=None, max_length=200)
    timezone: Optional[str] = Field(default=None, max_length=100)
    social_links: Optional[SocialLinks] = Field(default=None, alias="socialLinks")
    is_public: Optional[bool] = Field(default=None, alias="isPublic")

    class Config:
        populate_by_name = True
