"""
Project and task models for OdeCloud API.
"""

from datetime import datetime
from typing import List, Optional

from pydantic import BaseModel, Field


class ProjectMember(BaseModel):
    """A project team member."""

    id: str = Field(..., alias="_id")
    first_name: Optional[str] = Field(default=None, alias="firstName")
    last_name: Optional[str] = Field(default=None, alias="lastName")
    avatar_url: Optional[str] = Field(default=None, alias="avatarUrl")

    class Config:
        populate_by_name = True

    @property
    def full_name(self) -> str:
        """Get the member's full name."""
        parts = [self.first_name, self.last_name]
        return " ".join(p for p in parts if p) or "Unknown"


class Project(BaseModel):
    """A project."""

    id: str = Field(..., alias="_id")
    title: str
    description: Optional[str] = None
    org_id: Optional[str] = Field(default=None, alias="orgId")
    status: Optional[str] = None
    start_date: Optional[datetime] = Field(default=None, alias="startDate")
    end_date: Optional[datetime] = Field(default=None, alias="endDate")
    created_at: Optional[datetime] = Field(default=None, alias="createdAt")
    updated_at: Optional[datetime] = Field(default=None, alias="updatedAt")
    total_time: float = Field(default=0, alias="totalTime")
    member_count: int = Field(default=0, alias="memberCount")

    class Config:
        populate_by_name = True

    @property
    def total_time_hours(self) -> float:
        """Get total tracked time in hours."""
        return self.total_time / (1000 * 60 * 60)


class ProjectDetail(Project):
    """Detailed project with members."""

    members: List[ProjectMember] = Field(default_factory=list)


class Task(BaseModel):
    """A project task."""

    id: str = Field(..., alias="_id")
    title: str
    description: Optional[str] = None
    project_id: Optional[str] = Field(default=None, alias="projectId")
    status: Optional[str] = None
    current_time: float = Field(default=0, alias="currentTime")
    estimated_time: Optional[float] = Field(default=None, alias="estimatedTime")
    created_at: Optional[datetime] = Field(default=None, alias="createdAt")
    updated_at: Optional[datetime] = Field(default=None, alias="updatedAt")

    class Config:
        populate_by_name = True

    @property
    def current_time_hours(self) -> float:
        """Get current tracked time in hours."""
        return self.current_time / (1000 * 60 * 60)

    @property
    def estimated_time_hours(self) -> Optional[float]:
        """Get estimated time in hours."""
        if self.estimated_time is None:
            return None
        return self.estimated_time / (1000 * 60 * 60)

    @property
    def progress_percent(self) -> Optional[float]:
        """Get progress as percentage of estimated time."""
        if not self.estimated_time or self.estimated_time == 0:
            return None
        return min(100, (self.current_time / self.estimated_time) * 100)
