"""
Time entry models for OdeCloud API.
"""

from datetime import datetime
from typing import List, Optional

from pydantic import BaseModel, Field


class TimePair(BaseModel):
    """A start/end time pair in milliseconds."""

    started_at: float = Field(..., alias="startedAt")
    ended_at: float = Field(..., alias="endedAt")

    class Config:
        populate_by_name = True


class TimeEntry(BaseModel):
    """A time tracking entry."""

    id: str = Field(..., alias="_id")
    task_id: str = Field(..., alias="taskId")
    project_id: Optional[str] = Field(default=None, alias="projectId")
    project_title: Optional[str] = Field(default=None, alias="projectTitle")
    task_title: Optional[str] = Field(default=None, alias="taskTitle")
    times: List[TimePair] = Field(default_factory=list)
    accumulative: float = Field(..., description="Total time in milliseconds")
    billable: bool = True
    notes: Optional[str] = None
    created_at: datetime = Field(..., alias="createdAt")
    updated_at: Optional[datetime] = Field(default=None, alias="updatedAt")

    class Config:
        populate_by_name = True

    @property
    def total_hours(self) -> float:
        """Get total time in hours."""
        return self.accumulative / (1000 * 60 * 60)

    @property
    def total_minutes(self) -> float:
        """Get total time in minutes."""
        return self.accumulative / (1000 * 60)


class TimeEntryCreate(BaseModel):
    """Request body for creating a time entry."""

    task_id: str = Field(..., alias="taskId")
    times: List[TimePair] = Field(default_factory=list)
    preset: Optional[float] = Field(
        default=None,
        description="Preset timestamp (ms) for when work was done",
    )
    billable: bool = True
    notes: Optional[str] = Field(default=None, max_length=5000)

    class Config:
        populate_by_name = True


class TimeEntryUpdate(BaseModel):
    """Request body for updating a time entry."""

    times: Optional[List[TimePair]] = None
    billable: Optional[bool] = None
    notes: Optional[str] = Field(default=None, max_length=5000)

    class Config:
        populate_by_name = True


class TimeEntrySummary(BaseModel):
    """Summary of time entries for a period."""

    total_time_ms: float = Field(..., alias="totalTimeMs")
    total_time_hours: float = Field(..., alias="totalTimeHours")
    billable_time_ms: float = Field(..., alias="billableTimeMs")
    billable_time_hours: float = Field(..., alias="billableTimeHours")
    non_billable_time_ms: float = Field(..., alias="nonBillableTimeMs")
    non_billable_time_hours: float = Field(..., alias="nonBillableTimeHours")
    entry_count: int = Field(..., alias="entryCount")
    period_start: Optional[datetime] = Field(default=None, alias="periodStart")
    period_end: Optional[datetime] = Field(default=None, alias="periodEnd")

    class Config:
        populate_by_name = True
