"""
Time entries resource for OdeCloud API.
"""

from datetime import datetime
from typing import TYPE_CHECKING, Any, Dict, Iterator, Optional, Union

from odecloud.models.time_entries import (
    TimeEntry,
    TimeEntryCreate,
    TimeEntryUpdate,
    TimeEntrySummary,
)
from odecloud.pagination import AsyncPaginator, PaginatedResponse, SyncPaginator

if TYPE_CHECKING:
    from odecloud.client import AsyncOdeCloud, OdeCloud


class TimeEntriesResource:
    """
    Synchronous time entries resource.

    Example:
        # List time entries
        entries = client.time_entries.list(page=1, page_size=20)
        for entry in entries:
            print(f"{entry.task_title}: {entry.total_hours}h")

        # Create a time entry
        entry = client.time_entries.create(
            task_id="task-123",
            times=[{"startedAt": 1704067200000, "endedAt": 1704070800000}],
            billable=True,
            notes="Worked on feature X"
        )

        # Auto-paginate through all entries
        for entry in client.time_entries.list_all():
            print(entry.id)
    """

    def __init__(self, client: "OdeCloud"):
        self._client = client

    def list(
        self,
        *,
        page: int = 1,
        page_size: int = 20,
        task_id: Optional[str] = None,
        project_id: Optional[str] = None,
        billable: Optional[bool] = None,
        start_date: Optional[Union[datetime, str]] = None,
        end_date: Optional[Union[datetime, str]] = None,
    ) -> PaginatedResponse[TimeEntry]:
        """
        List time entries with optional filters.

        Args:
            page: Page number (default: 1)
            page_size: Items per page (default: 20, max: 100)
            task_id: Filter by task ID
            project_id: Filter by project ID
            billable: Filter by billable status
            start_date: Filter entries after this date
            end_date: Filter entries before this date

        Returns:
            PaginatedResponse containing TimeEntry objects
        """
        params: Dict[str, Any] = {"page": page, "pageSize": page_size}

        if task_id:
            params["taskId"] = task_id
        if project_id:
            params["projectId"] = project_id
        if billable is not None:
            params["billable"] = billable
        if start_date:
            params["startDate"] = (
                start_date.isoformat() if isinstance(start_date, datetime) else start_date
            )
        if end_date:
            params["endDate"] = (
                end_date.isoformat() if isinstance(end_date, datetime) else end_date
            )

        response = self._client._request("GET", "/time-entries", params=params)

        return PaginatedResponse(
            data=[TimeEntry.model_validate(e) for e in response["data"]],
            page=response["page"],
            page_size=response["pageSize"],
            total=response["total"],
            has_next_page=response["hasNextPage"],
        )

    def list_all(
        self,
        *,
        page_size: int = 100,
        task_id: Optional[str] = None,
        project_id: Optional[str] = None,
        billable: Optional[bool] = None,
        start_date: Optional[Union[datetime, str]] = None,
        end_date: Optional[Union[datetime, str]] = None,
    ) -> Iterator[TimeEntry]:
        """
        Auto-paginate through all time entries.

        Args:
            page_size: Items per page (default: 100)
            task_id: Filter by task ID
            project_id: Filter by project ID
            billable: Filter by billable status
            start_date: Filter entries after this date
            end_date: Filter entries before this date

        Yields:
            TimeEntry objects
        """
        return SyncPaginator(
            self.list,
            page_size=page_size,
            task_id=task_id,
            project_id=project_id,
            billable=billable,
            start_date=start_date,
            end_date=end_date,
        )

    def get(self, entry_id: str) -> TimeEntry:
        """
        Get a specific time entry by ID.

        Args:
            entry_id: The time entry ID

        Returns:
            TimeEntry object
        """
        response = self._client._request("GET", f"/time-entries/{entry_id}")
        return TimeEntry.model_validate(response)

    def create(
        self,
        task_id: str,
        *,
        times: Optional[list] = None,
        preset: Optional[float] = None,
        billable: bool = True,
        notes: Optional[str] = None,
    ) -> TimeEntry:
        """
        Create a new time entry.

        Args:
            task_id: The task ID to track time against
            times: List of time pairs [{"startedAt": ms, "endedAt": ms}]
            preset: Preset timestamp (ms) for when work was done
            billable: Whether this time is billable (default: True)
            notes: Notes about the work done

        Returns:
            Created TimeEntry object
        """
        body: Dict[str, Any] = {"taskId": task_id, "billable": billable}

        if times:
            body["times"] = times
        if preset is not None:
            body["preset"] = preset
        if notes:
            body["notes"] = notes

        response = self._client._request("POST", "/time-entries", json=body)
        return TimeEntry.model_validate(response)

    def update(
        self,
        entry_id: str,
        *,
        times: Optional[list] = None,
        billable: Optional[bool] = None,
        notes: Optional[str] = None,
    ) -> TimeEntry:
        """
        Update a time entry.

        Args:
            entry_id: The time entry ID
            times: Updated time pairs
            billable: Updated billable status
            notes: Updated notes

        Returns:
            Updated TimeEntry object
        """
        body: Dict[str, Any] = {}

        if times is not None:
            body["times"] = times
        if billable is not None:
            body["billable"] = billable
        if notes is not None:
            body["notes"] = notes

        response = self._client._request("PATCH", f"/time-entries/{entry_id}", json=body)
        return TimeEntry.model_validate(response)

    def delete(self, entry_id: str) -> None:
        """
        Delete a time entry.

        Args:
            entry_id: The time entry ID
        """
        self._client._request("DELETE", f"/time-entries/{entry_id}")

    def summary(
        self,
        *,
        start_date: Optional[Union[datetime, str]] = None,
        end_date: Optional[Union[datetime, str]] = None,
        project_id: Optional[str] = None,
    ) -> TimeEntrySummary:
        """
        Get a summary of time entries.

        Args:
            start_date: Start of period
            end_date: End of period
            project_id: Filter by project ID

        Returns:
            TimeEntrySummary object
        """
        params: Dict[str, Any] = {}

        if start_date:
            params["startDate"] = (
                start_date.isoformat() if isinstance(start_date, datetime) else start_date
            )
        if end_date:
            params["endDate"] = (
                end_date.isoformat() if isinstance(end_date, datetime) else end_date
            )
        if project_id:
            params["projectId"] = project_id

        response = self._client._request("GET", "/time-entries/summary", params=params)
        return TimeEntrySummary.model_validate(response)


class AsyncTimeEntriesResource:
    """Asynchronous time entries resource."""

    def __init__(self, client: "AsyncOdeCloud"):
        self._client = client

    async def list(
        self,
        *,
        page: int = 1,
        page_size: int = 20,
        task_id: Optional[str] = None,
        project_id: Optional[str] = None,
        billable: Optional[bool] = None,
        start_date: Optional[Union[datetime, str]] = None,
        end_date: Optional[Union[datetime, str]] = None,
    ) -> PaginatedResponse[TimeEntry]:
        """List time entries with optional filters."""
        params: Dict[str, Any] = {"page": page, "pageSize": page_size}

        if task_id:
            params["taskId"] = task_id
        if project_id:
            params["projectId"] = project_id
        if billable is not None:
            params["billable"] = billable
        if start_date:
            params["startDate"] = (
                start_date.isoformat() if isinstance(start_date, datetime) else start_date
            )
        if end_date:
            params["endDate"] = (
                end_date.isoformat() if isinstance(end_date, datetime) else end_date
            )

        response = await self._client._request("GET", "/time-entries", params=params)

        return PaginatedResponse(
            data=[TimeEntry.model_validate(e) for e in response["data"]],
            page=response["page"],
            page_size=response["pageSize"],
            total=response["total"],
            has_next_page=response["hasNextPage"],
        )

    def list_all(
        self,
        *,
        page_size: int = 100,
        task_id: Optional[str] = None,
        project_id: Optional[str] = None,
        billable: Optional[bool] = None,
        start_date: Optional[Union[datetime, str]] = None,
        end_date: Optional[Union[datetime, str]] = None,
    ) -> AsyncPaginator[TimeEntry]:
        """Auto-paginate through all time entries."""
        return AsyncPaginator(
            self.list,
            page_size=page_size,
            task_id=task_id,
            project_id=project_id,
            billable=billable,
            start_date=start_date,
            end_date=end_date,
        )

    async def get(self, entry_id: str) -> TimeEntry:
        """Get a specific time entry by ID."""
        response = await self._client._request("GET", f"/time-entries/{entry_id}")
        return TimeEntry.model_validate(response)

    async def create(
        self,
        task_id: str,
        *,
        times: Optional[list] = None,
        preset: Optional[float] = None,
        billable: bool = True,
        notes: Optional[str] = None,
    ) -> TimeEntry:
        """Create a new time entry."""
        body: Dict[str, Any] = {"taskId": task_id, "billable": billable}

        if times:
            body["times"] = times
        if preset is not None:
            body["preset"] = preset
        if notes:
            body["notes"] = notes

        response = await self._client._request("POST", "/time-entries", json=body)
        return TimeEntry.model_validate(response)

    async def update(
        self,
        entry_id: str,
        *,
        times: Optional[list] = None,
        billable: Optional[bool] = None,
        notes: Optional[str] = None,
    ) -> TimeEntry:
        """Update a time entry."""
        body: Dict[str, Any] = {}

        if times is not None:
            body["times"] = times
        if billable is not None:
            body["billable"] = billable
        if notes is not None:
            body["notes"] = notes

        response = await self._client._request("PATCH", f"/time-entries/{entry_id}", json=body)
        return TimeEntry.model_validate(response)

    async def delete(self, entry_id: str) -> None:
        """Delete a time entry."""
        await self._client._request("DELETE", f"/time-entries/{entry_id}")

    async def summary(
        self,
        *,
        start_date: Optional[Union[datetime, str]] = None,
        end_date: Optional[Union[datetime, str]] = None,
        project_id: Optional[str] = None,
    ) -> TimeEntrySummary:
        """Get a summary of time entries."""
        params: Dict[str, Any] = {}

        if start_date:
            params["startDate"] = (
                start_date.isoformat() if isinstance(start_date, datetime) else start_date
            )
        if end_date:
            params["endDate"] = (
                end_date.isoformat() if isinstance(end_date, datetime) else end_date
            )
        if project_id:
            params["projectId"] = project_id

        response = await self._client._request("GET", "/time-entries/summary", params=params)
        return TimeEntrySummary.model_validate(response)
