import time
import urllib.parse
from typing import Optional

import httpx
import jwt
from jose import jwt as jose_jwt
from jose import jwk
from jose.utils import base64url_decode
from PKCEManager import HybridPKCEManager, logger


def now() -> int:
    return int(time.time())


class JWKSManager:
    """
    Грузит JWKS с Keycloak и кэширует в Redis.
    """

    def __init__(self, aio_redis_client, jwks_url: str, ttl: int = 3600):
        self.redis = aio_redis_client
        self.jwks_url = jwks_url
        self.ttl = ttl
        self.redis_key = "jwks_cache"

    async def fetch(self) -> dict:
        cached = await self.redis.get(self.redis_key)
        if cached:
            import json
            return json.loads(cached)

        # получить JWKS с Keycloak
        async with httpx.AsyncClient(timeout=10) as client:
            r = await client.get(self.jwks_url)
            r.raise_for_status()
            jwks_text = r.text
            jwks = r.json()

        await self.redis.set(self.redis_key, jwks_text, ex=self.ttl)
        return jwks

    async def force_refresh(self):
        await self.redis.delete(self.redis_key)
        return await self.fetch()




class OIDCClient:
    """
    Проверяет токены Keycloak, использует PKCE и JWKS менеджеры.
    """

    def __init__(self, issuer: str, jwks_manager: JWKSManager):
        self.issuer = issuer
        self.jwks_manager = jwks_manager


    def _verify_signature(self, token: str, jwk_dict: dict) -> bool:
        public_key = jwk.construct(jwk_dict)
        message, encoded_sig = token.rsplit(".", 1)
        decoded_sig = base64url_decode(encoded_sig.encode())
        return public_key.verify(message.encode(), decoded_sig)


    def _validate_claims(self, claims: dict, audience: Optional[str]):
        if now() >= int(claims.get("exp", 0)):
            raise Exception("Token expired")

        if claims.get("iss") != self.issuer:
            raise Exception("Invalid issuer")

        if audience:
            aud = claims.get("aud")
            if isinstance(aud, list):
                if audience not in aud:
                    raise Exception("Invalid audience")
            else:
                if aud != audience:
                    raise Exception("Invalid audience")


    async def verify_token(self, token: str, audience: Optional[str] = None) -> dict:
        # unverified header
        try:
            header = jose_jwt.get_unverified_header(token)
        except Exception:
            raise Exception("Invalid token header")

        kid = header.get("kid")
        if not kid:
            raise Exception("Missing `kid` in token")

        # load JWKS
        jwks = await self.jwks_manager.fetch()
        key = next((k for k in jwks.get("keys", []) if k.get("kid") == kid), None)

        # if key not found → refresh JWKS
        if not key:
            jwks = await self.jwks_manager.force_refresh()
            key = next((k for k in jwks.get("keys", []) if k.get("kid") == kid), None)
            if not key:
                raise Exception("Key not found in JWKS")

        # verify signature
        if not self._verify_signature(token, key):
            raise Exception("Invalid signature")

        # decode claims without verifying (already verified)
        claims = jose_jwt.get_unverified_claims(token)

        # validate standard OIDC claims
        self._validate_claims(claims, audience)

        return claims



class CreateOIDCClient:
    """
        API для oidc

        Как пользоваться:
        Изначально нам необходим редирект на Keycloak
        например в Sanic мы создаем объект:
        Передаем url БД redis и имя Realm

        app.ctx.oidc_client = CreateOIDCClient(config["REDIS_URL"],
                                   config['KEYCLOAK_REALM_NAME'])

        Теперь получаем редирект передаем конфиг: config:dict там переменные
        окружения передаем с .env

        app.ctx.redirect_uri = app.ctx.oidc_client.get_redirect_uri(config)
        
        Пример:
        
        # Открываем соединение с Redis при старте
        @api.before_server_start
        async def setup_redis(app, _):
            # Подключаем Redis
            app.ctx.redis = redis.from_url(
                config["REDIS_URL"],
                decode_responses=True
            )
        
            # Создаём OIDC + PKCE менеджеры на основе redis URL
            createOIDCClient = CreateOIDCClient(config, app.ctx.redis)
            app.ctx.oidc_client = createOIDCClient
        
        например на маршрут /login
        @api.get("/login")
        async def auth_login(request):
            return response.redirect(request.app.ctx.redirect_uri)

        Нас перебрасывает на сервис Keycloak

        Авторизируемся

        После успешной авторизации мы пропускаем в наше приложение:

        @api.get("/callback")
        async def auth_callback(request: Request):
            # Достаем код и state
            
            code = request.args.get("code")
            state = request.args.get("state")
        
            if not code or not state:
                return response.json({"error": "Missing code or state"}, status=400)
        
            code_verifier = await request.app.ctx.oidc_client.pkce.pop(state)
            
            if not code_verifier:
                return response.json({"error": "Invalid/expired state"}, status=400)
        
            try:
                tokens = await request.app.ctx.oidc_client.exchange_code(code, code_verifier)
            except Exception as e:
                return response.json(
                    {"error": "Token exchange failed", "details": str(e)},
                    status=400
                )
        
            claims = await request.app.ctx.oidc_client.oidc.verify_token(tokens["access_token"])
        
            await request.app.ctx.oidc_client.save_session(claims["sub"], tokens, claims)
        
            return response.json({"message": "ok", "user": claims})
        
        Для проверки токена и пользователя, мы используем middleware 
        
        Пример:
        
        @api.middleware("request")
        async def keycloak_middleware(request: Request):
            auth = request.headers.get("Authorization")
            if not auth:
                request.ctx.user = None
                logger.info(f"User is : {request.ctx.user}")
                return
        
            token = auth.replace("Bearer ", "")
            request.ctx.user = await (request.app.ctx.oidc_client.get_user_from_token(token))
    
    """


    def __init__(self, config:dict, redis_client=None,):

        self.config = config
        self.redis = redis_client
        self.jwks = JWKSManager(
        aio_redis_client=self.redis,
        jwks_url=f"{self.config['KEYCLOAK_URL_HOST']}/realms/{self.config['KEYCLOAK_REALM_NAME']}/protocol/openid-connect/certs",
        ttl=3600)

        self.oidc = OIDCClient(
        issuer=f"{self.config['KEYCLOAK_URL_HOST']}/realms/{self.config['KEYCLOAK_REALM_NAME']}",
        jwks_manager=self.jwks)

        self.pkce = HybridPKCEManager(
            aio_redis_client=self.redis,
            ttl=300)


    async def get_redirect_uri(self) -> str:

        """
        Генерирует полный Keycloak authorize URL с PKCE.
        Получаем редирект ссылку

        :return: redirect url
        """

        state = self.pkce.generate_state()
        verifier, challenge = self.pkce.generate_pkce_pair()
        await self.pkce.store(state, verifier)

        params = {
            "client_id": self.config['KEYCLOAK_CLIENT_ID'],
            "response_type": "code",
            "redirect_uri": self.config['KEYCLOAK_URL'],
            "scope": "openid email profile",
            "code_challenge": challenge,
            "code_challenge_method": "S256",
            "state": state
        }
        return f"{self.config['KEYCLOAK_URL_OIDC']}?{urllib.parse.urlencode(params)}"



    async def exchange_code(self, code: str, verifier: str):
        """Меняем code → tokens"""

        logger.info(f"\n🔄 EXCHANGE_CODE:")
        # logger.info(f"  code={code[:20]}...")
        # logger.info(f"  verifier={verifier[:20]}...")
        # logger.info(f"  redirect_uri={self.config['KEYCLOAK_URL']}")
        # logger.info(f"  token_url={self.config['KEYCLOAK_TOKEN_URL']}")

        data = {
            "grant_type": "authorization_code",
            "client_id": self.config['KEYCLOAK_CLIENT_ID'],
            "code": code,
            "redirect_uri": self.config['KEYCLOAK_URL'],
            "code_verifier": verifier,
        }

        try:
            async with httpx.AsyncClient(timeout=10) as client:
                r = await client.post(self.config['KEYCLOAK_TOKEN_URL'], data=data)

                logger.info(f"📡 Token endpoint response:")
                # logger.info(f"  Status: {r.status_code}")
                # logger.info(f"  URL: {self.config['KEYCLOAK_TOKEN_URL']}")
                # logger.info(f"  Body: {r.text}")

                if r.status_code == 200:
                    return r.json()
                else:
                    error_text = r.text
                    raise Exception(f"Token exchange failed: {r.status_code}\n{error_text}")

        except Exception as e:
            logger.error(f"  ❌ ERROR: {e}")



    async def save_session(self, user_id: str, tokens: dict, claims: dict):
        """ сохранение в Redis"""
        await self.redis.hset(f"user:{user_id}",mapping={
            "access_token": tokens["access_token"],
            "refresh_token": tokens["refresh_token"],
            "username": claims.get("preferred_username"),
            "email": claims.get("email")})


    async def get_user_from_token(self, token: str) -> Optional[dict]:
        """Используем в middleware 
        
        Единая функция: decode → refresh → verify → user dict."""
        try:
            claims = jwt.decode(token, options={"verify_signature": False})
            user_id = claims["sub"]

            valid_token = await self._ensure_valid_token(user_id)
            if not valid_token:
                return None

            verified = jwt.decode(valid_token, options={"verify_signature": False})
            return {
                "id": user_id,
                "username": verified.get("preferred_username"),
                "email": verified.get("email"),
                "roles": verified.get("realm_access", {}).get("roles", []),
                "claims": verified
            }

        except Exception as e:
            return logger.error(f"Error in middleware: {e.__str__()}")


    async def _ensure_valid_token(self, user_id):

        stored = await self.redis.hgetall(f"user:{user_id}")
        if not stored:
            return None

        access_token = stored["access_token"]
        refresh_token = stored["refresh_token"]

        # проверяем время жизни access_token
        claims = jwt.decode(access_token, options={"verify_signature": False})
        exp = claims["exp"]
        now = int(time.time())

        # если осталось < 30 сек → обновляем!
        if exp - now > 30:
            return access_token

        # refresh
        async with httpx.AsyncClient() as client:
            r = await client.post(
                f"{self.config['KEYCLOAK_TOKEN_URL']}/token",
                data={
                    "grant_type": "refresh_token",
                    "refresh_token": refresh_token,
                    "client_id": self.config["CLIENT_ID"],
                    "client_secret": self.config["CLIENT_SECRET"],
                }
            )

        if r.status_code != 200:
            return logger.error(f"Error status code not equal 200: {r.text}")

        tokens = r.json()
        new_access = tokens["access_token"]
        new_refresh = tokens["refresh_token"]

        # обновляем в Redis
        await self.redis.hset(
            f"user:{user_id}",
            mapping={
                "access_token": new_access,
                "refresh_token": new_refresh,
            }
        )
        return new_access