from __future__ import annotations
from typing import Any, Dict, List, Tuple

from omnibioai_tool_exec.models.run_models import RunRequest, ValidationReport
from omnibioai_tool_exec.registry.tool_registry import ToolRegistry
from omnibioai_tool_exec.registry.server_registry import ServerRegistry
from omnibioai_tool_exec.routing.compatibility import check_compatibility


def pick_server(
    tool_registry: ToolRegistry,
    server_registry: ServerRegistry,
    req: RunRequest,
) -> Tuple[str, ValidationReport]:
    tool = tool_registry.get(req.tool_id)
    servers = server_registry.list()

    preferred = req.constraints.get("preferred_server_id")
    ordered: List[str] = []
    if preferred:
        ordered.append(preferred)
    ordered.extend([s.server_id for s in servers if s.server_id != preferred])

    per_server: Dict[str, Any] = {}

    for sid in ordered:
        srv = server_registry.get(sid)
        ok, details = check_compatibility(tool, srv, req.inputs, req.resources)
        per_server[sid] = {"ok": ok, "details": details}
        if ok:
            return sid, ValidationReport(ok=True, selected_server_id=sid, per_server=per_server)

    return "", ValidationReport(
        ok=False,
        errors=[{"code": "NO_COMPATIBLE_SERVER", "message": "No compatible server found"}],
        per_server=per_server,
    )

