from __future__ import annotations
import time
import secrets
from fastapi import APIRouter

from omnibioai_tool_exec.models.run_models import RunRequest, RunRecord
from omnibioai_tool_exec.registry.tool_registry import ToolRegistry
from omnibioai_tool_exec.registry.server_registry import ServerRegistry
from omnibioai_tool_exec.routing.router import pick_server
from omnibioai_tool_exec.execution.runner import Runner
from omnibioai_tool_exec.store.run_store import RunStore

router = APIRouter()

def _new_run_id() -> str:
    return f"run_{secrets.token_urlsafe(10)}"

def attach(
    tool_registry: ToolRegistry,
    server_registry: ServerRegistry,
    runner: Runner,
    store: RunStore,
) -> APIRouter:

    @router.post("/runs/validate")
    def validate(req: RunRequest):
        server_id, report = pick_server(tool_registry, server_registry, req)
        if not report.ok or not server_id:
            return report.model_dump()

        srv = server_registry.get(server_id)
        adapter = runner.get_adapter(srv.adapter_type)
        deep = adapter.validate(req.tool_id, req.inputs, req.resources)
        if not deep.get("ok", False):
            report.ok = False
            report.errors.extend(deep.get("errors", []))
        report.per_server[server_id]["adapter_validation"] = deep
        return report.model_dump()

    @router.post("/runs/submit")
    def submit(req: RunRequest):
        server_id, report = pick_server(tool_registry, server_registry, req)
        if not report.ok or not server_id:
            return {"ok": False, "error": report.model_dump()}

        srv = server_registry.get(server_id)
        adapter = runner.get_adapter(srv.adapter_type)

        deep = adapter.validate(req.tool_id, req.inputs, req.resources)
        if not deep.get("ok", False):
            return {"ok": False, "error": {"code": "VALIDATION_FAILED", "details": deep}}

        now = int(time.time())
        run_id = _new_run_id()
        rec = RunRecord(
            run_id=run_id,
            tool_id=req.tool_id,
            server_id=server_id,
            state="QUEUED",
            created_epoch=now,
            updated_epoch=now,
            inputs=req.inputs,
            resources=req.resources,
        )

        remote_id = adapter.submit(req.tool_id, req.inputs, req.resources)
        rec.remote_run_id = remote_id
        rec.state = "RUNNING"
        rec.updated_epoch = int(time.time())

        store.create(rec)
        return {"ok": True, "run_id": run_id, "server_id": server_id, "remote_run_id": remote_id}

    @router.get("/runs")
    def list_runs():
        return [r.model_dump() for r in store.list()]

    @router.get("/runs/{run_id}")
    def get_run(run_id: str):
        rec = store.get(run_id)
        srv = server_registry.get(rec.server_id)
        adapter = runner.get_adapter(srv.adapter_type)

        rec = runner.refresh_status(adapter, rec)
        store.update(rec)
        return rec.model_dump()

    @router.get("/runs/{run_id}/logs")
    def get_logs(run_id: str, tail: int = 200):
        rec = store.get(run_id)
        srv = server_registry.get(rec.server_id)
        adapter = runner.get_adapter(srv.adapter_type)
        return {"run_id": run_id, "logs": adapter.logs(rec.remote_run_id or "", tail=tail)}

    @router.get("/runs/{run_id}/results")
    def get_results(run_id: str):
        rec = store.get(run_id)
        srv = server_registry.get(rec.server_id)
        adapter = runner.get_adapter(srv.adapter_type)

        rec = runner.refresh_status(adapter, rec)
        store.update(rec)

        if rec.state != "COMPLETED":
            return {"ok": False, "error": {"code": "NOT_READY", "message": f"state={rec.state}"}}

        res = adapter.results(rec.remote_run_id or "")
        rec.results = res
        rec.updated_epoch = int(time.time())
        store.update(rec)
        return {"ok": True, "run_id": run_id, "results": res}

    return router

