from __future__ import annotations
import subprocess
import time
from pathlib import Path
from typing import Any, Dict

from omnibioai_tool_exec.execution.adapters.base import Adapter
from omnibioai_tool_exec.models.capabilities import ServerCapabilities, ToolCapability


class SlurmAdapter(Adapter):
    def __init__(self, config: Dict[str, Any]) -> None:
        self.config = config

    def adapter_type(self) -> str:
        return "slurm"

    def handshake(self) -> ServerCapabilities:
        return ServerCapabilities(
            engines=["slurm"],
            tools=[ToolCapability(tool_id="echo_test", version="1.0")],
            resources={"max_cpu": 20, "max_ram_gb": 128},
            policies={"max_runtime_minutes": 1440},
        )

    def validate(self, tool_id, inputs, resources):
        if tool_id != "echo_test":
            return {"ok": False, "errors": [{"message": "Unsupported tool"}]}
        if "message" not in inputs:
            return {"ok": False, "errors": [{"field": "message"}]}
        return {"ok": True, "errors": []}

    def submit(self, tool_id, inputs, resources):
        work_root = Path(self.config["work_root"])
        work_root.mkdir(parents=True, exist_ok=True)

        run_dir = work_root / f"run_{int(time.time() * 1000)}"
        run_dir.mkdir()

        sbatch_script = run_dir / "job.sbatch"
        out_file = run_dir / "stdout.txt"

        sbatch_script.write_text(f"""#!/bin/bash
#SBATCH -p {self.config.get("partition", "debug")}
#SBATCH -c {resources.get("cpu", 1)}
#SBATCH --output={out_file}

echo "{inputs['message']}"
""")

        r = subprocess.run(
            ["sbatch", str(sbatch_script)],
            capture_output=True,
            text=True,
            check=True,
        )

        return r.stdout.strip().split()[-1]

    def status(self, remote_run_id):
        r = subprocess.run(
            ["squeue", "-j", remote_run_id, "-h"],
            capture_output=True,
            text=True,
        )
        return {"state": "RUNNING" if r.stdout.strip() else "COMPLETED"}

    def logs(self, remote_run_id, tail=200):
        return "Logs stored in Slurm output file"

    def results(self, remote_run_id):
        return {"ok": True, "note": f"Slurm job {remote_run_id} completed"}
