from __future__ import annotations
import time
from typing import Any, Dict

from omnibioai_tool_exec.execution.adapters.base import Adapter
from omnibioai_tool_exec.models.capabilities import ServerCapabilities, ToolCapability


class LocalAdapter(Adapter):
    """
    Demo adapter to prove end-to-end flow.
    Replace/extend later to run real tools locally (subprocess/container).
    """
    def __init__(self, config: Dict[str, Any] | None = None) -> None:
        self.config = config or {}
        self._runs: Dict[str, Dict[str, Any]] = {}

    def adapter_type(self) -> str:
        return "local"

    def handshake(self) -> ServerCapabilities:
        return ServerCapabilities(
            engines=["local"],
            tools=[
                ToolCapability(tool_id="blastn", version="demo", features={"databases": ["ecoli_demo", "nt_small"]}),
            ],
            resources={"max_cpu": 8, "max_ram_gb": 32},
            storage={"allowed_uri_schemes": ["file"]},
            policies={"max_runtime_minutes": 30},
        )

    def validate(self, tool_id: str, inputs: Dict[str, Any], resources: Dict[str, Any]) -> Dict[str, Any]:
        errors = []
        if tool_id == "blastn":
            seq = inputs.get("sequence", "")
            db = inputs.get("database")
            if not isinstance(seq, str) or not seq.strip():
                errors.append({"field": "sequence", "message": "sequence is required"})
            if not db:
                errors.append({"field": "database", "message": "database is required"})
        else:
            errors.append({"code": "UNSUPPORTED_TOOL", "message": f"{tool_id} not supported in LocalAdapter demo"})
        return {"ok": len(errors) == 0, "errors": errors, "warnings": []}

    def submit(self, tool_id: str, inputs: Dict[str, Any], resources: Dict[str, Any]) -> str:
        remote_id = f"local_{int(time.time() * 1000)}"
        self._runs[remote_id] = {"state": "RUNNING", "started": time.time(), "tool_id": tool_id, "inputs": inputs}
        return remote_id

    def status(self, remote_run_id: str) -> Dict[str, Any]:
        run = self._runs.get(remote_run_id)
        if not run:
            return {"state": "FAILED", "message": "unknown run"}
        if time.time() - run["started"] > 1.5:
            run["state"] = "COMPLETED"
        return {"state": run["state"], "progress": 1.0 if run["state"] == "COMPLETED" else 0.5}

    def logs(self, remote_run_id: str, tail: int = 200) -> str:
        return f"[{remote_run_id}] demo logs (tail={tail})"

    def results(self, remote_run_id: str) -> Dict[str, Any]:
        run = self._runs.get(remote_run_id)
        if not run:
            return {"ok": False, "error": "unknown run"}
        if run["tool_id"] == "blastn":
            return {
                "ok": True,
                "hits": [
                    {"accession": "ABC123", "description": "E. coli demo hit", "evalue": 1e-50, "bitscore": 200.0},
                    {"accession": "XYZ999", "description": "Another demo hit", "evalue": 1e-20, "bitscore": 120.0},
                ],
            }
        return {"ok": True, "note": "no results"}

