from __future__ import annotations

import json
import time
from pathlib import Path
from typing import Any, Dict

from omnibioai_tool_exec.execution.adapters.base import Adapter
from omnibioai_tool_exec.models.capabilities import ServerCapabilities, ToolCapability

from omnibioai_tool_exec.execution.tools.bcftools_view_real import (
    BcftoolsConfig,
    run_bcftools_view_local,
    validate_inputs as validate_bcftools_inputs,
)


class LocalAdapter(Adapter):
    """
    Local adapter that runs real tools via subprocess on the same host as TES.

    Supports:
      - echo_test (simple)
      - bcftools_view (real bcftools)
      - blastn can remain demo OR you can add a real local blastn runner later.

    Strategy:
      - submit(): create run_dir, run tool synchronously (MVP) and persist results.json
      - status(): based on stored state
      - logs(): tail stdout/stderr files
      - results(): read results.json
    """

    def __init__(self, config: Dict[str, Any] | None = None) -> None:
        self.config = config or {}
        self.work_root = Path(self.config.get("work_root", "/tmp/omnibioai_tes_runs"))
        self.work_root.mkdir(parents=True, exist_ok=True)

        # remote_run_id -> metadata
        self._runs: Dict[str, Dict[str, Any]] = {}

    def adapter_type(self) -> str:
        return "local"

    # ----------------------------
    # Config helpers
    # ----------------------------
    def _bcftools_cfg(self) -> BcftoolsConfig:
        bc = (self.config.get("bcftools") or {}) if isinstance(self.config.get("bcftools"), dict) else {}
        return BcftoolsConfig(
            bcftools_path=str(bc.get("bcftools_path", "bcftools")),
            max_stdout_bytes=int(bc.get("max_stdout_bytes", 2_000_000) or 2_000_000),
            default_format=str(bc.get("default_format", "vcf")).lower(),
            allowed_formats=list(bc.get("allowed_formats", ["vcf", "bcf"]) or ["vcf", "bcf"]),
            require_index_for_region=bool(bc.get("require_index_for_region", True)),
        )

    def _new_remote_id(self) -> str:
        return f"local_{int(time.time() * 1000)}"

    def _tail_lines(self, text: str, tail: int) -> str:
        lines = (text or "").splitlines()
        if tail and tail > 0:
            lines = lines[-tail:]
        return "\n".join(lines)

    # ----------------------------
    # Capabilities
    # ----------------------------
    def handshake(self) -> ServerCapabilities:
        # For local: we can advertise bcftools_view always; validate will fail if bcftools missing.
        bc_cfg = self._bcftools_cfg()

        return ServerCapabilities(
            engines=["local"],
            tools=[
                ToolCapability(tool_id="echo_test", version="1.0", features={}),
                ToolCapability(tool_id="bcftools_view", version="real", features={"formats": bc_cfg.allowed_formats}),
            ],
            resources={
                "max_cpu": int(self.config.get("resources", {}).get("max_cpu", 8) or 8),
                "max_ram_gb": int(self.config.get("resources", {}).get("max_ram_gb", 32) or 32),
            },
            storage={"allowed_uri_schemes": ["file"]},
            policies={"max_runtime_minutes": int(self.config.get("policies", {}).get("max_runtime_minutes", 30) or 30)},
        )

    # ----------------------------
    # Validate
    # ----------------------------
    def validate(self, tool_id: str, inputs: Dict[str, Any], resources: Dict[str, Any]) -> Dict[str, Any]:
        errors = []

        if tool_id == "echo_test":
            msg = inputs.get("message")
            if not isinstance(msg, str) or not msg.strip():
                errors.append({"field": "message", "message": "message is required"})

        elif tool_id == "bcftools_view":
            input_path = inputs.get("input_path")
            if not isinstance(input_path, str) or not input_path.strip():
                errors.append({"field": "input_path", "message": "input_path is required (local file path)"})
            else:
                try:
                    p = Path(input_path).expanduser().resolve()
                    deep = validate_bcftools_inputs(self._bcftools_cfg(), inputs, p)
                    if not deep.get("ok", False):
                        errors.extend(deep.get("errors", []))
                except Exception as e:
                    errors.append({"field": "input_path", "message": str(e)})

        else:
            errors.append({"code": "UNSUPPORTED_TOOL", "message": f"{tool_id} not supported by LocalAdapter"})

        return {"ok": len(errors) == 0, "errors": errors, "warnings": []}

    # ----------------------------
    # Submit (runs locally)
    # ----------------------------
    def submit(self, tool_id: str, inputs: Dict[str, Any], resources: Dict[str, Any]) -> str:
        remote_id = self._new_remote_id()
        run_dir = self.work_root / remote_id
        run_dir.mkdir(parents=True, exist_ok=True)

        stdout_path = run_dir / "stdout.txt"
        stderr_path = run_dir / "stderr.txt"
        results_path = run_dir / "results.json"

        # write initial metadata
        meta = {
            "tool_id": tool_id,
            "run_dir": str(run_dir),
            "stdout": str(stdout_path),
            "stderr": str(stderr_path),
            "results": str(results_path),
            "created_epoch": int(time.time()),
        }
        self._runs[remote_id] = {"state": "RUNNING", **meta}

        try:
            if tool_id == "echo_test":
                msg = str(inputs.get("message", ""))
                stdout_path.write_text(msg + "\n")
                stderr_path.write_text("")
                results_path.write_text(json.dumps({"ok": True, "message": msg}, indent=2))

            elif tool_id == "bcftools_view":
                input_path = Path(str(inputs.get("input_path", ""))).expanduser().resolve()
                cfg = self._bcftools_cfg()

                rc, _stdout, _stderr, results = run_bcftools_view_local(
                    cfg=cfg,
                    run_dir=run_dir,
                    input_path=input_path,
                    inputs=inputs,
                    resources=resources,
                )

                # persist human-readable outputs
                stdout_path.write_text(results.get("stdout_tail", "") + "\n")
                stderr_path.write_text(results.get("stderr_tail", "") + "\n")
                results_path.write_text(json.dumps({"ok": rc == 0, **results}, indent=2))

                # mark exit status
                self._runs[remote_id]["exit_code"] = int(rc)

                if rc != 0:
                    self._runs[remote_id]["state"] = "FAILED"
                else:
                    self._runs[remote_id]["state"] = "COMPLETED"

            else:
                raise RuntimeError(f"Unsupported tool_id={tool_id}")

        except Exception as e:
            stderr_path.write_text(str(e) + "\n")
            results_path.write_text(json.dumps({"ok": False, "error": str(e)}, indent=2))
            self._runs[remote_id]["state"] = "FAILED"
            self._runs[remote_id]["exit_code"] = 1

        self._runs[remote_id]["updated_epoch"] = int(time.time())
        return remote_id

    # ----------------------------
    # Status / Logs / Results
    # ----------------------------
    def status(self, remote_run_id: str) -> Dict[str, Any]:
        meta = self._runs.get(remote_run_id)
        if not meta:
            return {"state": "FAILED", "message": "unknown run"}
        return {"state": meta.get("state", "RUNNING")}

    def logs(self, remote_run_id: str, tail: int = 200) -> str:
        meta = self._runs.get(remote_run_id)
        if not meta:
            return f"[{remote_run_id}] unknown run"

        p = Path(meta["stdout"])
        if not p.exists():
            return f"[{remote_run_id}] stdout not created yet"
        return self._tail_lines(p.read_text(errors="replace"), tail)

    def results(self, remote_run_id: str) -> Dict[str, Any]:
        meta = self._runs.get(remote_run_id)
        if not meta:
            return {"ok": False, "error": "unknown run"}

        run_dir = Path(meta["run_dir"])
        results_path = run_dir / "results.json"
        if not results_path.exists():
            return {"ok": False, "error": "results not found"}

        try:
            obj = json.loads(results_path.read_text())
        except Exception as e:
            return {"ok": False, "error": f"failed to parse results.json: {e}"}

        return obj
