from __future__ import annotations

import time

from omnibioai_tool_exec.models.server_spec import ServerSpec


def _register_local_demo(app):
    # Register a local server dynamically for tests
    srv = ServerSpec(
        server_id="local_demo",
        display_name="Local Demo",
        adapter_type="local",
        config={},
    )
    # local adapter exists in app.state.runner.adapters["local"]
    local_adapter = app.state.runner.get_adapter("local")
    srv.capabilities = local_adapter.handshake()
    srv.capabilities_last_refreshed_epoch = int(time.time())
    app.state.server_registry.register(srv)


def test_validate_blastn_local(client):
    app = client.app
    _register_local_demo(app)

    payload = {
        "tool_id": "blastn",
        "inputs": {"sequence": "ACTGACTG", "database": "ecoli_demo"},
        "resources": {"cpu": 1},
        "constraints": {"preferred_server_id": "local_demo"},
    }
    r = client.post("/api/runs/validate", json=payload)
    assert r.status_code == 200
    data = r.json()
    assert data["ok"] is True
    assert data["selected_server_id"] == "local_demo"


def test_submit_and_get_results_local(client):
    app = client.app
    _register_local_demo(app)

    payload = {
        "tool_id": "blastn",
        "inputs": {"sequence": "ACTGACTG", "database": "ecoli_demo"},
        "resources": {"cpu": 1},
        "constraints": {"preferred_server_id": "local_demo"},
    }

    # submit
    r = client.post("/api/runs/submit", json=payload)
    assert r.status_code == 200
    out = r.json()
    assert out["ok"] is True
    run_id = out["run_id"]

    # poll run until completed (LocalAdapter completes after ~1.5s)
    deadline = time.time() + 5
    state = None
    while time.time() < deadline:
        rr = client.get(f"/api/runs/{run_id}")
        assert rr.status_code == 200
        rec = rr.json()
        state = rec["state"]
        if state == "COMPLETED":
            break
        time.sleep(0.2)

    assert state == "COMPLETED"

    # results
    res = client.get(f"/api/runs/{run_id}/results")
    assert res.status_code == 200
    data = res.json()
    assert data["ok"] is True
    assert data["run_id"] == run_id
    assert "results" in data
    assert "hits" in data["results"]
    assert len(data["results"]["hits"]) >= 1
