from __future__ import annotations
from typing import Any, Dict
import httpx

from omnibioai_tool_exec.execution.adapters.base import Adapter
from omnibioai_tool_exec.models.capabilities import ServerCapabilities


class HttpToolServerAdapter(Adapter):
    """
    Connects to a remote ToolServer shim that implements:
      GET  /capabilities
      POST /validate
      POST /runs
      GET  /runs/{id}
      GET  /runs/{id}/logs
      GET  /runs/{id}/results
    """
    def __init__(self, config: Dict[str, Any]) -> None:
        self.base_url = str(config["base_url"]).rstrip("/")
        self.auth = config.get("auth", {"type": "none"})

    def adapter_type(self) -> str:
        return "http_toolserver"

    def _headers(self) -> Dict[str, str]:
        if self.auth.get("type") == "bearer":
            return {"Authorization": f"Bearer {self.auth['token']}"}
        return {}

    def handshake(self) -> ServerCapabilities:
        with httpx.Client(timeout=10.0, headers=self._headers()) as c:
            r = c.get(f"{self.base_url}/capabilities")
            r.raise_for_status()
            return ServerCapabilities(**r.json())

    def validate(self, tool_id: str, inputs: Dict[str, Any], resources: Dict[str, Any]) -> Dict[str, Any]:
        with httpx.Client(timeout=30.0, headers=self._headers()) as c:
            r = c.post(f"{self.base_url}/validate", json={"tool_id": tool_id, "inputs": inputs, "resources": resources})
            r.raise_for_status()
            return r.json()

    def submit(self, tool_id: str, inputs: Dict[str, Any], resources: Dict[str, Any]) -> str:
        with httpx.Client(timeout=30.0, headers=self._headers()) as c:
            r = c.post(f"{self.base_url}/runs", json={"tool_id": tool_id, "inputs": inputs, "resources": resources})
            r.raise_for_status()
            return r.json()["run_id"]

    def status(self, remote_run_id: str) -> Dict[str, Any]:
        with httpx.Client(timeout=10.0, headers=self._headers()) as c:
            r = c.get(f"{self.base_url}/runs/{remote_run_id}")
            r.raise_for_status()
            return r.json()

    def logs(self, remote_run_id: str, tail: int = 200) -> str:
        with httpx.Client(timeout=10.0, headers=self._headers()) as c:
            r = c.get(f"{self.base_url}/runs/{remote_run_id}/logs", params={"tail": tail})
            r.raise_for_status()
            return r.json().get("logs", "")

    def results(self, remote_run_id: str) -> Dict[str, Any]:
        with httpx.Client(timeout=30.0, headers=self._headers()) as c:
            r = c.get(f"{self.base_url}/runs/{remote_run_id}/results")
            r.raise_for_status()
            return r.json()

