from __future__ import annotations
from typing import Any, Dict, Tuple

from omnibioai_tool_exec.models.tool_spec import ToolSpec
from omnibioai_tool_exec.models.server_spec import ServerSpec


def check_compatibility(
    tool: ToolSpec,
    server: ServerSpec,
    req_inputs: Dict[str, Any],
    req_resources: Dict[str, Any],
) -> Tuple[bool, Dict[str, Any]]:
    details: Dict[str, Any] = {"reasons": []}

    if server.capabilities is None:
        details["reasons"].append("Server capabilities missing (handshake not performed).")
        return False, details

    cap = server.capabilities
    tool_cap = cap.tool_map().get(tool.tool_id)
    if tool_cap is None:
        details["reasons"].append(f"Tool '{tool.tool_id}' not supported on this server.")
        return False, details

    # Optional resource checks
    max_cpu = cap.resources.get("max_cpu")
    if max_cpu is not None and "cpu" in req_resources:
        if int(req_resources["cpu"]) > int(max_cpu):
            details["reasons"].append(f"cpu={req_resources['cpu']} exceeds max_cpu={max_cpu}")
            return False, details

    max_ram_gb = cap.resources.get("max_ram_gb")
    if max_ram_gb is not None and "ram_gb" in req_resources:
        if float(req_resources["ram_gb"]) > float(max_ram_gb):
            details["reasons"].append(f"ram_gb={req_resources['ram_gb']} exceeds max_ram_gb={max_ram_gb}")
            return False, details

    # Tool-specific example: BLAST db requirement
    if tool.requires.get("blast_db_required"):
        db = req_inputs.get("database")
        if not db:
            details["reasons"].append("Missing required input: database")
            return False, details

        supported_dbs = tool_cap.features.get("databases")
        if isinstance(supported_dbs, list) and supported_dbs and db not in supported_dbs:
            details["reasons"].append(f"Database '{db}' not available. Supported: {supported_dbs}")
            return False, details

    return True, {"reasons": []}

